/***************************************************************
 *             Finite Element Method Object Library            *
 *              class mesh : declaration for mesh              *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.6.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2002,2003,2004,2005,2006 CREUSE Emmanuel
 * copyright © 2002,2003,2004,2005,2006 SOUALEM Nadir
 * copyright © 2011 COLLARD Christophe
 * copyright © 2003,2004,2005,2006,2011 Centre National de la Recherche Scientifique
 * copyright © 2003,2004,2005,2006 Arts et Métiers ParisTech
 * copyright © 2003,2004,2005,2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2003,2004,2005,2006 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2003,2004,2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2011 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \class mesh
    \brief mesh library \n

    \htmlonly 
    <FONT color="#838383">

    mesh belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2002, 2003, 2004, 2005, 2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2002, 2003, 2004, 2005, 2006 SOUALEM Nadir \n
	     copyright \htmlonly &#169; \endhtmlonly 2011 Christophe COLLARD \n
             copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2011 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
	     copyright \htmlonly &#169; 2011 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 2.6.0
    \date 2003-2011
    \bug none
    \warning none
*/

/*
                      O____O_______O
                     /    /\      /\
                    /    /  \    /  \
              |===>/    /    \  /    \  
elements<====>|   O____O______\O______O <===========|
              |=======>\      /\      /             |========>nodes
                        \    /  \    /       /======|
                         \  /    \  /       /
                          \O______\O <=====/
*/
 
#ifndef __cplusplus
#error Must use C++ for the type mesh
#endif

#ifndef _mesh_h
#define _mesh_h


#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__TIME_H)
#include <time.h>
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__NODE_H)
#include "node.h"
#endif

#if !defined(__TRIANGLE_H)
#include "triangle.h"
#endif

#if !defined(__TETRAHEDRON_H)
#include "tetrahedron.h"
#endif

using namespace std;
using namespace mol;

namespace femol
{


//======================================
template <class obj, class T> class mesh
//======================================
{
  protected:
    int alive;          // 1 if the mesh exists else 0
    int dimspace_;      // 2 or 3 <===> R2 or R3
    int nbvertex_;      // number of vertices
    int nbelement_;     // number of elements
    node<T>* vertices;  // array of vertices
    obj* element;       // array of elements

  public:
    mesh () {alive=0;}  // default constructor
    mesh (const char*, const char* = "");  // constructor (file of nodes, file of elements)
    mesh (const mesh<obj,T>&);  // copy constructor
    ~mesh ();

    mesh<obj,T>& operator = (const mesh<obj,T>&);  // mesh affectation
    int dimspace () {return dimspace_;}    // returns space dimension
    int nbvertex () {return nbvertex_;}    // returns number of vertex
    int nbelement () {return nbelement_;}  // returns number of element
    obj& operator [] (int) const;          // returns or modify an element
    node<T>& operator () (int) const;      // returns or modify a node
    void make_anisotropic (const T&, const int&) const;  // make the mesh anisotropic
    int number (const node<T>&) const;     // returns the number of the node
    template <class Fobj, class Tf> friend ostream& operator << (ostream&, const mesh<Fobj,Tf>&);
    template <class Fobj, class Tf> friend bool operator == (const mesh<Fobj,Tf>&, const mesh<Fobj,Tf>&);
    void visu2d () const;  // mesh visualisation
    node<T>& vertex (int num) {return vertices[num-1];}
};


//=====Private methods for mesh==============================================


//=====Public methods for mesh===============================================


/*!
  \brief Constructor
  \n \n
  \param filename1 file with nodes and / or elements
  \param filename2 file with elements (if not included in filename1)
*/

//============================================================================================
template <class obj, class T> mesh<obj,T>::mesh (const char* filename1, const char* filename2)
//============================================================================================
{
  // read nodes
  alive = 1;
  ifstream file (filename1);
  assert (!file.fail());
  file >> dimspace_;
  file >> nbvertex_;
  vertices = new node<T> [nbvertex_];
  for (int i=0; i<nbvertex_; i++)
    vertices[i] = node<T> (dimspace_);
  for (int i=0; i<nbvertex_; ++i)
    file >> vertices[i];

  if (filename2 != "")
    { file.close();
      file.open (filename2);
      assert (!file.fail());
    }

  // read elements
  file >> nbelement_;
  element = new obj [nbelement_];
  node<T> ** nodes_of_element;
  nodes_of_element = new node<T>* [element[0].nb_nodes()];
  int num;

  for (int i=0; i<nbelement_; ++i)
    { for (int j=0; j<element[0].nb_nodes(); ++j)
	{ file >> num;
         nodes_of_element[j] = &vertices[num-1];
        }
      element[i] = obj (nodes_of_element);
    }
  delete [] nodes_of_element;
  file.close();
}


/*!
  \brief Copy constructor
  \param Th mesh to duplicate
*/

//=====================================================================
template <class obj, class T> mesh<obj,T>::mesh (const mesh<obj,T>& Th)
//=====================================================================
{
  alive = Th.alive;
  dimspace_ = Th.dimspace_;
  nbvertex_ = Th.nbvertex_;
  nbelement_ = Th.nbelement_;
  vertices = new node<T> [nbvertex_];
  for (int i=0; i<nbvertex_; i++)
    vertices[i] = node<T> (dimspace_);
  for (int i=0; i<nbvertex_; i++)
    vertices[i] = Th.vertices[i];

  element = new obj [nbelement_];

  node<T>** nodes_of_element;
  nodes_of_element = new node<T>* [element[0].nb_nodes()];

  for (int i=0; i<nbelement_; i++)
    { for (int j=0; j<element[0].nb_nodes(); j++)
	nodes_of_element[j] =& vertices[Th.number(Th[i+1][j+1])-1];
      element[i] = obj (nodes_of_element);
    }

  delete [] nodes_of_element;
}


/*!
  \brief Destructor
  \n \n
  Deletes vertices and element and set the scalar to 0
*/

//=================================================
template <class obj, class T> mesh<obj,T>::~mesh ()
//=================================================
{
  if (alive)
    { delete [] vertices;
      delete [] element;
    }
  dimspace_ = 0;
  nbvertex_ = 0;
  nbelement_ = 0;
  alive = 0;
}


/*!
  \brief Standard operator = for mesh \n \n
  \param Th mesh to copy \n
  \return reference of the left hand size mesh (for multiple equalities)
  \n\n
*/

//========================================================================================
template <class obj, class T> mesh<obj,T>& mesh<obj,T>::operator = (const mesh<obj,T>& Th)
//========================================================================================
{
  assert (Th.alive);
  if (!alive)
    { alive = 1;
      dimspace_ = Th.dimspace_;
      nbvertex_ = Th.nbvertex_;
      vertices = new node<T> [nbvertex_];
      for (int i=0; i<nbvertex_; i++)
	vertices[i] = node<T> (dimspace_);
      element = new obj [nbelement_ = Th.nbelement_];
    }
  else
    { assert (nbvertex_ == Th.nbvertex_);
      assert (nbelement_ == Th.nbelement_);
      assert (dimspace_ == Th.dimspace_);
    }

  for (int i=0; i<nbvertex_; i++)
    vertices[i] = Th.vertices[i];

  node<T>** nodes_of_element;
  nodes_of_element = new node<T>* [element[0].nb_nodes()];

  for (int i=0; i<nbelement_; i++)
    { for (int j=0; j<element[0].nb_nodes(); j++)
	nodes_of_element[j] =& vertices[Th.number(Th[i+1][j+1])-1];
      element[i] = obj (nodes_of_element);
    }
  delete [] nodes_of_element;

  return (*this);
}


/*!
  \brief Returns one element of the mesh \n \n
  Warning : we use the mathematical notations so the nodes adresses are in the set [1;nbelement_], and not [0:nbelement_-1] \n \n
  \param k element number \n \n
  \return the element k of the mesh \n \n
*/

//=======================================================================
template <class obj, class T> obj& mesh<obj,T>::operator [] (int k) const
//=======================================================================
{
  assert ((k > 0) && (k <= nbelement_));
  return element[k-1];
}


/*!
  \brief Returns one node of the mesh \n \n
  Warning : we use the mathematical notations so the nodes adresses are in the set [1;nbvertex_], and not [0:nbvertex_-1] \n \n
  \param k node number \n \n
  \return the node k of the mesh \n \n
*/

//===========================================================================
template <class obj, class T> node<T>& mesh<obj,T>::operator () (int k) const
//===========================================================================
{
  assert ((k > 0) && (k <= nbvertex_));
  return vertices[k-1];
}


/*!
  \brief  overloads output stream for mesh \n \n
  \param s output flow \n
  \param Th triangle
*/

//========================================================================================
template <class Fobj, class Tf> ostream& operator << (ostream& s, const mesh<Fobj,Tf>& Th)
//========================================================================================
{
  assert (Th.alive);
  s << "*===========================================================*\n";
  s << "*                 Datas Listing of the Mesh                 *\n";
  s << "*===========================================================*\n";
  s << "_______________________" << "\n";
  s << "Elements     |" << setw(8) << Th.nbelement_ << "|\n";
  s << "_______________________" << "\n";
  s << "Nodes        |" << setw(8) << Th.nbvertex_ << "|\n";
  s << "_______________________" << "\n";
  s << endl;

  return s;
}


/*!
  \brief  Gives the global number of a given node in the mesh \n \n
  \param A node \n
  \return int The global number of node A in the mesh
*/

//============================================================================
template <class obj, class T> int mesh<obj,T>::number (const node<T>& A) const
//============================================================================
{
  int number = (&A - vertices) + 1;
  assert ((number > 0) && (number <= nbvertex_));
  return number;
}


/*!
  \brief Changes an isotropic mesh in an anisotropic one \n \n
  \param para_tau tau parameter to set the degree of anisotropy \n
  \param cas allows to give the way to refine (read the function itself)
*/

//========================================================================================================
template <class obj, class T> void mesh<obj,T>::make_anisotropic (const T& para_tau, const int& cas) const
//========================================================================================================
{
  if (!cas) // raffinement sur le cube a la Shishkin pour le 2D
    { T dx, dxfin, dxgros;
      dx = 1 / (powl(nbvertex_, 0.5) - 1);
      dxfin = 2 * para_tau * dx;
      dxgros = 2 * (1 - para_tau) * dx;
      cout << "\n";
      cout << "Attention, le maillage est rendu anisotrope (shishkin) \n";
      cout << "Cela ne concerne que un type très particulier de maillages \n";

      for (int i=0; i<nbvertex_; i++)
	{ if (vertices[i][1] <= 0.5)
	    vertices[i][1] = vertices[i][1] / dx * dxfin;
	  else
	    vertices[i][1] = para_tau + (vertices[i][1] - 0.5) / dx * dxgros;
	}
    }
  else if (cas == 1) // raffinement sur le cube a la Shishkin pour le 3D
    { T dx, dxfin, dxgros;
      dx = 1 / (pow (nbvertex_, 0.33333333) - 1);
      dxfin = 2 * para_tau * dx;
      dxgros = 2 * (1 - para_tau) * dx;
      cout << "\n";
      cout << "Attention, le maillage est rendu anisotrope (shishkin) \n";
      cout << "Cela ne concerne que un type très particulier de maillages \n";

      for (int i=0; i<nbvertex_; i++)
	{ if (vertices[i][1] <= 0.5)
	    vertices[i][1] = vertices[i][1] / dx * dxfin;
	  else
	    vertices[i][1] = para_tau + (vertices[i][1] - 0.5) / dx * dxgros;
	}
    }
  else if ((cas == 2) || (cas == 3)) // raffinement sur le L Shape arrondi
    { T r;
      cout << "\n";
      cout << "Attention, le maillage est rendu anisotrope avec mu = " << para_tau << "\n";
      cout << "Cela ne concerne que un type très particulier de maillages \n";
      for (int i=0; i<nbvertex_; i++)
	{ r = sqrt (pow (vertices[i][1], 2) + pow (vertices[i][2], 2));
	  vertices[i][1] = vertices[i][1] * pow (r, 1 / para_tau - 1);
	  vertices[i][2] = vertices[i][2] * pow (r, 1 / para_tau - 1);
	}
    }
  else if (cas == 4) // raffinement sur le L Shape arrondi de rayon 0.1
    { T r;
      cout << "\n";
      cout << "Attention, le maillage est rendu anisotrope avec mu = " << para_tau << "\n";
      cout << "Cela ne concerne que un type très particulier de maillages \n";
      for (int i=0; i<nbvertex_; i++)
	{ r = sqrt (pow (vertices[i][1], 2) + pow (vertices[i][2], 2));
	  vertices[i][1] = vertices[i][1] * pow (r, 1 / para_tau - 1) * pow (10, 1 / para_tau);
	  vertices[i][2] = vertices[i][2] * pow (r, 1 / para_tau - 1) * pow (10, 1 / para_tau);
	}
    }
  else
    { cout << "comment faut-il raffiner le maillage ?" << "\n";
      assert (cas == 1);
    }
}


/*!
  \brief  Check the equality of two meshes \n \n
  \param Sh mesh \n
  \param Th mesh \n
  \return 1 if equal, 0 if not
*/

//=================================================================
template <class Fobj, class Tf>
bool operator == (const mesh<Fobj,Tf>& Sh, const mesh<Fobj,Tf>& Th)
//=================================================================
{
  bool boolean = (Sh.alive == Th.alive) && (Sh.dimspace_ == Th.dimspace_);
  boolean *= (Sh.nbvertex_ == Th.nbvertex_) && (Sh.nbelement_ == Th.nbelement_);

  for (int i=0; i<Sh.nbvertex_ && boolean; i++)
    boolean *= (Sh.vertices[i] == Th.vertices[i]);

  for(int i=0; i<Sh.nbelement_ && boolean; i++)
    boolean *= (Sh.element[i] == Th.element[i]);

  return boolean;
}


/*!
  \brief  Gives a graphical representation of the mesh \n \n
  \return A graphical windows containing the mesh drawn by gnuplot
*/

//===============================
template <class obj, class T>
void mesh<obj,T>::visu2d () const
//===============================
{
  ofstream file1 ("mesh.dat");
  for (int i=1; i<=nbelement_; ++i)
    { for (int j=1; j<=element[0].nb_nodes(); ++j)
	{ for (int k=1; k<=dimspace_; ++k)
	    file1 << element[i-1][j][k] << "  ";
	  file1 << endl;
	}
      for (int k=1; k<=dimspace_; ++k)
	file1 << element[i-1][1][k] << "  ";
      file1 << endl << endl;
    }
  file1.close ();
  ofstream file2 ("mesh.gnu");
  //file2 << "set terminal postscript" << endl;
  //file2 << "set output \" maillage.ps \" " << endl;
  file2 << "plot \"mesh.dat\" with linespoints" << endl;
  file2 << "pause -1 \"Press Enter\" " << endl;
  file2.close ();
  system ("gnuplot mesh.gnu");
  remove ("mesh.dat");
  remove ("mesh.gnu");
}


}


#endif
