/***************************************************************************
 *   Copyright (C) 2004 by Predrag Viceic                                  *
 *   viceic@net2000.ch                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "soundfilepreview.h"

SoundFilePreview::SoundFilePreview(SoundManager* sm,SoundPlayer* pl, FileManager* fm,
                                                            QWidget *parent, const char *name)
 : QWidget(parent, name), QFilePreview()
{
    setSizePolicy(QSizePolicy::MinimumExpanding,QSizePolicy::MinimumExpanding);
    playing=FALSE;
    fileManager=fm;
    soundManager=sm;
    soundPlayer=pl;
    setFixedHeight(parentWidget()->height()); //can do better...
    mainFrame=new QVBox(this);

    mainFrame->setLineWidth(1);
    //mainFrame->setFixedWidth(210);
    mainFrame->setSizePolicy(QSizePolicy::MinimumExpanding,QSizePolicy::MinimumExpanding);
    mainFrame->setFixedHeight(height());

    audioPreviewFrame=new QHBox(mainFrame);
    audioPreviewFrame->setFrameStyle(QFrame::Box | QFrame::Raised);
    audioPreviewFrame->setSizePolicy(QSizePolicy::MinimumExpanding,QSizePolicy::MinimumExpanding);
    //audioPreviewFrame->setFixedWidth(mainFrame->contentsRect().width());

    vbox=new QVBox(audioPreviewFrame);
    hbox=new QHBox(vbox);
    hbox->setFixedHeight(33);
    vbox->setSizePolicy(QSizePolicy::MinimumExpanding,QSizePolicy::MinimumExpanding);

    playButton=new QPushButton("Play",hbox);
    playButton->setPixmap(QPixmap(qembed_findImage("player_play")));
    playButton->setToggleButton(TRUE);

    pauseButton=new QPushButton("Pause",hbox);
    pauseButton->setPixmap(QPixmap(qembed_findImage("player_pause")));
    pauseButton->setToggleButton(TRUE);


    stopButton=new QPushButton("Stop",hbox);
    stopButton->setPixmap(QPixmap(qembed_findImage("player_stop")));
    stopButton->setToggleButton(TRUE);
    stopButton->setOn(TRUE);


    waveDisplay=new PreviewWaveDisplay(vbox);
    waveDisplay->setSizePolicy(QSizePolicy::MinimumExpanding,QSizePolicy::MinimumExpanding);
    waveDisplay->setFrameStyle(QFrame::Box |QFrame::Sunken);
    vbox->setFixedHeight(210);
    positionScrollBar=new QScrollBar(vbox);
    positionScrollBar->setOrientation(Qt::Horizontal);
    positionScrollBar->setMinValue(0);
    positionScrollBar->setMaxValue(0);
    positionScrollBar->setValue(positionScrollBar->maxValue());

    zoomSlider=new QSlider(vbox);
    zoomSlider->setOrientation(Qt::Horizontal);
    zoomSlider->setMinValue(1000);
    zoomSlider->setMaxValue(10000000);
    zoomSlider->setValue(zoomSlider->maxValue());

    QWidget* empty=new QWidget(mainFrame);
    mainFrame->setStretchFactor(empty,10);

    mainFrame->hide();

    connect(playButton,SIGNAL(clicked()),this,SLOT(play()));
    connect(stopButton,SIGNAL(clicked()),this,SLOT(stop()));
    connect(pauseButton,SIGNAL(clicked()),this,SLOT(pause()));
    connect(zoomSlider,SIGNAL(valueChanged(int)),waveDisplay,SLOT(setDisplayedLength(int)));
    connect(positionScrollBar,SIGNAL(valueChanged(int)),waveDisplay,SLOT(setStartDisplayPosition(int)));
    connect(waveDisplay,SIGNAL(maxStartPositionChangedSig(int)),
                        this,SLOT(maxStartDisplayPositionChanged(int)));
    connect(waveDisplay,SIGNAL(stopSig()),
            this,SLOT(stopButtonOnly()));
    connect(waveDisplay,SIGNAL(pauseSig()),
            this,SLOT(pause()));
    connect(waveDisplay,SIGNAL(playSig()),
            this,SLOT(play()));
    connect(waveDisplay,SIGNAL(startPositionChangedSig(int)),
            this,SLOT(startPositionChanged(int)));
    connect(waveDisplay,SIGNAL(playPositionChangedSig(long)),
            this,SLOT(setPlayPosition(long)));


}

void SoundFilePreview::resizeEvent ( QResizeEvent * ev){
    mainFrame->resize(ev->size());
    waveDisplay->update();
}


SoundFilePreview::~SoundFilePreview()
{
    if(soundPlayer) soundPlayer->removeSoundMonitor(waveDisplay);
}




/*!
    \fn SoundFilePreview::previewUrl(const QUrl & url)
 */
void SoundFilePreview::previewUrl(const QUrl & url)
{
    if (QFileInfo(url.path()).isFile()){
        path=url.path();              
        if(waveDisplay){
            mainFrame->show();
            positionScrollBar->setValue(0);
            positionScrollBar->setMaxValue(0);
            if(playing){
                stop();
                soundPlayer->setPosition(0);
                waveDisplay->setSoundFile(&path);
                //play();
            }else{
                soundPlayer->setPosition(0);
                waveDisplay->setSoundFile(&path);
            }            
            
            zoomSlider->setMaxValue(waveDisplay->getWaveLength());
            zoomSlider->setValue(zoomSlider->maxValue());
            waveDisplay->update();            
        }
    }else{
            path='0';
            stop();
            mainFrame->hide();
            soundPlayer->setPosition(0);
            if(waveDisplay){              
                waveDisplay->setSoundFile(0);
            }
    }
}


/*!
    \fn SoundFilePreview::play()
 */
void SoundFilePreview::play()
{
    if(soundPlayer->getOutputType()==SoundPlayer::OUTPUT_DEVDSP){
        if(!soundPlayer->running()) soundPlayer->start();
    }
    if(soundPlayer->getOutputType()!=SoundPlayer::OUTPUT_NONE){
        soundPlayer->playFromFile(path);
        playing=TRUE;
        playButton->setOn(TRUE);
        pauseButton->setOn(FALSE);
        stopButton->setOn(FALSE);
    }
}


/*!
    \fn SoundFilePreview::stop()
 */
void SoundFilePreview::stop()
{
    if(soundPlayer->getOutputType()!=SoundPlayer::OUTPUT_NONE){
        if(playing){
            soundPlayer->stopPlayFromFile();
            playing=FALSE;
        }
        playButton->setOn(FALSE);
        pauseButton->setOn(FALSE);
        stopButton->setOn(TRUE);
    }
}

/*!
    \fn SoundFilePreview::stop()
 */
void SoundFilePreview::stopButtonOnly()
{
    if(soundPlayer->getOutputType()!=SoundPlayer::OUTPUT_NONE){
        if(playing){
            playing=FALSE;
            playButton->setOn(FALSE);
            pauseButton->setOn(FALSE);
            stopButton->setOn(TRUE);
        }
    }
}

/*!
    \fn SoundFilePreview::pause()
 */
void SoundFilePreview::pause()
{
    if(soundPlayer->getOutputType()!=SoundPlayer::OUTPUT_NONE){
        if(playing){
            soundPlayer->pausePlayFromFile();
        }//playing=FALSE;
        playButton->setOn(FALSE);
        stopButton->setOn(FALSE);
        pauseButton->setOn(TRUE);
    }
}



/*!
    \fn SoundFilePreview::getPlaying()
 */
bool SoundFilePreview::getPlaying()
{
    return playing;
}


/*!
    \fn SoundFilePreview::maxStartDisplayPositionChanged(int)
 */
void SoundFilePreview::maxStartDisplayPositionChanged(int pos)
{
    positionScrollBar->setRange(0,pos);
    positionScrollBar->setLineStep(waveDisplay->getJump()*10);
    positionScrollBar->setPageStep(waveDisplay->getJump()*waveDisplay->width());
}




/*!
    \fn SoundFilePreview::startPositionChanged(int sp)
 */
void SoundFilePreview::startPositionChanged(int sp)
{
    positionScrollBar->setValue(sp);
}

void SoundFilePreview::closeEvent( QCloseEvent* ce )
{
    soundPlayer->removeSoundMonitor(waveDisplay);
    if(soundPlayer->running()){
        soundPlayer->exit(0);
    }
    ce->accept();
    return;
}




/*!
    \fn SoundFilePreview::init()
 */
void SoundFilePreview::init()
{
    soundPlayer->addSoundMonitor(waveDisplay);
}

/*!
    \fn SoundFilePreview::sizeHint()
 */
QSize SoundFilePreview::minimumSizeHint() const
{
    return QSize(250,200);
}

void SoundFilePreview::setPlayPosition(long pp){
    soundPlayer->setPosition(pp);
}


