!------------------------------------------------------------------------------
!    This code is part of the MondoSCF suite of programs for linear scaling
!    electronic structure theory and ab initio molecular dynamics.
!
!    Copyright (2004). The Regents of the University of California. This
!    material was produced under U.S. Government contract W-7405-ENG-36
!    for Los Alamos National Laboratory, which is operated by the University
!    of California for the U.S. Department of Energy. The U.S. Government has
!    rights to use, reproduce, and distribute this software.  NEITHER THE
!    GOVERNMENT NOR THE UNIVERSITY MAKES ANY WARRANTY, EXPRESS OR IMPLIED,
!    OR ASSUMES ANY LIABILITY FOR THE USE OF THIS SOFTWARE.
!
!    This program is free software; you can redistribute it and/or modify
!    it under the terms of the GNU General Public License as published by the
!    Free Software Foundation; either version 2 of the License, or (at your
!    option) any later version. Accordingly, this program is distributed in
!    the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
!    the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
!    PURPOSE. See the GNU General Public License at www.gnu.org for details.
!
!    While you may do as you like with this software, the GNU license requires
!    that you clearly mark derivative software.  In addition, you are encouraged
!    to return derivative works to the MondoSCF group for review, and possible
!    disemination in future releases.
!------------------------------------------------------------------------------
MODULE PBlokGuess
   USE DerivedTypes
   USE GlobalScalars
   USE Parse
   USE PrettyPrint
   USE LinAlg
   USE MatFunk
   IMPLICIT NONE
   CONTAINS
   SUBROUTINE FillPBlok(N,K,PBlok,Random_O)
      INTEGER,                    INTENT(IN)  :: N,K
      REAL(DOUBLE),DIMENSION(N,N),INTENT(OUT) :: PBlok
      REAL(DOUBLE),DIMENSION(50,104)          :: AvP
      INTEGER                                 :: I,J
      LOGICAL,OPTIONAL                        :: Random_O
!------------------------------------------------------
      !
      IF(K>55.OR.PRESENT(Random_O))THEN
         DO I=1,N
            DO J=1,N
               PBlok(I,J)=Random((/-1D0,1D0/))
            ENDDO
         ENDDO
         ! Semi-normalize
         PBlok=PBlok*DBLE(2*N)/SUM(PBlok)
         RETURN
      ENDIF
!------------------------------------------------------
!     H
!
      AvP(1,1)=1.D0/2.D0
!--------------------------------------
!     He
!
      AvP(1,2)=1.0D0
!--------------------------------------
!     Li
!
      AvP(1,3)=1.0D0
      AvP(2,3)=1.0D0/8.0D0
      AvP(3,3)=1.0D0/8.0D0
      AvP(4,3)=1.0D0/8.0D0
      AvP(5,3)=1.0D0/8.0D0
!--------------------------------------
!     Be
!
      AvP(1,4)=1.0D0
      AvP(2,4)=1.0D0/2.0D0
      AvP(3,4)=1.0D0/6.0D0
      AvP(4,4)=1.0D0/6.0D0
      AvP(5,4)=1.0D0/6.0D0
!---------------------------------------
!     B
!
      AvP(1,5)=1.0D0
      AvP(2,5)=1.0D0/2.0D0
      AvP(3,5)=1.0D0/3.0D0
      AvP(4,5)=1.0D0/3.0D0
      AvP(5,5)=1.0D0/3.0D0
!--------------------------------------
!     C Z=6 C [1s2 2s2 2p2] 3P
!
      AvP(1,6)=1.0D0
      AvP(2,6)=1.0D0
      AvP(3,6)=1.0D0/3.0D0
      AvP(4,6)=1.0D0/3.0D0
      AvP(5,6)=1.0D0/3.0D0
!--------------------------------------
!     N Z=7 N [1s2 2s2 2p3] 4S
!
      AvP(1,7)=1.0D0
      AvP(2,7)=1.0D0
      AvP(3,7)=1.0D0/2.0D0
      AvP(4,7)=1.0D0/2.0D0
      AvP(5,7)=1.0D0/2.0D0
!--------------------------------------
!     O Z=8 O [1s2 2s2 2p4] 3P
!
      AvP(1,8)=1.0D0
      AvP(2,8)=1.0D0
      AvP(3,8)=2.0D0/3.0D0
      AvP(4,8)=2.0D0/3.0D0
      AvP(5,8)=2.0D0/3.0D0
!------------------------------------
!     F
!
      AvP(1,9)=1.0D0
      AvP(2,9)=1.0D0
      AvP(3,9)=5.0D0/6.0D0
      AvP(4,9)=5.0D0/6.0D0
      AvP(5,9)=5.0D0/6.0D0
!------------------------------------
!     Ne
!
      AvP(1,10)=1.0D0
      AvP(2,10)=1.0D0
      AvP(3,10)=1.0D0
      AvP(4,10)=1.0D0
      AvP(5,10)=1.0D0
!------------------------------------
!     Na
!
      AvP(1,11)=1.0D0
      AvP(2,11)=1.0D0
      AvP(3,11)=1.0D0
      AvP(4,11)=1.0D0
      AvP(5,11)=1.0D0
      AvP(6,11)=0.5D0
      AvP(7,11)=0.0D0
      AvP(8,11)=0.0D0
      AvP(9,11)=0.0D0
!------------------------------------
!     Mg
!
      AvP(1,12)=1.0D0
      AvP(2,12)=1.0D0
      AvP(3,12)=1.0D0
      AvP(4,12)=1.0D0
      AvP(5,12)=1.0D0
      AvP(6,12)=1.0D0/2.0D0      ! 3s
      AvP(7,12)=1.0D0/6.0D0      ! 3p
      AvP(8,12)=1.0D0/6.0D0      ! 3p
      AvP(9,12)=1.0D0/6.0D0      ! 3p
!------------------------------------------
!     Al
!
      AvP(1,13)=1.0D0
      AvP(2,13)=1.0D0
      AvP(3,13)=1.0D0
      AvP(4,13)=1.0D0
      AvP(5,13)=1.0D0
      AvP(6,13)=1.0D0/2.0D0   ! 3s
      AvP(7,13)=1.0D0/3.0D0   ! 3p
      AvP(8,13)=1.0D0/3.0D0   ! 3p
      AvP(9,13)=1.0D0/3.0D0   ! 3p
!------------------------------------------
!     Si
!
      AvP(1,14)=1.0D0         ! 1s
      AvP(2,14)=1.0D0         ! 2s
      AvP(3,14)=1.0D0         ! 2p
      AvP(4,14)=1.0D0         ! 2p
      AvP(5,14)=1.0D0         ! 2p
      AvP(6,14)=1.0D0/2.0D0   ! 3s
      AvP(7,14)=1.0D0/2.0D0   ! 3p
      AvP(8,14)=1.0D0/2.0D0   ! 3p
      AvP(9,14)=1.0D0/2.0D0   ! 3p
!------------------------------------------
!     P
!
      AvP(1,15)=1.0D0
      AvP(2,15)=1.0D0
      AvP(3,15)=1.0D0
      AvP(4,15)=1.0D0
      AvP(5,15)=1.0D0
      AvP(6,15)=1.0D0         ! 3s
      AvP(7,15)=1.0D0/2.0D0   ! 3p
      AvP(8,15)=1.0D0/2.0D0   ! 3p
      AvP(9,15)=1.0D0/2.0D0   ! 3p
!-----------------------------------------
!     S Z=16 S [1s2 2s2 2p6 3s2 3p4] 3P
!
      AvP(1,16)=1.0D0         ! 1s
      AvP(2,16)=1.0D0         ! 2s
      AvP(3,16)=1.0D0         ! 2p
      AvP(4,16)=1.0D0         ! 2p
      AvP(5,16)=1.0D0         ! 2p
      AvP(6,16)=1.0D0         ! 3s
      AvP(7,16)=2.0D0/3.0D0   ! 3p
      AvP(8,16)=2.0D0/3.0D0   ! 3p
      AvP(9,16)=2.0D0/3.0D0   ! 3p
!-----------------------------------------
!     Cl Z=17 S [1s2 2s2 2p6 3s2 3p5] 3P
!
      AvP(1,17)=1.0D0         ! 1s
      AvP(2,17)=1.0D0         ! 2s
      AvP(3,17)=1.0D0         ! 2p
      AvP(4,17)=1.0D0         ! 2p
      AvP(5,17)=1.0D0         ! 2p
      AvP(6,17)=1.0D0         ! 3s
      AvP(7,17)=5.0D0/6.0D0   ! 3p
      AvP(8,17)=5.0D0/6.0D0   ! 3p
      AvP(9,17)=5.0D0/6.0D0   ! 3p
!-----------------------------------------
!     Ar Z=18 S [1s2 2s2 2p6 3s2 3p6] 3P
!
      AvP(1,18)=1.0D0         ! 1s
      AvP(2,18)=1.0D0         ! 2s
      AvP(3,18)=1.0D0         ! 2p
      AvP(4,18)=1.0D0         ! 2p
      AvP(5,18)=1.0D0         ! 2p
      AvP(6,18)=1.0D0         ! 3s
      AvP(7,18)=1.0D0         ! 3p
      AvP(8,18)=1.0D0         ! 3p
      AvP(9,18)=1.0D0         ! 3p
!-----------------------------------------
!     K Z=19 [1s2 2s2 2p6 3s2 3p6 4s1]
!
      AvP( 1,19)=1.0D0        ! 1s
      AvP( 2,19)=1.0D0        ! 2s
      AvP( 3,19)=1.0D0        ! 2p
      AvP( 4,19)=1.0D0        ! 2p
      AvP( 5,19)=1.0D0        ! 2p
      AvP( 6,19)=1.0D0        ! 3s
      AvP( 7,19)=1.0D0        ! 3p
      AvP( 8,19)=1.0D0        ! 3p
      AvP( 9,19)=1.0D0        ! 3p
      AvP(10,19)=1.0D0/2.0D0  ! 4s
!-----------------------------------------
!     Ca Z=20 [1s2 2s2 2p6 3s2 3p6 4s2]
!
      AvP( 1,20)=1.0D0        ! 1s
      AvP( 2,20)=1.0D0        ! 2s
      AvP( 3,20)=1.0D0        ! 2p
      AvP( 4,20)=1.0D0        ! 2p
      AvP( 5,20)=1.0D0        ! 2p
      AvP( 6,20)=1.0D0        ! 3s
      AvP( 7,20)=1.0D0        ! 3p
      AvP( 8,20)=1.0D0        ! 3p
      AvP( 9,20)=1.0D0        ! 3p
      AvP(10,20)=1.0D0        ! 4s
!-----------------------------------------
!     Sc Z=21 [1s2 2s2 2p6 3s2 3p6 4s2 3d1 ]
!
      AvP( 1,21)=1.0D0         ! 1s   2
      AvP( 2,21)=1.0D0         ! 2s   2
      AvP( 3,21)=1.0D0         ! 2p
      AvP( 4,21)=1.0D0         ! 2p   6
      AvP( 5,21)=1.0D0         ! 2p
      AvP( 6,21)=1.0D0         ! 3s   2
      AvP( 7,21)=1.0D0         ! 3p
      AvP( 8,21)=1.0D0         ! 3p   6
      AvP( 9,21)=1.0D0         ! 3p
      AvP(10,21)=1.0D0/12.0D0  ! 3d2
      AvP(11,21)=1.0D0/12.0D0  ! 3d2
      AvP(12,21)=1.0D0/12.0D0  ! 3d2
      AvP(13,21)=1.0D0/12.0D0  ! 3d2  1
      AvP(14,21)=1.0D0/12.0D0  ! 3d2
      AvP(15,21)=1.0D0/12.0D0  ! 3d2
      AvP(16,21)=1.0D0         ! 4s2  2
      AvP(17,21)=0.0D0         ! 4p6
      AvP(18,21)=0.0D0         ! 4p6  0
      AvP(19,21)=0.0D0         ! 46p
!-----------------------------------------
!     Ti Z=22 S [1s2 2s2 2p6 3s2 3p6 4s2 3d2 ] 4P
!
      AvP(1,22)=1.0D0          ! 1s   2
      AvP(2,22)=1.0D0          ! 2s   2
      AvP(3,22)=1.0D0          ! 2p
      AvP(4,22)=1.0D0          ! 2p   6
      AvP(5,22)=1.0D0          ! 2p
      AvP(6,22)=1.0D0          ! 3s   2
      AvP(7,22)=1.0D0          ! 3p
      AvP(8,22)=1.0D0          ! 3p   6
      AvP(9,22)=1.0D0          ! 3p
      AvP(10,22)=1.0D0/6.0D0   ! 3d2
      AvP(11,22)=1.0D0/6.0D0   ! 3d2
      AvP(12,22)=1.0D0/6.0D0   ! 3d2
      AvP(13,22)=1.0D0/6.0D0   ! 3d2  2
      AvP(14,22)=1.0D0/6.0D0   ! 3d2
      AvP(15,22)=1.0D0/6.0D0   ! 3d2
      AvP(16,22)=1.0D0         ! 4s2  2
      AvP(17,22)=0.0D0         ! 4p6
      AvP(18,22)=0.0D0         ! 4p6  0
      AvP(19,22)=0.0D0         ! 46p
!-----------------------------------------
!     Cr Z=24 S [1s2 2s2 2p6 3s2 3p6 4s1 3d5 ] 4P0
!
      AvP(1,24)=1.0D0          ! 1s   2
      AvP(2,24)=1.0D0          ! 2s   2
      AvP(3,24)=1.0D0          ! 2p
      AvP(4,24)=1.0D0          ! 2p   6
      AvP(5,24)=1.0D0          ! 2p
      AvP(6,24)=1.0D0          ! 3s   2
      AvP(7,24)=1.0D0          ! 3p
      AvP(8,24)=1.0D0          ! 3p   6
      AvP(9,24)=1.0D0          ! 3p
      AvP(10,24)=5.0D0/12.0D0  ! 3d5
      AvP(11,24)=5.0D0/12.0D0  ! 3d5
      AvP(12,24)=5.0D0/12.0D0  ! 3d5
      AvP(13,24)=5.0D0/12.0D0  ! 3d5  5
      AvP(14,24)=5.0D0/12.0D0  ! 3d5
      AvP(15,24)=5.0D0/12.0D0  ! 3d5
      AvP(16,24)=1.0D0/2.0D0   ! 4s1  1
      AvP(17,24)=0.0D0         ! 4p0
      AvP(18,24)=0.0D0         ! 4p0  0
      AvP(19,24)=0.0D0         ! 4p0
!-----------------------------------------
!     Mn Z=25 S [1s2 2s2 2p6 3s2 3p6 4s2 3d5 ] 4P0
!
      AvP(1,25)=1.0D0          ! 1s   2
      AvP(2,25)=1.0D0          ! 2s   2
      AvP(3,25)=1.0D0          ! 2p
      AvP(4,25)=1.0D0          ! 2p   6
      AvP(5,25)=1.0D0          ! 2p
      AvP(6,25)=1.0D0          ! 3s   2
      AvP(7,25)=1.0D0          ! 3p
      AvP(8,25)=1.0D0          ! 3p   6
      AvP(9,25)=1.0D0          ! 3p
      AvP(10,25)=5.0D0/12.0D0  ! 3d5
      AvP(11,25)=5.0D0/12.0D0  ! 3d5
      AvP(12,25)=5.0D0/12.0D0  ! 3d5
      AvP(13,25)=5.0D0/12.0D0  ! 3d5  5
      AvP(14,25)=5.0D0/12.0D0  ! 3d5
      AvP(15,25)=5.0D0/12.0D0  ! 3d5
      AvP(16,25)=1.0D0         ! 4s2  2
      AvP(17,25)=0.0D0         ! 4p0
      AvP(18,25)=0.0D0         ! 4p0  0
      AvP(19,25)=0.0D0         ! 4p0
!-----------------------------------------
!     Fe Z=26 S [1s2 2s2 2p6 3s2 3p6 3d6 4s2] 4p0
!
      AvP(1 ,26)=1.0D0         ! 1s2   2
      AvP(2 ,26)=1.0D0         ! 2s2
      AvP(3 ,26)=1.0D0         ! 2p6
      AvP(4 ,26)=1.0D0         ! 2p6   8
      AvP(5 ,26)=1.0D0         ! 2p6
      AvP(6 ,26)=1.0D0         ! 3s2
      AvP(7 ,26)=1.0D0         ! 3p6
      AvP(8 ,26)=1.0D0         ! 3p6   8
      AvP(9 ,26)=1.0D0         ! 3p6
      AvP(10,26)=1.0D0/2.0D0   ! 3d6
      AvP(11,26)=1.0D0/2.0D0   ! 3d6
      AvP(12,26)=1.0D0/2.0D0   ! 3d6
      AvP(13,26)=1.0D0/2.0D0   ! 3d6   6
      AvP(14,26)=1.0D0/2.0D0   ! 3d6
      AvP(15,26)=1.0D0/2.0D0   ! 3d6
      AvP(16,26)=1.0D0         ! 4s2   2
      AvP(17,26)=0.0D0         ! 4p0
      AvP(18,26)=0.0D0         ! 4p0   0
      AvP(19,26)=0.0D0         ! 4p0
!-----------------------------------------
!     Cu Z=29 S [1s2 2s2 2p6 3s2 3p6 3d6 4s2 4p1]
!
      AvP(1 ,29)=1.0D0         ! 1s2  2
      AvP(2 ,29)=1.0D0         ! 2s2
      AvP(3 ,29)=1.0D0         ! 2p2
      AvP(4 ,29)=1.0D0         ! 2p2  8
      AvP(5 ,29)=1.0D0         ! 2p2
      AvP(6 ,29)=1.0D0         ! 3s2
      AvP(7 ,29)=1.0D0         ! 3p2
      AvP(8 ,29)=1.0D0         ! 3p2  8
      AvP(9 ,29)=1.0D0         ! 3p2
      AvP(10,29)=2.0D0/3.0D0   ! 3d10
      AvP(11,29)=2.0D0/3.0D0   ! 3d10
      AvP(12,29)=2.0D0/3.0D0   ! 3d10
      AvP(13,29)=2.0D0/3.0D0   ! 3d10 8
      AvP(14,29)=2.0D0/3.0D0   ! 3d10
      AvP(15,29)=2.0D0/3.0D0   ! 3d10
      AvP(16,29)=1.0D0         ! 4s2  2
      AvP(17,29)=1.0D0/6.0D0   ! 4p6
      AvP(18,29)=1.0D0/6.0D0   ! 4p6  1
      AvP(19,29)=1.0D0/6.0D0   ! 4p6
!-----------------------------------------
!     Zn Z=30 S [1s2 2s2 2p6 3s2 3p6 3d6 4s2 4p2]
!
      AvP(1 ,30)=1.0D0         ! 1s2
      AvP(2 ,30)=1.0D0         ! 2s2
      AvP(3 ,30)=1.0D0         ! 2p2
      AvP(4 ,30)=1.0D0         ! 2p2
      AvP(5 ,30)=1.0D0         ! 2p2
      AvP(6 ,30)=1.0D0         ! 3s2
      AvP(7 ,30)=1.0D0         ! 3p2
      AvP(8 ,30)=1.0D0         ! 3p2
      AvP(9 ,30)=1.0D0         ! 3p2
      AvP(10,30)=2.0D0/3.0D0   ! 3d10
      AvP(11,30)=2.0D0/3.0D0   ! 3d10
      AvP(12,30)=2.0D0/3.0D0   ! 3d10
      AvP(13,30)=2.0D0/3.0D0   ! 3d10
      AvP(14,30)=2.0D0/3.0D0   ! 3d10
      AvP(15,30)=2.0D0/3.0D0   ! 3d10
      AvP(16,30)=1.0D0         ! 4s2
      AvP(17,30)=1.0D0/3.0D0   ! 4p2
      AvP(18,30)=1.0D0/3.0D0   ! 4p2
      AvP(19,30)=1.0D0/3.0D0   ! 4p2
!-----------------------------------------
!     Ga Z=31   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p1]
!
      AvP(1 ,31)=1.0D0         ! 1s2
      AvP(2 ,31)=1.0D0         ! 2s2
      AvP(3 ,31)=1.0D0         ! 2p6
      AvP(4 ,31)=1.0D0         ! 2p6
      AvP(5 ,31)=1.0D0         ! 2p6
      AvP(6 ,31)=1.0D0         ! 3s2
      AvP(7 ,31)=1.0D0         ! 3p6
      AvP(8 ,31)=1.0D0         ! 3p6
      AvP(9 ,31)=1.0D0         ! 3p6
      AvP(10,31)=5.0D0/6.0D0   ! 3d10
      AvP(11,31)=5.0D0/6.0D0   ! 3d10
      AvP(12,31)=5.0D0/6.0D0   ! 3d10
      AvP(13,31)=5.0D0/6.0D0   ! 3d10
      AvP(14,31)=5.0D0/6.0D0   ! 3d10
      AvP(15,31)=5.0D0/6.0D0   ! 3d10
      AvP(16,31)=1.0D0         ! 4s2
      AvP(17,31)=1.0D0/6.0D0   ! 4p1
      AvP(18,31)=1.0D0/6.0D0   ! 4p1
      AvP(19,31)=1.0D0/6.0D0   ! 4p1
!-----------------------------------------
!     Ge Z=32   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p2]
!
      AvP(1 ,32)=1.0D0         ! 1s2
      AvP(2 ,32)=1.0D0         ! 2s2
      AvP(3 ,32)=1.0D0         ! 2p6
      AvP(4 ,32)=1.0D0         ! 2p6
      AvP(5 ,32)=1.0D0         ! 2p6
      AvP(6 ,32)=1.0D0         ! 3s2
      AvP(7 ,32)=1.0D0         ! 3p6
      AvP(8 ,32)=1.0D0         ! 3p6
      AvP(9 ,32)=1.0D0         ! 3p6
      AvP(10,32)=5.0D0/6.0D0   ! 3d10
      AvP(11,32)=5.0D0/6.0D0   ! 3d10
      AvP(12,32)=5.0D0/6.0D0   ! 3d10
      AvP(13,32)=5.0D0/6.0D0   ! 3d10
      AvP(14,32)=5.0D0/6.0D0   ! 3d10
      AvP(15,32)=5.0D0/6.0D0   ! 3d10
      AvP(16,32)=1.0D0         ! 4s2
      AvP(17,32)=2.0D0/6.0D0   ! 4p2
      AvP(18,32)=2.0D0/6.0D0   ! 4p2
      AvP(19,32)=2.0D0/6.0D0   ! 4p2
!-----------------------------------------
!     As Z=33   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p3]
!
      AvP(1 ,33)=1.0D0         ! 1s2
      AvP(2 ,33)=1.0D0         ! 2s2
      AvP(3 ,33)=1.0D0         ! 2p6
      AvP(4 ,33)=1.0D0         ! 2p6
      AvP(5 ,33)=1.0D0         ! 2p6
      AvP(6 ,33)=1.0D0         ! 3s2
      AvP(7 ,33)=1.0D0         ! 3p6
      AvP(8 ,33)=1.0D0         ! 3p6
      AvP(9 ,33)=1.0D0         ! 3p6
      AvP(10,33)=5.0D0/6.0D0   ! 3d10
      AvP(11,33)=5.0D0/6.0D0   ! 3d10
      AvP(12,33)=5.0D0/6.0D0   ! 3d10
      AvP(13,33)=5.0D0/6.0D0   ! 3d10
      AvP(14,33)=5.0D0/6.0D0   ! 3d10
      AvP(15,33)=5.0D0/6.0D0   ! 3d10
      AvP(16,33)=1.0D0         ! 4s2
      AvP(17,33)=3.0D0/6.0D0   ! 4p3
      AvP(18,33)=3.0D0/6.0D0   ! 4p3
      AvP(19,33)=3.0D0/6.0D0   ! 4p3
!-----------------------------------------
!     Se Z=34   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p4]
!
      AvP(1 ,34)=1.0D0         ! 1s2
      AvP(2 ,34)=1.0D0         ! 2s2
      AvP(3 ,34)=1.0D0         ! 2p6
      AvP(4 ,34)=1.0D0         ! 2p6
      AvP(5 ,34)=1.0D0         ! 2p6
      AvP(6 ,34)=1.0D0         ! 3s2
      AvP(7 ,34)=1.0D0         ! 3p6
      AvP(8 ,34)=1.0D0         ! 3p6
      AvP(9 ,34)=1.0D0         ! 3p6
      AvP(10,34)=5.0D0/6.0D0   ! 3d10
      AvP(11,34)=5.0D0/6.0D0   ! 3d10
      AvP(12,34)=5.0D0/6.0D0   ! 3d10
      AvP(13,34)=5.0D0/6.0D0   ! 3d10
      AvP(14,34)=5.0D0/6.0D0   ! 3d10
      AvP(15,34)=5.0D0/6.0D0   ! 3d10
      AvP(16,34)=1.0D0         ! 4s2
      AvP(17,34)=2.0D0/3.0D0   ! 4p4
      AvP(18,34)=2.0D0/3.0D0   ! 4p4
      AvP(19,34)=2.0D0/3.0D0   ! 4p4
!-----------------------------------------
!     Br Z=35   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p5]
!
      AvP(1 ,35)=1.0D0         ! 1s2
      AvP(2 ,35)=1.0D0         ! 2s2
      AvP(3 ,35)=1.0D0         ! 2p6
      AvP(4 ,35)=1.0D0         ! 2p6
      AvP(5 ,35)=1.0D0         ! 2p6
      AvP(6 ,35)=1.0D0         ! 3s2
      AvP(7 ,35)=1.0D0         ! 3p6
      AvP(8 ,35)=1.0D0         ! 3p6
      AvP(9 ,35)=1.0D0         ! 3p6
      AvP(10,35)=5.0D0/6.0D0   ! 3d10
      AvP(11,35)=5.0D0/6.0D0   ! 3d10
      AvP(12,35)=5.0D0/6.0D0   ! 3d10
      AvP(13,35)=5.0D0/6.0D0   ! 3d10
      AvP(14,35)=5.0D0/6.0D0   ! 3d10
      AvP(15,35)=5.0D0/6.0D0   ! 3d10
      AvP(16,35)=1.0D0         ! 4s2
      AvP(17,35)=5.0D0/6.0D0   ! 4p5
      AvP(18,35)=5.0D0/6.0D0   ! 4p5
      AvP(19,35)=5.0D0/6.0D0   ! 4p5
!-----------------------------------------
!     Kr Z=36   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6]
!
      AvP(1 ,36)=1.0D0         ! 1s2
      AvP(2 ,36)=1.0D0         ! 2s2
      AvP(3 ,36)=1.0D0         ! 2p6
      AvP(4 ,36)=1.0D0         ! 2p6
      AvP(5 ,36)=1.0D0         ! 2p6
      AvP(6 ,36)=1.0D0         ! 3s2
      AvP(7 ,36)=1.0D0         ! 3p6
      AvP(8 ,36)=1.0D0         ! 3p6
      AvP(9 ,36)=1.0D0         ! 3p6
      AvP(10,36)=5.0D0/6.0D0   ! 3d10
      AvP(11,36)=5.0D0/6.0D0   ! 3d10
      AvP(12,36)=5.0D0/6.0D0   ! 3d10
      AvP(13,36)=5.0D0/6.0D0   ! 3d10
      AvP(14,36)=5.0D0/6.0D0   ! 3d10
      AvP(15,36)=5.0D0/6.0D0   ! 3d10
      AvP(16,36)=1.0D0         ! 4s2
      AvP(17,36)=1.0D0         ! 4p6
      AvP(18,36)=1.0D0         ! 4p6
      AvP(19,36)=1.0D0         ! 4p6
!-----------------------------------------
!     Rb Z=37   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 5s1 5p0]
!
      AvP(1 ,37)=1.0D0         ! 1s2
      AvP(2 ,37)=1.0D0         ! 2s2
      AvP(3 ,37)=1.0D0         ! 2p6
      AvP(4 ,37)=1.0D0         ! 2p6
      AvP(5 ,37)=1.0D0         ! 2p6
      AvP(6 ,37)=1.0D0         ! 3s2
      AvP(7 ,37)=1.0D0         ! 3p6
      AvP(8 ,37)=1.0D0         ! 3p6
      AvP(9 ,37)=1.0D0         ! 3p6
      AvP(10,37)=5.0D0/6.0D0   ! 3d10
      AvP(11,37)=5.0D0/6.0D0   ! 3d10
      AvP(12,37)=5.0D0/6.0D0   ! 3d10
      AvP(13,37)=5.0D0/6.0D0   ! 3d10
      AvP(14,37)=5.0D0/6.0D0   ! 3d10
      AvP(15,37)=5.0D0/6.0D0   ! 3d10
      AvP(16,37)=1.0D0         ! 4s2
      AvP(17,37)=1.0D0         ! 4p6
      AvP(18,37)=1.0D0         ! 4p6
      AvP(19,37)=1.0D0         ! 4p6
      AvP(20,37)=1.0D0/2.0D0   ! 5s1
      AvP(21,37)=0.0D0         ! 5p0
      AvP(22,37)=0.0D0         ! 5p0
      AvP(23,37)=0.0D0         ! 5p0
!-----------------------------------------
!     Sr Z=38   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 5s2 5p0]
!
      AvP(1 ,38)=1.0D0         ! 1s2
      AvP(2 ,38)=1.0D0         ! 2s2
      AvP(3 ,38)=1.0D0         ! 2p6
      AvP(4 ,38)=1.0D0         ! 2p6
      AvP(5 ,38)=1.0D0         ! 2p6
      AvP(6 ,38)=1.0D0         ! 3s2
      AvP(7 ,38)=1.0D0         ! 3p6
      AvP(8 ,38)=1.0D0         ! 3p6
      AvP(9 ,38)=1.0D0         ! 3p6
      AvP(10,38)=5.0D0/6.0D0   ! 3d10
      AvP(11,38)=5.0D0/6.0D0   ! 3d10
      AvP(12,38)=5.0D0/6.0D0   ! 3d10
      AvP(13,38)=5.0D0/6.0D0   ! 3d10
      AvP(14,38)=5.0D0/6.0D0   ! 3d10
      AvP(15,38)=5.0D0/6.0D0   ! 3d10
      AvP(16,38)=1.0D0         ! 4s2
      AvP(17,38)=1.0D0         ! 4p6
      AvP(18,38)=1.0D0         ! 4p6
      AvP(19,38)=1.0D0         ! 4p6
      AvP(20,38)=1.0D0         ! 5s2
      AvP(21,38)=0.0D0         ! 5p0
      AvP(22,38)=0.0D0         ! 5p0
      AvP(23,38)=0.0D0         ! 5p0
!-----------------------------------------
!     Y Z=39   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d1 5s1 5p0]
!
      AvP(1 ,39)=1.0D0         ! 1s2
      AvP(2 ,39)=1.0D0         ! 2s2
      AvP(3 ,39)=1.0D0         ! 2p6
      AvP(4 ,39)=1.0D0         ! 2p6
      AvP(5 ,39)=1.0D0         ! 2p6
      AvP(6 ,39)=1.0D0         ! 3s2
      AvP(7 ,39)=1.0D0         ! 3p6
      AvP(8 ,39)=1.0D0         ! 3p6
      AvP(9 ,39)=1.0D0         ! 3p6
      AvP(10,39)=5.0D0/6.0D0   ! 3d10
      AvP(11,39)=5.0D0/6.0D0   ! 3d10
      AvP(12,39)=5.0D0/6.0D0   ! 3d10
      AvP(13,39)=5.0D0/6.0D0   ! 3d10
      AvP(14,39)=5.0D0/6.0D0   ! 3d10
      AvP(15,39)=5.0D0/6.0D0   ! 3d10
      AvP(16,39)=1.0D0         ! 4s2
      AvP(17,39)=1.0D0         ! 4p6
      AvP(18,39)=1.0D0         ! 4p6
      AvP(19,39)=1.0D0         ! 4p6
      AvP(20,39)=1.0D0/12.0D0  ! 4d1
      AvP(21,39)=1.0D0/12.0D0  ! 4d1
      AvP(22,39)=1.0D0/12.0D0  ! 4d1
      AvP(23,39)=1.0D0/12.0D0  ! 4d1
      AvP(24,39)=1.0D0/12.0D0  ! 4d1
      AvP(25,39)=1.0D0/12.0D0  ! 4d1
      AvP(26,39)=1.0D0/2.0D0   ! 5s1
      AvP(27,39)=0.0D0         ! 5p0
      AvP(28,39)=0.0D0         ! 5p0
      AvP(29,39)=0.0D0         ! 5p0
!-----------------------------------------
!     Zr Z=40   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d2 5s2 5p0]
!
      AvP(1 ,40)=1.0D0         ! 1s2
      AvP(2 ,40)=1.0D0         ! 2s2
      AvP(3 ,40)=1.0D0         ! 2p6
      AvP(4 ,40)=1.0D0         ! 2p6
      AvP(5 ,40)=1.0D0         ! 2p6
      AvP(6 ,40)=1.0D0         ! 3s2
      AvP(7 ,40)=1.0D0         ! 3p6
      AvP(8 ,40)=1.0D0         ! 3p6
      AvP(9 ,40)=1.0D0         ! 3p6
      AvP(10,40)=5.0D0/6.0D0   ! 3d10
      AvP(11,40)=5.0D0/6.0D0   ! 3d10
      AvP(12,40)=5.0D0/6.0D0   ! 3d10
      AvP(13,40)=5.0D0/6.0D0   ! 3d10
      AvP(14,40)=5.0D0/6.0D0   ! 3d10
      AvP(15,40)=5.0D0/6.0D0   ! 3d10
      AvP(16,40)=1.0D0         ! 4s2
      AvP(17,40)=1.0D0         ! 4p6
      AvP(18,40)=1.0D0         ! 4p6
      AvP(19,40)=1.0D0         ! 4p6
      AvP(20,40)=1.0D0/6.0D0   ! 4d2
      AvP(21,40)=1.0D0/6.0D0   ! 4d2
      AvP(22,40)=1.0D0/6.0D0   ! 4d2
      AvP(23,40)=1.0D0/6.0D0   ! 4d2
      AvP(24,40)=1.0D0/6.0D0   ! 4d2
      AvP(25,40)=1.0D0/6.0D0   ! 4d2
      AvP(26,40)=1.0D0         ! 5s1
      AvP(27,40)=0.0D0         ! 5p0
      AvP(28,40)=0.0D0         ! 5p0
      AvP(29,40)=0.0D0         ! 5p0
!-----------------------------------------
!     Nb Z=41   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d4 5s1 5p0]
!
      AvP(1 ,41)=1.0D0         ! 1s2
      AvP(2 ,41)=1.0D0         ! 2s2
      AvP(3 ,41)=1.0D0         ! 2p6
      AvP(4 ,41)=1.0D0         ! 2p6
      AvP(5 ,41)=1.0D0         ! 2p6
      AvP(6 ,41)=1.0D0         ! 3s2
      AvP(7 ,41)=1.0D0         ! 3p6
      AvP(8 ,41)=1.0D0         ! 3p6
      AvP(9 ,41)=1.0D0         ! 3p6
      AvP(10,41)=5.0D0/6.0D0   ! 3d10
      AvP(11,41)=5.0D0/6.0D0   ! 3d10
      AvP(12,41)=5.0D0/6.0D0   ! 3d10
      AvP(13,41)=5.0D0/6.0D0   ! 3d10
      AvP(14,41)=5.0D0/6.0D0   ! 3d10
      AvP(15,41)=5.0D0/6.0D0   ! 3d10
      AvP(16,41)=1.0D0         ! 4s2
      AvP(17,41)=1.0D0         ! 4p6
      AvP(18,41)=1.0D0         ! 4p6
      AvP(19,41)=1.0D0         ! 4p6
      AvP(20,41)=2.0D0/6.0D0   ! 4d4
      AvP(21,41)=2.0D0/6.0D0   ! 4d4
      AvP(22,41)=2.0D0/6.0D0   ! 4d4
      AvP(23,41)=2.0D0/6.0D0   ! 4d4
      AvP(24,41)=2.0D0/6.0D0   ! 4d4
      AvP(25,41)=2.0D0/6.0D0   ! 4d4
      AvP(26,41)=1.0D0/2.0D0   ! 5s1
      AvP(27,41)=0.0D0         ! 5p0
      AvP(28,41)=0.0D0         ! 5p0
      AvP(29,41)=0.0D0         ! 5p0
!-----------------------------------------
!     Mo Z=42   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d5 5s1 5p0]
!
      AvP(1 ,42)=1.0D0         ! 1s2
      AvP(2 ,42)=1.0D0         ! 2s2
      AvP(3 ,42)=1.0D0         ! 2p6
      AvP(4 ,42)=1.0D0         ! 2p6
      AvP(5 ,42)=1.0D0         ! 2p6
      AvP(6 ,42)=1.0D0         ! 3s2
      AvP(7 ,42)=1.0D0         ! 3p6
      AvP(8 ,42)=1.0D0         ! 3p6
      AvP(9 ,42)=1.0D0         ! 3p6
      AvP(10,42)=5.0D0/6.0D0   ! 3d10
      AvP(11,42)=5.0D0/6.0D0   ! 3d10
      AvP(12,42)=5.0D0/6.0D0   ! 3d10
      AvP(13,42)=5.0D0/6.0D0   ! 3d10
      AvP(14,42)=5.0D0/6.0D0   ! 3d10
      AvP(15,42)=5.0D0/6.0D0   ! 3d10
      AvP(16,42)=1.0D0         ! 4s2
      AvP(17,42)=1.0D0         ! 4p6
      AvP(18,42)=1.0D0         ! 4p6
      AvP(19,42)=1.0D0         ! 4p6
      AvP(20,42)=5.0D0/12.0D0  ! 4d5
      AvP(21,42)=5.0D0/12.0D0  ! 4d5
      AvP(22,42)=5.0D0/12.0D0  ! 4d5
      AvP(23,42)=5.0D0/12.0D0  ! 4d5
      AvP(24,42)=5.0D0/12.0D0  ! 4d5
      AvP(25,42)=5.0D0/12.0D0  ! 4d5
      AvP(26,42)=1.0D0/2.0D0   ! 5s1
      AvP(27,42)=0.0D0         ! 5p0
      AvP(28,42)=0.0D0         ! 5p0
      AvP(29,42)=0.0D0         ! 5p0
!-----------------------------------------
!     Tc Z=43   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d5 5s2 5p0]
!
      AvP(1 ,43)=1.0D0         ! 1s2
      AvP(2 ,43)=1.0D0         ! 2s2
      AvP(3 ,43)=1.0D0         ! 2p6
      AvP(4 ,43)=1.0D0         ! 2p6
      AvP(5 ,43)=1.0D0         ! 2p6
      AvP(6 ,43)=1.0D0         ! 3s2
      AvP(7 ,43)=1.0D0         ! 3p6
      AvP(8 ,43)=1.0D0         ! 3p6
      AvP(9 ,43)=1.0D0         ! 3p6
      AvP(10,43)=5.0D0/6.0D0   ! 3d10
      AvP(11,43)=5.0D0/6.0D0   ! 3d10
      AvP(12,43)=5.0D0/6.0D0   ! 3d10
      AvP(13,43)=5.0D0/6.0D0   ! 3d10
      AvP(14,43)=5.0D0/6.0D0   ! 3d10
      AvP(15,43)=5.0D0/6.0D0   ! 3d10
      AvP(16,43)=1.0D0         ! 4s2
      AvP(17,43)=1.0D0         ! 4p6
      AvP(18,43)=1.0D0         ! 4p6
      AvP(19,43)=1.0D0         ! 4p6
      AvP(20,43)=5.0D0/12.0D0  ! 4d5
      AvP(21,43)=5.0D0/12.0D0  ! 4d5
      AvP(22,43)=5.0D0/12.0D0  ! 4d5
      AvP(23,43)=5.0D0/12.0D0  ! 4d5
      AvP(24,43)=5.0D0/12.0D0  ! 4d5
      AvP(25,43)=5.0D0/12.0D0  ! 4d5
      AvP(26,43)=1.0D0         ! 5s2
      AvP(27,43)=0.0D0         ! 5p0
      AvP(28,43)=0.0D0         ! 5p0
      AvP(29,43)=0.0D0         ! 5p0
!-----------------------------------------
!     Ru Z=44   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d7 5s1 5p0]
!
      AvP(1 ,44)=1.0D0         ! 1s2
      AvP(2 ,44)=1.0D0         ! 2s2
      AvP(3 ,44)=1.0D0         ! 2p6
      AvP(4 ,44)=1.0D0         ! 2p6
      AvP(5 ,44)=1.0D0         ! 2p6
      AvP(6 ,44)=1.0D0         ! 3s2
      AvP(7 ,44)=1.0D0         ! 3p6
      AvP(8 ,44)=1.0D0         ! 3p6
      AvP(9 ,44)=1.0D0         ! 3p6
      AvP(10,44)=5.0D0/6.0D0   ! 3d10
      AvP(11,44)=5.0D0/6.0D0   ! 3d10
      AvP(12,44)=5.0D0/6.0D0   ! 3d10
      AvP(13,44)=5.0D0/6.0D0   ! 3d10
      AvP(14,44)=5.0D0/6.0D0   ! 3d10
      AvP(15,44)=5.0D0/6.0D0   ! 3d10
      AvP(16,44)=1.0D0         ! 4s2
      AvP(17,44)=1.0D0         ! 4p6
      AvP(18,44)=1.0D0         ! 4p6
      AvP(19,44)=1.0D0         ! 4p6
      AvP(20,44)=7.0D0/12.0D0  ! 4d7
      AvP(21,44)=7.0D0/12.0D0  ! 4d7
      AvP(22,44)=7.0D0/12.0D0  ! 4d7
      AvP(23,44)=7.0D0/12.0D0  ! 4d7
      AvP(24,44)=7.0D0/12.0D0  ! 4d7
      AvP(25,44)=7.0D0/12.0D0  ! 4d7
      AvP(26,44)=1.0D0/2.0D0   ! 5s1
      AvP(27,44)=0.0D0         ! 5p0
      AvP(28,44)=0.0D0         ! 5p0
      AvP(29,44)=0.0D0         ! 5p0
!-----------------------------------------
!     Rh Z=45   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d8 5s1 5p0]
!
      AvP(1 ,45)=1.0D0         ! 1s2
      AvP(2 ,45)=1.0D0         ! 2s2
      AvP(3 ,45)=1.0D0         ! 2p6
      AvP(4 ,45)=1.0D0         ! 2p6
      AvP(5 ,45)=1.0D0         ! 2p6
      AvP(6 ,45)=1.0D0         ! 3s2
      AvP(7 ,45)=1.0D0         ! 3p6
      AvP(8 ,45)=1.0D0         ! 3p6
      AvP(9 ,45)=1.0D0         ! 3p6
      AvP(10,45)=5.0D0/6.0D0   ! 3d10
      AvP(11,45)=5.0D0/6.0D0   ! 3d10
      AvP(12,45)=5.0D0/6.0D0   ! 3d10
      AvP(13,45)=5.0D0/6.0D0   ! 3d10
      AvP(14,45)=5.0D0/6.0D0   ! 3d10
      AvP(15,45)=5.0D0/6.0D0   ! 3d10
      AvP(16,45)=1.0D0         ! 4s2
      AvP(17,45)=1.0D0         ! 4p6
      AvP(18,45)=1.0D0         ! 4p6
      AvP(19,45)=1.0D0         ! 4p6
      AvP(20,45)=8.0D0/12.0D0  ! 4d8
      AvP(21,45)=8.0D0/12.0D0  ! 4d8
      AvP(22,45)=8.0D0/12.0D0  ! 4d8
      AvP(23,45)=8.0D0/12.0D0  ! 4d8
      AvP(24,45)=8.0D0/12.0D0  ! 4d8
      AvP(25,45)=8.0D0/12.0D0  ! 4d8
      AvP(26,45)=1.0D0/2.0D0   ! 5s1
      AvP(27,45)=0.0D0         ! 5p0
      AvP(28,45)=0.0D0         ! 5p0
      AvP(29,45)=0.0D0         ! 5p0
!-----------------------------------------
!     Pd Z=46   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s0 5p0]
!
      AvP(1 ,46)=1.0D0         ! 1s2
      AvP(2 ,46)=1.0D0         ! 2s2
      AvP(3 ,46)=1.0D0         ! 2p6
      AvP(4 ,46)=1.0D0         ! 2p6
      AvP(5 ,46)=1.0D0         ! 2p6
      AvP(6 ,46)=1.0D0         ! 3s2
      AvP(7 ,46)=1.0D0         ! 3p6
      AvP(8 ,46)=1.0D0         ! 3p6
      AvP(9 ,46)=1.0D0         ! 3p6
      AvP(10,46)=5.0D0/6.0D0   ! 3d10
      AvP(11,46)=5.0D0/6.0D0   ! 3d10
      AvP(12,46)=5.0D0/6.0D0   ! 3d10
      AvP(13,46)=5.0D0/6.0D0   ! 3d10
      AvP(14,46)=5.0D0/6.0D0   ! 3d10
      AvP(15,46)=5.0D0/6.0D0   ! 3d10
      AvP(16,46)=1.0D0         ! 4s2
      AvP(17,46)=1.0D0         ! 4p6
      AvP(18,46)=1.0D0         ! 4p6
      AvP(19,46)=1.0D0         ! 4p6
      AvP(20,46)=10.0D0/12.0D0 ! 4d10
      AvP(21,46)=10.0D0/12.0D0 ! 4d10
      AvP(22,46)=10.0D0/12.0D0 ! 4d10
      AvP(23,46)=10.0D0/12.0D0 ! 4d10
      AvP(24,46)=10.0D0/12.0D0 ! 4d10
      AvP(25,46)=10.0D0/12.0D0 ! 4d10
      AvP(26,46)=0.0D0         ! 5s0
      AvP(27,46)=0.0D0         ! 5p0
      AvP(28,46)=0.0D0         ! 5p0
      AvP(29,46)=0.0D0         ! 5p0
!-----------------------------------------
!     Ag Z=47   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s1 5p0]
!
      AvP(1 ,47)=1.0D0         ! 1s2
      AvP(2 ,47)=1.0D0         ! 2s2
      AvP(3 ,47)=1.0D0         ! 2p6
      AvP(4 ,47)=1.0D0         ! 2p6
      AvP(5 ,47)=1.0D0         ! 2p6
      AvP(6 ,47)=1.0D0         ! 3s2
      AvP(7 ,47)=1.0D0         ! 3p6
      AvP(8 ,47)=1.0D0         ! 3p6
      AvP(9 ,47)=1.0D0         ! 3p6
      AvP(10,47)=5.0D0/6.0D0   ! 3d10
      AvP(11,47)=5.0D0/6.0D0   ! 3d10
      AvP(12,47)=5.0D0/6.0D0   ! 3d10
      AvP(13,47)=5.0D0/6.0D0   ! 3d10
      AvP(14,47)=5.0D0/6.0D0   ! 3d10
      AvP(15,47)=5.0D0/6.0D0   ! 3d10
      AvP(16,47)=1.0D0         ! 4s2
      AvP(17,47)=1.0D0         ! 4p6
      AvP(18,47)=1.0D0         ! 4p6
      AvP(19,47)=1.0D0         ! 4p6
      AvP(20,47)=10.0D0/12.0D0 ! 4d10
      AvP(21,47)=10.0D0/12.0D0 ! 4d10
      AvP(22,47)=10.0D0/12.0D0 ! 4d10
      AvP(23,47)=10.0D0/12.0D0 ! 4d10
      AvP(24,47)=10.0D0/12.0D0 ! 4d10
      AvP(25,47)=10.0D0/12.0D0 ! 4d10
      AvP(26,47)=1.0D0/2.0D0   ! 5s1
      AvP(27,47)=0.0D0         ! 5p0
      AvP(28,47)=0.0D0         ! 5p0
      AvP(29,47)=0.0D0         ! 5p0
!-----------------------------------------
!     Cd Z=48   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s2 5p0]
!
      AvP(1 ,48)=1.0D0         ! 1s2
      AvP(2 ,48)=1.0D0         ! 2s2
      AvP(3 ,48)=1.0D0         ! 2p6
      AvP(4 ,48)=1.0D0         ! 2p6
      AvP(5 ,48)=1.0D0         ! 2p6
      AvP(6 ,48)=1.0D0         ! 3s2
      AvP(7 ,48)=1.0D0         ! 3p6
      AvP(8 ,48)=1.0D0         ! 3p6
      AvP(9 ,48)=1.0D0         ! 3p6
      AvP(10,48)=5.0D0/6.0D0   ! 3d10
      AvP(11,48)=5.0D0/6.0D0   ! 3d10
      AvP(12,48)=5.0D0/6.0D0   ! 3d10
      AvP(13,48)=5.0D0/6.0D0   ! 3d10
      AvP(14,48)=5.0D0/6.0D0   ! 3d10
      AvP(15,48)=5.0D0/6.0D0   ! 3d10
      AvP(16,48)=1.0D0         ! 4s2
      AvP(17,48)=1.0D0         ! 4p6
      AvP(18,48)=1.0D0         ! 4p6
      AvP(19,48)=1.0D0         ! 4p6
      AvP(20,48)=10.0D0/12.0D0 ! 4d10
      AvP(21,48)=10.0D0/12.0D0 ! 4d10
      AvP(22,48)=10.0D0/12.0D0 ! 4d10
      AvP(23,48)=10.0D0/12.0D0 ! 4d10
      AvP(24,48)=10.0D0/12.0D0 ! 4d10
      AvP(25,48)=10.0D0/12.0D0 ! 4d10
      AvP(26,48)=1.0D0         ! 5s2
      AvP(27,48)=0.0D0         ! 5p0
      AvP(28,48)=0.0D0         ! 5p0
      AvP(29,48)=0.0D0         ! 5p0
!-----------------------------------------
!     In Z=49   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s2 5p1]
!
      AvP(1 ,49)=1.0D0         ! 1s2
      AvP(2 ,49)=1.0D0         ! 2s2
      AvP(3 ,49)=1.0D0         ! 2p6
      AvP(4 ,49)=1.0D0         ! 2p6
      AvP(5 ,49)=1.0D0         ! 2p6
      AvP(6 ,49)=1.0D0         ! 3s2
      AvP(7 ,49)=1.0D0         ! 3p6
      AvP(8 ,49)=1.0D0         ! 3p6
      AvP(9 ,49)=1.0D0         ! 3p6
      AvP(10,49)=5.0D0/6.0D0   ! 3d10
      AvP(11,49)=5.0D0/6.0D0   ! 3d10
      AvP(12,49)=5.0D0/6.0D0   ! 3d10
      AvP(13,49)=5.0D0/6.0D0   ! 3d10
      AvP(14,49)=5.0D0/6.0D0   ! 3d10
      AvP(15,49)=5.0D0/6.0D0   ! 3d10
      AvP(16,49)=1.0D0         ! 4s2
      AvP(17,49)=1.0D0         ! 4p6
      AvP(18,49)=1.0D0         ! 4p6
      AvP(19,49)=1.0D0         ! 4p6
      AvP(20,49)=10.0D0/12.0D0 ! 4d10
      AvP(21,49)=10.0D0/12.0D0 ! 4d10
      AvP(22,49)=10.0D0/12.0D0 ! 4d10
      AvP(23,49)=10.0D0/12.0D0 ! 4d10
      AvP(24,49)=10.0D0/12.0D0 ! 4d10
      AvP(25,49)=10.0D0/12.0D0 ! 4d10
      AvP(26,49)=1.0D0         ! 5s2
      AvP(27,49)=1.0D0/6.0D0   ! 5p1
      AvP(28,49)=1.0D0/6.0D0   ! 5p1
      AvP(29,49)=1.0D0/6.0D0   ! 5p1
!-----------------------------------------
!     Sn Z=50   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s2 5p2]
!
      AvP(1 ,50)=1.0D0         ! 1s2
      AvP(2 ,50)=1.0D0         ! 2s2
      AvP(3 ,50)=1.0D0         ! 2p6
      AvP(4 ,50)=1.0D0         ! 2p6
      AvP(5 ,50)=1.0D0         ! 2p6
      AvP(6 ,50)=1.0D0         ! 3s2
      AvP(7 ,50)=1.0D0         ! 3p6
      AvP(8 ,50)=1.0D0         ! 3p6
      AvP(9 ,50)=1.0D0         ! 3p6
      AvP(10,50)=5.0D0/6.0D0   ! 3d10
      AvP(11,50)=5.0D0/6.0D0   ! 3d10
      AvP(12,50)=5.0D0/6.0D0   ! 3d10
      AvP(13,50)=5.0D0/6.0D0   ! 3d10
      AvP(14,50)=5.0D0/6.0D0   ! 3d10
      AvP(15,50)=5.0D0/6.0D0   ! 3d10
      AvP(16,50)=1.0D0         ! 4s2
      AvP(17,50)=1.0D0         ! 4p6
      AvP(18,50)=1.0D0         ! 4p6
      AvP(19,50)=1.0D0         ! 4p6
      AvP(20,50)=10.0D0/12.0D0 ! 4d10
      AvP(21,50)=10.0D0/12.0D0 ! 4d10
      AvP(22,50)=10.0D0/12.0D0 ! 4d10
      AvP(23,50)=10.0D0/12.0D0 ! 4d10
      AvP(24,50)=10.0D0/12.0D0 ! 4d10
      AvP(25,50)=10.0D0/12.0D0 ! 4d10
      AvP(26,50)=1.0D0         ! 5s2
      AvP(27,50)=2.0D0/6.0D0   ! 5p2
      AvP(28,50)=2.0D0/6.0D0   ! 5p2
      AvP(29,50)=2.0D0/6.0D0   ! 5p2
!-----------------------------------------
!     Sb Z=51   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s2 5p3]
!
      AvP(1 ,51)=1.0D0         ! 1s2
      AvP(2 ,51)=1.0D0         ! 2s2
      AvP(3 ,51)=1.0D0         ! 2p6
      AvP(4 ,51)=1.0D0         ! 2p6
      AvP(5 ,51)=1.0D0         ! 2p6
      AvP(6 ,51)=1.0D0         ! 3s2
      AvP(7 ,51)=1.0D0         ! 3p6
      AvP(8 ,51)=1.0D0         ! 3p6
      AvP(9 ,51)=1.0D0         ! 3p6
      AvP(10,51)=5.0D0/6.0D0   ! 3d10
      AvP(11,51)=5.0D0/6.0D0   ! 3d10
      AvP(12,51)=5.0D0/6.0D0   ! 3d10
      AvP(13,51)=5.0D0/6.0D0   ! 3d10
      AvP(14,51)=5.0D0/6.0D0   ! 3d10
      AvP(15,51)=5.0D0/6.0D0   ! 3d10
      AvP(16,51)=1.0D0         ! 4s2
      AvP(17,51)=1.0D0         ! 4p6
      AvP(18,51)=1.0D0         ! 4p6
      AvP(19,51)=1.0D0         ! 4p6
      AvP(20,51)=10.0D0/12.0D0 ! 4d10
      AvP(21,51)=10.0D0/12.0D0 ! 4d10
      AvP(22,51)=10.0D0/12.0D0 ! 4d10
      AvP(23,51)=10.0D0/12.0D0 ! 4d10
      AvP(24,51)=10.0D0/12.0D0 ! 4d10
      AvP(25,51)=10.0D0/12.0D0 ! 4d10
      AvP(26,51)=1.0D0         ! 5s2
      AvP(27,51)=3.0D0/6.0D0   ! 5p3
      AvP(28,51)=3.0D0/6.0D0   ! 5p3
      AvP(29,51)=3.0D0/6.0D0   ! 5p3
!-----------------------------------------
!     Te Z=52   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s2 5p4]
!
      AvP(1 ,52)=1.0D0         ! 1s2
      AvP(2 ,52)=1.0D0         ! 2s2
      AvP(3 ,52)=1.0D0         ! 2p6
      AvP(4 ,52)=1.0D0         ! 2p6
      AvP(5 ,52)=1.0D0         ! 2p6
      AvP(6 ,52)=1.0D0         ! 3s2
      AvP(7 ,52)=1.0D0         ! 3p6
      AvP(8 ,52)=1.0D0         ! 3p6
      AvP(9 ,52)=1.0D0         ! 3p6
      AvP(10,52)=5.0D0/6.0D0   ! 3d10
      AvP(11,52)=5.0D0/6.0D0   ! 3d10
      AvP(12,52)=5.0D0/6.0D0   ! 3d10
      AvP(13,52)=5.0D0/6.0D0   ! 3d10
      AvP(14,52)=5.0D0/6.0D0   ! 3d10
      AvP(15,52)=5.0D0/6.0D0   ! 3d10
      AvP(16,52)=1.0D0         ! 4s2
      AvP(17,52)=1.0D0         ! 4p6
      AvP(18,52)=1.0D0         ! 4p6
      AvP(19,52)=1.0D0         ! 4p6
      AvP(20,52)=10.0D0/12.0D0 ! 4d10
      AvP(21,52)=10.0D0/12.0D0 ! 4d10
      AvP(22,52)=10.0D0/12.0D0 ! 4d10
      AvP(23,52)=10.0D0/12.0D0 ! 4d10
      AvP(24,52)=10.0D0/12.0D0 ! 4d10
      AvP(25,52)=10.0D0/12.0D0 ! 4d10
      AvP(26,52)=1.0D0         ! 5s2
      AvP(27,52)=4.0D0/6.0D0   ! 5p4
      AvP(28,52)=4.0D0/6.0D0   ! 5p4
      AvP(29,52)=4.0D0/6.0D0   ! 5p4
!-----------------------------------------
!     I Z=53   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s2 5p5]
!
      AvP(1 ,53)=1.0D0         ! 1s2
      AvP(2 ,53)=1.0D0         ! 2s2
      AvP(3 ,53)=1.0D0         ! 2p6
      AvP(4 ,53)=1.0D0         ! 2p6
      AvP(5 ,53)=1.0D0         ! 2p6
      AvP(6 ,53)=1.0D0         ! 3s2
      AvP(7 ,53)=1.0D0         ! 3p6
      AvP(8 ,53)=1.0D0         ! 3p6
      AvP(9 ,53)=1.0D0         ! 3p6
      AvP(10,53)=5.0D0/6.0D0   ! 3d10
      AvP(11,53)=5.0D0/6.0D0   ! 3d10
      AvP(12,53)=5.0D0/6.0D0   ! 3d10
      AvP(13,53)=5.0D0/6.0D0   ! 3d10
      AvP(14,53)=5.0D0/6.0D0   ! 3d10
      AvP(15,53)=5.0D0/6.0D0   ! 3d10
      AvP(16,53)=1.0D0         ! 4s2
      AvP(17,53)=1.0D0         ! 4p6
      AvP(18,53)=1.0D0         ! 4p6
      AvP(19,53)=1.0D0         ! 4p6
      AvP(20,53)=10.0D0/12.0D0 ! 4d10
      AvP(21,53)=10.0D0/12.0D0 ! 4d10
      AvP(22,53)=10.0D0/12.0D0 ! 4d10
      AvP(23,53)=10.0D0/12.0D0 ! 4d10
      AvP(24,53)=10.0D0/12.0D0 ! 4d10
      AvP(25,53)=10.0D0/12.0D0 ! 4d10
      AvP(26,53)=1.0D0         ! 5s2
      AvP(27,53)=5.0D0/6.0D0   ! 5p5
      AvP(28,53)=5.0D0/6.0D0   ! 5p5
      AvP(29,53)=5.0D0/6.0D0   ! 5p5
!-----------------------------------------
!     Xe Z=54   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s2 5p6]
!
      AvP(1 ,54)=1.0D0         ! 1s2
      AvP(2 ,54)=1.0D0         ! 2s2
      AvP(3 ,54)=1.0D0         ! 2p6
      AvP(4 ,54)=1.0D0         ! 2p6
      AvP(5 ,54)=1.0D0         ! 2p6
      AvP(6 ,54)=1.0D0         ! 3s2
      AvP(7 ,54)=1.0D0         ! 3p6
      AvP(8 ,54)=1.0D0         ! 3p6
      AvP(9 ,54)=1.0D0         ! 3p6
      AvP(10,54)=5.0D0/6.0D0   ! 3d10
      AvP(11,54)=5.0D0/6.0D0   ! 3d10
      AvP(12,54)=5.0D0/6.0D0   ! 3d10
      AvP(13,54)=5.0D0/6.0D0   ! 3d10
      AvP(14,54)=5.0D0/6.0D0   ! 3d10
      AvP(15,54)=5.0D0/6.0D0   ! 3d10
      AvP(16,54)=1.0D0         ! 4s2
      AvP(17,54)=1.0D0         ! 4p6
      AvP(18,54)=1.0D0         ! 4p6
      AvP(19,54)=1.0D0         ! 4p6
      AvP(20,54)=10.0D0/12.0D0 ! 4d10
      AvP(21,54)=10.0D0/12.0D0 ! 4d10
      AvP(22,54)=10.0D0/12.0D0 ! 4d10
      AvP(23,54)=10.0D0/12.0D0 ! 4d10
      AvP(24,54)=10.0D0/12.0D0 ! 4d10
      AvP(25,54)=10.0D0/12.0D0 ! 4d10
      AvP(26,54)=1.0D0         ! 5s2
      AvP(27,54)=1.0D0         ! 5p6
      AvP(28,54)=1.0D0         ! 5p6
      AvP(29,54)=1.0D0         ! 5p6
!-----------------------------------------
!     Cs Z=55   [1s2 2s2 2p6 3s2 3p6 3d10 4s2 4p6 4d10 5s2 5p6 6s1]
!
      AvP(1 ,55)=1.0D0         ! 1s2
      AvP(2 ,55)=1.0D0         ! 2s2
      AvP(3 ,55)=1.0D0         ! 2p6
      AvP(4 ,55)=1.0D0         ! 2p6
      AvP(5 ,55)=1.0D0         ! 2p6
      AvP(6 ,55)=1.0D0         ! 3s2
      AvP(7 ,55)=1.0D0         ! 3p6
      AvP(8 ,55)=1.0D0         ! 3p6
      AvP(9 ,55)=1.0D0         ! 3p6
      AvP(10,55)=5.0D0/6.0D0   ! 3d10
      AvP(11,55)=5.0D0/6.0D0   ! 3d10
      AvP(12,55)=5.0D0/6.0D0   ! 3d10
      AvP(13,55)=5.0D0/6.0D0   ! 3d10
      AvP(14,55)=5.0D0/6.0D0   ! 3d10
      AvP(15,55)=5.0D0/6.0D0   ! 3d10
      AvP(16,55)=1.0D0         ! 4s2
      AvP(17,55)=1.0D0         ! 4p6
      AvP(18,55)=1.0D0         ! 4p6
      AvP(19,55)=1.0D0         ! 4p6
      AvP(20,55)=10.0D0/12.0D0 ! 4d10
      AvP(21,55)=10.0D0/12.0D0 ! 4d10
      AvP(22,55)=10.0D0/12.0D0 ! 4d10
      AvP(23,55)=10.0D0/12.0D0 ! 4d10
      AvP(24,55)=10.0D0/12.0D0 ! 4d10
      AvP(25,55)=10.0D0/12.0D0 ! 4d10
      AvP(26,55)=1.0D0         ! 5s2
      AvP(27,55)=1.0D0         ! 5p6
      AvP(28,55)=1.0D0         ! 5p6
      AvP(29,55)=1.0D0         ! 5p6
      AvP(30,55)=1.0D0/2.0D0   ! 6s1
      AvP(31,55)=0.0D0         ! 6p0
      AvP(32,55)=0.0D0         ! 6p0
      AvP(33,55)=0.0D0         ! 6p0
!-----------------------------------------
!
!
      PBlok=Zero
      DO I=1,N
         PBlok(I,I)=AvP(I,K)
      ENDDO

   END SUBROUTINE FillPBlok
END MODULE
