;;;;"dielectric.scm" FreeSnell optics validation suite	-*-scheme-*-
;;; Copyright (C) 2003, 2004, 2005 Aubrey Jaffer

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation, either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

;;; http://swiss.csail.mit.edu/~jaffer/FreeSnell/dielectric.html

(require 'FreeSnell)

;;; Reducing the size of plots to closer match the web plots.
;;;(define graph:dimensions '(512 256))
;;;(define graph:font-size 13)
;;; Output png files for validation webpage.
;;;(define *output-format* 'png)

;;;; Handy for comparing FreeSnell graphs to optics on the web
(define optics:tmp "/tmp/optics.url")
(define (browse-optics-url url)
  (define current-url (and (file-exists? optics:tmp)
			   (call-with-input-file optics:tmp read)))
  (cond ((equal? url current-url))
	(else (call-with-output-file optics:tmp
		(lambda (oport) (write url oport)))
	      (browse-url url))))
(define (browse-optics-url url) url)

;; works
(define (wide-bp)
  (define ZnS 2.2)			;znscub +0.01i
  (define Ge 4.2)			;ge
  (browse-optics-url "http://www.sspectra.com/designs/irbp.html")
  (plot-response
   (title "Wide Infrared Bandpass Filter" "wide-bp")
   (output-format 'png 512 256)
   (font 13)
   (samples 300)
   (incident 0 'T)
   ;;(IR Ge 'real 'imag) (IR ZnS 'real 'imag)
   (wavelengths 2e-6 7e-6)
   (optical-stack
    (substrate 1)
    (layer	ZnS   528.64e-9)
    (layer	Ge    178.96e-9)
    (layer	ZnS   250.12e-9)
    (layer	Ge    123.17e-9)
    (layer	ZnS   294.15e-9)
    (layer	Ge    156.86e-9)
    (layer	ZnS   265.60e-9)
    (layer	Ge    134.34e-9)
    (layer	ZnS   266.04e-9)
    (layer	Ge    147.63e-9)
    (layer	ZnS   289.60e-9)
    (layer	Ge    133.04e-9)
    (layer	ZnS   256.22e-9)
    (layer	Ge    165.16e-9)
    (layer	ZnS   307.19e-9)
    (layer	Ge    125.25e-9)
    (layer	ZnS   254.28e-9)
    (layer	Ge    150.14e-9)
    (layer	ZnS   168.55e-9)
    (layer	Ge     68.54e-9)
    (layer	ZnS   232.65e-9)
    (layer	Ge    125.48e-9)
    (layer	ZnS   238.01e-9)
    (layer	Ge    138.25e-9)
    (layer	ZnS   268.21e-9)
    (layer	Ge     98.28e-9)
    (layer	ZnS   133.58e-9)
    (layer	Ge    125.31e-9)
    (layer	ZnS   224.72e-9)
    (layer	Ge     40.79e-9)
    (layer	ZnS   564.95e-9)
    (layer	Ge    398.52e-9)
    (layer	ZnS   710.47e-9)
    (layer	Ge    360.01e-9)
    (layer	ZnS   724.86e-9)
    (layer	Ge    353.08e-9)
    (layer	ZnS   718.52e-9)
    (layer	Ge    358.23e-9)
    (layer	ZnS   709.26e-9)
    (layer	Ge    370.42e-9)
    (layer	ZnS   705.03e-9)
    (layer	Ge    382.28e-9)
    (layer	ZnS   720.06e-9)
    (layer	Ge    412.85e-9)
    (layer	ZnS   761.47e-9)
    (layer	Ge     48.60e-9)
    (layer	ZnS    97.33e-9)
    (substrate  4)		 ;sspectra.com claims 4 for substrate.
    )))

;;; Matches
(define (anti)
  (define H 4.2)			;Ge
  (define L 2.2)			;ZnS
  (browse-optics-url "http://www.sspectra.com/designs/IR_AR.html")
  (plot-response
   (title "Growing an Antireflection Coating" "anti")
   (output-format 'png 600 256)
   (font 13)
   (wavelengths 7700e-9 12300e-9)
   (range 0 .02)
   (incident 0 'R)
   (optical-stack
    (substrate 1)
    (layer L   1076.97e-9)
    (layer H    629.03e-9)
    (layer L    457.49e-9)
    (layer H    131.55e-9)
    (layer L   1234.92e-9)
    (layer H   1142.59e-9)
    (layer L   2590.54e-9)
    (layer H     81.75e-9)
    (layer L   1827.50e-9)
    (layer H     84.60e-9)
    (layer L    563.34e-9)
    (layer H    285.97e-9)
    (layer L    157.14e-9)
    (substrate 4.0)			;Ge
    )))

;;; Matches
(define (immersed)
  (define SiO2 1.45)
  (define TiO2 2.35)
  (define BK7 1.5164)
  (browse-optics-url "http://www.sspectra.com/designs/immersedWPol.html")
  (plot-response
   (title "Immersed Wide-Angle Polarizer" "immersed")
   (output-format 'png 560 250)
   (font 13)
   (samples 50)
   (range .9 1)
   (wavelength .653e-6 'T_p)
   (wavelength .643e-6 'T_p)
   (wavelength .633e-6 'T_p)
   (wavelength .623e-6 'T_p)
   (wavelength .613e-6 'T_p)
   (angles 51 71)
   (stack-colors 'azure)
   (optical-stack
    (substrate BK7)
    (layer TIO2      83.62e-9)
    (layer SIO2      74.91e-9)
    (layer TIO2      94.86e-9)
    (layer SIO2     119.09e-9)
    (layer TIO2      90.21e-9)
    (layer SIO2     129.48e-9)
    (layer TIO2      81.87e-9)
    (layer SIO2     123.04e-9)
    (layer TIO2      84.03e-9)
    (layer SIO2     138.23e-9)
    (layer TIO2      81.34e-9)
    (layer SIO2     131.15e-9)
    (layer TIO2      83.85e-9)
    (layer SIO2     152.21e-9)
    (layer TIO2      80.48e-9)
    (layer SIO2     126.78e-9)
    (layer TIO2      59.95e-9)
    (substrate BK7)
    )))

;;; This gives performance as described.
(define (prism)
  (define H 2.35)
  (define L 1.45)
  (let ((stack
	 (optical-stack
	  (nominal  .55e-6)
	  (substrate  1.85)
	  (layer	    H     11.16e-9)
	  (layer	    L     48.87e-9)
	  (layer	    H     35.94e-9)
	  (layer	    L     68.58e-9)
	  (layer	    H     39.50e-9)
	  (layer	    L     79.77e-9)
	  (layer	    H     46.20e-9)
	  (layer	    L     96.61e-9)
	  (layer	    H     49.71e-9)
	  (layer	    L    102.74e-9)
	  (layer	    H     50.49e-9)
	  (layer	    L    102.74e-9)
	  (layer	    H     49.71e-9)
	  (layer	    L     96.61e-9)
	  (layer	    H     46.20e-9)
	  (layer	    L     79.77e-9)
	  (layer	    H     39.50e-9)
	  (layer	    L     68.58e-9)
	  (layer	    H     35.94e-9)
	  (layer	    L     48.87e-9)
	  (layer	    H     11.16e-9)
	  (substrate  1.85))))
    (browse-optics-url "http://www.sspectra.com/designs/ftr_polar.html")
    (plot-response (title "New Thin Film Polarizing Beamsplitter - S" "prism-s")
		   (range .99 1)
		   (output-format 'png 560 250)
		   (font 13)
		   (samples 50)
		   (stack-colors 'lusty)
		   (wavelengths .4e-6 .7e-6)
		   (incident 65 'T_s)
		   (incident 66 'T_s)
		   (incident 67 'T_s)
		   (incident 68 'T_s)
		   (incident 69 'T_s)
		   (incident 70 'T_s)
		   (incident 71 'T_s)
		   (incident 72 'T_s)
		   (incident 73 'T_s)
		   (incident 74 'T_s)
		   (incident 75 'T_s)
		   stack)
    (plot-response (title "New Thin Film Polarizing Beamsplitter - P" "prism-p")
		   (range -200 -10 'db)
		   (output-format 'png 560 250)
		   (font 13)
		   (samples 10)
		   (stack-colors 'azure)
		   (wavelengths .4e-6 .7e-6)
		   (incident 65 'T_p)
		   (incident 66 'T_p)
		   (incident 67 'T_p)
		   (incident 68 'T_p)
		   (incident 69 'T_p)
		   (incident 70 'T_p)
		   (incident 71 'T_p)
		   (incident 72 'T_p)
		   (incident 73 'T_p)
		   (incident 74 'T_p)
		   (incident 75 'T_p)
		   stack)))

;;;Sh. A. Furman and A. V. Tikhonravov.
;;;Basics of Optics of Multilayer Systems.
;;;Editions Fronti`eres, Gif-sur-Yvette Cedex -- France, 1992.
;;;http://www.optilayer.com/cgi-bin/od/odn.pl?ID=2
;;
;;Fig. 2.2: Reflectances of the 6-layer antireflection coatings
;;with nS = 1.52, nH = 2.30, nL = 1.45 (lambda = 500nm)
;; (1)6S .318H .34L 1.977H .106L .375H 1.099L,
;; (2)6S .34H .391L 1.104H .188L .710H 1.216L
(define (fig.2.2)
  (define nH 2.30)
  (define nL 1.45)
  (define nS 1.52)
  (plot-response (title "fig-2.2" "fig-2.2")
		 (output-format 'png 625 315)
		 (font 13)
		 (incident 0 'R)
		 (wavelengths 400e-9 900e-9)
		 (range 0 .02)
		 (optical-stack (nominal (/ 500e-9 4))
				(substrate 1)
				(layer nL 1.099)
				(layer nH 0.375)
				(layer nL 0.106)
				(layer nH 1.977)
				(layer nL 0.34)
				(layer nH 0.318)
				(substrate nS))
		 (optical-stack (nominal (/ 500e-9 4))
				(substrate 1)
				(layer nL 1.216)
				(layer nH 0.710)
				(layer nL 0.188)
				(layer nH 1.104)
				(layer nL 0.391)
				(layer nH 0.34)
				(substrate nS))))

;;Fig. 2.5: Reflectance of the 8-layer antireflection coating
;;with nS = 1.52, nH = 2.30, nL = 1.38 (lambda = 500nm)
;;  8S .295H .291L .149H .074L 1.915H ?? .107H 1.052L
;;The geometric thicknesses of the layers in nanometers
;;(starting from the substrate) are equal to
;;14.1; 26.4; 8.1; 6.7; 104.1; 9.7; 16.9; 95.3.
;;The optical thicknesses of the layers are equal to
;;32.4; 36.4; 18.6; 9.3; 239.4; 13.4; 38.9; 131.5.
(define (fig.2.5)
  (define nH 2.30)
  (define nL 1.38)
  (define nS 1.52)
  (plot-response (title "fig-2.5" "fig-2.5")
		 (output-format 'png 695 325)
		 (font 13)
		 (incident 0 'R)
		 (wavelengths 400e-9 800e-9)
		 (range 0 .004)
		 (optical-stack
		  (nominal 500e-9)
		  (substrate 1)
		  (layer nL 95.3e-9)
		  (layer nH 16.9e-9)
		  (layer nL 9.7e-9)
		  (layer nH 104.1e-9)
		  (layer nL 6.7e-9)
		  (layer nH 8.1e-9)
		  (layer nL 26.4e-9)
		  (layer nH 14.1e-9)
		  (substrate nS))))

;;; works
(define (HL7)
  (define glass 1.52)
  (define SiO2 1.46)
  (define TiO2 2.40)
  (browse-optics-url "http://www.kruschwitz.com/HR%27s.htm#The Basic HR")
  (plot-response
   (title "(HL)^7H on glass" "hl7")
   (output-format 'png 400 175)
   (font 13)
   (range .99 1.0)
   (wavelengths 400e-9 700e-9)
   (incident 0 'R)
   (stack-colors 'blue)
   (optical-stack
    (nominal  550e-9)
    (substrate 1)
    (layer TiO2 1/4)
    (layer SiO2 1/4)
    (layer TiO2 1/4)
    (layer SiO2 1/4)
    (layer TiO2 1/4)
    (layer SiO2 1/4)
    (layer TiO2 1/4)
    (layer SiO2 1/4)
    (layer TiO2 1/4)
    (layer SiO2 1/4)
    (layer TiO2 1/4)
    (layer SiO2 1/4)
    (layer TiO2 1/4)
    (layer SiO2 1/4)
    (layer TiO2 1/4)
    (substrate glass)
    )))

(define (cold-mirror)
  (define glass 1.52)
  (define SiO2 1.46)
  (define TiO2 2.40)
  (define nom 550e-9)
  (browse-optics-url "http://www.kruschwitz.com/Cold/hot.htm")
  (plot-response
   (title "Cold Mirror" "cold-mirror")
   (output-format 'png 435 175)
   (font 13)
   (wavelengths 300e-9 1100e-9)
   (incident 0 'R)
   (optical-stack (substrate 1)
		  (nominal nom)
		  (layer SiO2 1/2)
		  (nominal (* 0.8 nom))
		  (repeat 5
			  (layer TiO2 1/8)
			  (layer SiO2 1/4)
			  (layer TiO2 1/8))
		  (nominal (* 1.0 nom))
		  (repeat 5
			  (layer TiO2 1/8)
			  (layer SiO2 1/4)
			  (layer TiO2 1/8))
		  (nominal (* 1.2 nom))
		  (repeat 5
			  (layer TiO2 1/8)
			  (layer SiO2 1/4)
			  (layer TiO2 1/8))
		  (substrate glass))))

(define (cold-mirrors)
  (define glass 1.52)
  (define SiO2 1.46)
  (define TiO2 2.40)
  (browse-optics-url "http://www.sspectra.com/designs/coldmirror.html")
  ;;"http://www.sspectra.com/designs/coldmirror2.gif"
  (plot-response
   (title "Cold Mirrors" "cold-mirror1")
   (output-format 'png 585 280)
   (font 13)
   (range 0 1)
   (wavelengths 380e-9 1000e-9)
   (incident 0 'R)
   (optical-stack
    (layer TIO2      34.40e-9)
    (layer SIO2     158.61e-9)
    (layer TIO2      53.24e-9)
    (layer SIO2     128.87e-9)
    (layer TIO2      67.77e-9)
    (layer SIO2     102.19e-9)
    (layer TIO2      79.13e-9)
    (layer SIO2      97.19e-9)
    (layer TIO2      53.07e-9)
    (layer SIO2     162.54e-9)
    (layer TIO2      36.16e-9)
    (layer SIO2      66.56e-9)
    (layer TIO2      46.48e-9)
    (layer SIO2      87.18e-9)
    (layer TIO2      48.92e-9)
    (layer SIO2      82.52e-9)
    (layer TIO2      42.56e-9)
    (layer SIO2      73.13e-9)
    (layer TIO2      43.97e-9)
    (layer SIO2     102.57e-9)
    (layer TIO2      44.21e-9)
    (substrate glass)))
  (plot-response
   (title "Cold Mirrors" "cold-mirror2")
   (output-format 'png 585 280)
   (font 13)
   (range 0 1)
   (wavelengths 380e-9 2000e-9)
   (incident 0 'R)
   (optical-stack
    (layer TIO2    19.03e-9)
    (layer SIO2   191.13e-9)
    (layer TIO2    51.30e-9)
    (layer SIO2   137.38e-9)
    (layer TIO2    61.86e-9)
    (layer SIO2    99.64e-9)
    (layer TIO2    81.48e-9)
    (layer SIO2   102.08e-9)
    (layer TIO2    53.15e-9)
    (layer SIO2   162.37e-9)
    (layer TIO2    36.85e-9)
    (layer SIO2    70.41e-9)
    (layer TIO2    39.55e-9)
    (layer SIO2    88.77e-9)
    (layer TIO2    51.97e-9)
    (layer SIO2    86.41e-9)
    (layer TIO2    51.05e-9)
    (layer SIO2    66.36e-9)
    (layer TIO2    28.39e-9)
    (layer SIO2   124.04e-9)
    (layer TIO2    33.76e-9)
    (substrate glass))))

;;; works
(define (reflector-50)
  (define L 1.45)
  (define H 2.35)
  (browse-optics-url "http://www.sspectra.com/designs/bbhr.html")
  (plot-response
   (title "Broadband High-Reflection Coating at 50 Degrees" "reflect-50")
   (output-format 'png 560 256)
   (font 13)
   (range .8 1)
   (wavelengths .4e-6 .8e-6) (incident 50 'R_p) ;(incident 40 'R_p)
   ;;(wavelength .6e-6 'R_p) (wavelength .7e-6 'R_p) (angles 0 90)
   (optical-stack
    (nominal  .6e-6)
    (substrate 1)
    (layer H      49.35e-9)
    (layer L      80.64e-9)
    (layer H      45.96e-9)
    (layer L      91.80e-9)
    (layer H      51.59e-9)
    (layer L      86.62e-9)
    (layer H      46.16e-9)
    (layer L      90.92e-9)
    (layer H      53.49e-9)
    (layer L      94.22e-9)
    (layer H      46.62e-9)
    (layer L      82.60e-9)
    (layer H      52.26e-9)
    (layer L     101.16e-9)
    (layer H      56.86e-9)
    (layer L     101.69e-9)
    (layer H      63.51e-9)
    (layer L     121.69e-9)
    (layer H      61.47e-9)
    (layer L     111.11e-9)
    (layer H      63.21e-9)
    (layer L     121.61e-9)
    (layer H      71.53e-9)
    (layer L     113.89e-9)
    (layer H      55.42e-9)
    (layer L     104.04e-9)
    (layer H      69.96e-9)
    (layer L     137.71e-9)
    (layer H      71.17e-9)
    (layer L     141.89e-9)
    (layer H      82.96e-9)
    (layer L     141.64e-9)
    (layer H      85.36e-9)
    (layer L     129.75e-9)
    (layer H      72.02e-9)
    (layer L     178.45e-9)
    (layer H      83.10e-9)
    (layer L     154.04e-9)
    (layer H      74.35e-9)
    (layer L     152.15e-9)
    (layer H      80.56e-9)
    (layer L     171.35e-9)
    (layer H      97.87e-9)
    (substrate 1.52)
    )))

;;; good match midband; not so good elsewhere.
(define (omni-mirror)
  (define H 4.6)			;tellurium
  (define L 1.6)			;polystyrene
  (browse-optics-url "http://www.sspectra.com/designs/omnirefl.html")
  (plot-response
   (title "Dielectric Omnidirectional Reflector" "omni-mirror")
   (output-format 'png 565 250)
   (font 13)
   (range 0 1)
   (wavelengths 6e-6 20e-6)
   (incident 10 'R)
   (incident 20 'R)
   (incident 30 'R)
   (incident 40 'R)
   (incident 50 'R)
   (incident 60 'R)
   (incident 70 'R)
   (incident 80 'R)
   (incident 89 'R)
   (stack-colors 'azure)
   (optical-stack
    (substrate 1)
    (layer	    H    399.00e-9)
    (layer	    L   2165.33e-9)
    (layer	    H    696.76e-9)
    (layer	    L   2097.40e-9)
    (layer	    H    701.51e-9)
    (layer	    L   2089.58e-9)
    (layer	    H    700.83e-9)
    (layer	    L   2112.33e-9)
    (layer	    H    691.88e-9)
    (substrate  1.5442)
    )))

(define (dielectric)
  (wide-bp)
  (anti)
  (immersed)
  (prism)
  (fig.2.2)
  (fig.2.5)
  (HL7)
  (cold-mirror)
  (cold-mirrors)
  (reflector-50)
  (omni-mirror))

;;(dielectric)
