/*
   (C) 2005 Anand Avati <avati@zresearch.com>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free
   Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301 USA
*/

#include <guile/gh.h>

#include "commands.h"
#include "freetalk.h"
#include "extensions.h"
#include "roster.h"
#include "presence.h"

SCM ex_load (SCM scm_file)
{
  char *file;

  file = gh_scm2newstr (scm_file, NULL);

  ft_load (file);

  free (file);

  return SCM_UNSPECIFIED;
}

SCM ex_get_jid (void)
{
  return gh_str02scm (do_get_jid ());
}

SCM ex_set_jid (SCM scm_jid)
{
  char *jid;

  jid = gh_scm2newstr (scm_jid, NULL);

  do_set_jid (jid);
  free (jid);

  return SCM_UNSPECIFIED;
}

SCM ex_get_server (void)
{
  return gh_str02scm (do_get_server ());
}

SCM ex_set_server (SCM scm_server)
{
  char *server;

  server = gh_scm2newstr (scm_server, NULL);

  do_set_server (server);
  free (server);

  return SCM_UNSPECIFIED;
}

SCM ex_get_password (void)
{
  return gh_str02scm (do_get_password ());
}

SCM ex_set_password (SCM scm_password)
{
  char *password;

  password = gh_scm2newstr (scm_password, NULL);

  do_set_password (password);
  free (password);

  return SCM_UNSPECIFIED;
}

SCM ex_get_sslconn (void)
{
  return gh_bool2scm (state.need_ssl);
}

SCM ex_set_sslconn (SCM scm_ssl)
{
  do_set_ssl (gh_scm2bool (scm_ssl));

  return SCM_UNSPECIFIED;
}

SCM ex_connect (void)
{
  return gh_int2scm (do_connect());
}

SCM ex_disconnect (void)
{
  do_disconnect ();

  return SCM_UNSPECIFIED;
}

SCM 
ex_send_message (SCM scm_to, SCM scm_msg)
{
  char *to = gh_scm2newstr (scm_to, NULL);
  char *msg = gh_scm2newstr (scm_msg, NULL);

  do_send_message (to, msg);

  free (to);
  free (msg);
  
  return SCM_UNSPECIFIED;
}

SCM
ex_send_message_no_hook (SCM scm_to, SCM scm_msg)
{
  char *to = gh_scm2newstr (scm_to, NULL);
  char *msg = gh_scm2newstr (scm_msg, NULL);
  
  do_send_message_no_hook (to, msg);

  free (to);
  free (msg);

  return SCM_UNSPECIFIED;
}

SCM
ex_set_daemon (void)
{
  do_set_daemon ();
  return SCM_UNSPECIFIED;
}

SCM
ex_get_daemon (void)
{
  return gh_int2scm (do_get_daemon ());
}

SCM
ex_set_port (SCM scm_port)
{
  do_set_port (gh_scm2int (scm_port));
  return SCM_UNSPECIFIED;
}

SCM
ex_get_port (void)
{
  return gh_int2scm (do_get_port ());
}

SCM
ex_get_prompt (void)
{
  return gh_str02scm (do_get_prompt ());
}

SCM
ex_set_prompt (SCM scm_prompt)
{
  char *prompt;

  prompt = gh_scm2newstr (scm_prompt, NULL);

  do_set_prompt (prompt);
  free (prompt);

  return SCM_UNSPECIFIED;
}

SCM
ex_display (SCM scm_line)
{
  char *line = gh_scm2newstr (scm_line, NULL);

  PRINTF ("%s", line);
  free (line);

  return SCM_UNSPECIFIED;
}

SCM
ex_get_conn_status (void)
{
  return gh_int2scm (do_get_conn_status ());
}

SCM
ex_add_buddy (SCM scm_jid)
{
  char *jid = SCM_STRING_CHARS (scm_jid);
  do_add (jid);
  return SCM_UNSPECIFIED;
}

SCM
ex_remove_buddy (SCM scm_jid)
{
  char *jid = SCM_STRING_CHARS (scm_jid);
  if (do_get_conn_status () != FT_AUTH)
    {
      PRINTF ("Not connected, (use /connect)");
      return gh_bool2scm (FALSE);
    }
  ft_roster_remove (jid);
  return gh_bool2scm (TRUE);
}

SCM
ex_hook_return (void)
{
  set_hook_return (1);
  return SCM_UNSPECIFIED;
}

SCM
ex_get_current_buddy (void)
{
  return gh_str02scm (do_get_current_buddy ());
}

SCM
ex_set_current_buddy (SCM scm_bud)
{
  char *buddy = gh_scm2newstr (scm_bud, NULL);
  do_set_current_buddy (buddy);
  g_free (buddy);
  return SCM_UNSPECIFIED;
}

SCM
ex_set_status_msg (SCM scm_status)
{
  char *status = gh_scm2newstr (scm_status, NULL);
  do_set_status_msg (status);
  g_free (status);
  return SCM_UNSPECIFIED;
}

SCM
ex_get_status_msg (void)
{
  return gh_str02scm (do_get_status_msg ());
}

/* For internal use only */
SCM
roster_item_to_list (FtRosterItem *item)
{
  return gh_list (gh_str02scm (item->jid),
		  gh_bool2scm (item->is_online),
		  gh_str02scm (item->nickname ? item->nickname : ""),
		  gh_str02scm (item->show_msg ? item->show_msg : ""),
		  gh_str02scm (item->status_msg ? item->status_msg : ""),
		  SCM_UNDEFINED);
}

static void
roster_iterator (gpointer r_item, gpointer retval)
{
  FtRosterItem *item = (FtRosterItem *)r_item;
  SCM *ret = (SCM *)retval;
  
  *ret = scm_cons (roster_item_to_list (item), *ret);
}

SCM
ex_get_roster_list (void)
{
  SCM ret = SCM_EOL;
  ft_roster_foreach (roster_iterator, (gpointer) &ret);
  return ret;
}

SCM
ex_get_roster_status_msg (SCM scm_jid)
{
  char *jid = SCM_STRING_CHARS (scm_jid);
  return gh_str02scm (ft_roster_lookup (jid)->status_msg);
}

SCM
ex_get_roster_is_online (SCM scm_jid)
{
  char *jid = SCM_STRING_CHARS (scm_jid);
  return ft_roster_lookup (jid)->is_online ? SCM_BOOL_T : SCM_BOOL_F;
}

SCM
ex_roster_lookup (SCM scm_jid)
{
  FtRosterItem *item = ft_roster_lookup (SCM_STRING_CHARS (scm_jid));
  if (item)
    return roster_item_to_list (item);
  else
    return SCM_EOL;
}

SCM
ex_reset_fs_state (void)
{
  return gh_int2scm (do_reset_fs_state ());
}

SCM
ex_dict_append (SCM scm_word)
{
  char *word = gh_scm2newstr (scm_word, NULL);

  do_dict_append (word);
  return SCM_UNSPECIFIED;
}

SCM
ex_dict_prepend (SCM scm_word)
{
  char *word = gh_scm2newstr (scm_word, NULL);

  do_dict_prepend (word);
  return SCM_UNSPECIFIED;
}

SCM
ex_dict_insert (SCM scm_word)
{
  char *word = gh_scm2newstr (scm_word, NULL);

  do_dict_insert (word);
  return SCM_UNSPECIFIED;
}

SCM
ex_dict_remove (SCM scm_word)
{
  char *word = gh_scm2newstr (scm_word, NULL);

  do_dict_remove (word);
  g_free (word);
  return SCM_UNSPECIFIED;
}

SCM
ex_version (void)
{
  return gh_str02scm (PACKAGE_VERSION);
}

SCM
ex_login_blocking (void)
{
  return gh_int2scm (do_connect_blocking ());
}

SCM
ex_cli_on (void)
{
  if (state.daemon)
    return gh_bool2scm (FALSE);

  interface_init ();
  return gh_bool2scm (TRUE);
}

SCM
ex_main_loop (void)
{
  do_main_loop ();

  return SCM_UNSPECIFIED;
}

SCM
ex_load_default_config (void)
{
  load_default_config ();

  return SCM_UNSPECIFIED;
}

SCM
ex_get_config_dir (void)
{
    return gh_str02scm (state.config_dir);   
}

SCM
ex_subscription_allow (SCM scm_jid)
{
  char *jid = gh_scm2newstr (scm_jid, NULL);
  ft_presence_subscription_allow (jid);
  free (jid);
  return SCM_UNSPECIFIED;
}

SCM
ex_subscription_deny (SCM scm_jid)
{
  char *jid = gh_scm2newstr (scm_jid, NULL);
  ft_presence_subscription_deny (jid);
  free (jid);
  return SCM_UNSPECIFIED;
}
