/*  ga_pat_get_fitness.c */
/* 	Copyright 2004-2006 Oswaldo Morizaki Hirakata */
/* 	This file is part of ga-nn-ag-2 

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "my_header.h"
#include "aux_prot.h"

/* 
-	Return 0 on success, different to zero otherwise
-	Get pattern fitness per category
-	Also get category fitness
- ga_pat_index->pat_fitness should be already allocated

pat_group_mode: (default whole as one)
	0 = whole as one
	1 = Group by column (num_elem_output)
	2 = Group by row (num_output)
	3 = No grouping at all 

pat_fitness_mode: (default exponential)
	0 = exponential: base^(-(error - error_min)*boost / error_range)
	1 = radix: 1 + base*(1 - (error - error_min)/error_range)^(1/boost)

pat_error_mode: (default rms)
	0 = rms
	1 = avg
	2 = max
*/


int ga_pat_get_fitness(struct ga_server_config * conf, 
													struct ga_service_client_index * index)
{
	int k,l,m,n,o,p;
	int ret;
	int num_cat;
	int num_column;
	int num_row;
	int counter;

	int * num_pat_cat;

	double ***error_max;
	double ***error_min;
	double ***range;
	double * cat_error;
	double cat_error_min;
	double cat_error_max;
	double cat_error_range;
	double temp;
	
	double base = conf->pat_base;
	double boost = conf->pat_boost;
	
	num_cat = index->clients[0]->pat_index->num_cat;
	
	/* Memory reserve for cat_error, num_pat_cat, error_max, error_min, range */	
	if (!(cat_error = (double *)calloc(num_cat,sizeof(double ))))
	{
		syslog(LOG_CRIT,"Error malloc cat_error in ga_pat_get_fitness: %s",strerror(errno));
		return(1);
	}
	if (!(num_pat_cat = (int *)calloc(num_cat,sizeof(int))))
	{
		syslog(LOG_CRIT,"Error malloc num_pat_cat in ga_pat_get_fitness: %s",strerror(errno));
		free(cat_error);
		return(1);
	}
	
	if (!(error_max = (double ***)malloc(num_cat*sizeof(double **))))
	{
		syslog(LOG_CRIT,"Error malloc error_max in ga_pat_get_fitness: %s",strerror(errno));
		free(cat_error);
		free(num_pat_cat);
		return(1);
	}
	if (!(error_min = (double ***)malloc(num_cat*sizeof(double **))))
	{
		syslog(LOG_CRIT,"Error malloc error_min in ga_pat_get_fitness: %s",strerror(errno));
		free(cat_error);
		free(num_pat_cat);
		free(error_max);
		return(1);
	}
	if (!(range = (double ***)malloc(num_cat*sizeof(double **))))
	{
		syslog(LOG_CRIT,"Error malloc range in ga_pat_get_fitness: %s",strerror(errno));
		free(cat_error);
		free(num_pat_cat);
		free(error_max);
		free(error_min);
		return(1);
	}
	for (k=0; k< num_cat; k++)
	{
		switch (conf->pat_group_mode)
		{
			case 1: // group by column
			{
				num_column = index->clients[0]->pat_index->cat[k]->pat[0]->num_elem_output;
				num_row = 1;
				break;
			}	
			case 2: // group by row
			{
				num_column = 1;
				num_row = index->clients[0]->pat_index->cat[k]->pat[0]->num_output;
				break;
			}
			case 3: // No grouping
			{
				num_column = index->clients[0]->pat_index->cat[k]->pat[0]->num_elem_output;
				num_row = index->clients[0]->pat_index->cat[k]->pat[0]->num_output;
				break;
			}
			default: // Everything as one
			{
				num_column = 1;
				num_row = 1;
				break;
			}
		}

/* FIXME: Memory leak in case of failure in allocation */
		if (!(error_max[k] = (double **)malloc(num_row * sizeof(double *))))
		{
			syslog(LOG_CRIT,"Error malloc error_max[%d] in ga_pat_get_fitness: %s", k, strerror(errno));
			return(1);
		}
		if (!(error_min[k] = (double **)malloc(num_row*sizeof(double *))))
		{
			syslog(LOG_CRIT,"Error malloc error_min[%d] in ga_pat_get_fitness: %s", k, strerror(errno));
			return(1);
		}
		if (!(range[k] = (double **)malloc(num_row*sizeof(double *))))
		{
			syslog(LOG_CRIT,"Error malloc range[%d] in ga_pat_get_fitness: %s", k, strerror(errno));
			return(1);
		}
	
		for (l=0; l< num_row; l++)
		{
			if (!(error_max[k][l] = (double *)calloc(num_column, sizeof(double))))
			{
				syslog(LOG_CRIT,"Error calloc error_max[%d][%d] in ga_pat_get_fitness: %s", k,l, strerror(errno));
				return(1);
			}
			if (!(error_min[k][l] = (double *)calloc(num_column, sizeof(double))))
			{
				syslog(LOG_CRIT,"Error calloc error_min[%d][%d] in ga_pat_get_fitness: %s", k,l, strerror(errno));
				return(1);
			}
			if (!(range[k][l] = (double *)calloc(num_column, sizeof(double))))
			{
				syslog(LOG_CRIT,"Error calloc range[%d][%d] in ga_pat_get_fitness: %s", k,l, strerror(errno));
				return(1);
			}
		}
	}

	/* Memory reserve for pat_fitness */
	for (k=0; k< index->num_client; k++)
	{
		for (l=0; l< index->clients[k]->pat_index->num_cat; l++)
		{
			for (m=0; m< index->clients[k]->pat_index->cat[l]->num_pat; m++)
			{
				switch(conf->pat_group_mode)
				{
					case 1: // group by column
					{
						num_column = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output;
						num_row = 1;
						break;
					}	
					case 2: // group by row
					{
						num_column = 1;
						num_row = index->clients[k]->pat_index->cat[l]->pat[m]->num_output;
						break;
					}
					case 3: // No grouping
					{
						num_column = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output;
						num_row = index->clients[k]->pat_index->cat[l]->pat[m]->num_output;
						break;
					}
					default: // Everything as one
					{
						num_column = 1;
						num_row = 1;
						break;
					}
				}

				if (!(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness = 
							(float **)malloc(num_row*sizeof(float *))))
				{
					syslog(LOG_CRIT,"Error malloc clients[%d]->pat_index->cat[%d]->pat[%d]->pat_fitness: %s",
								k, l, m, strerror(errno));
					return(1);
				}
				for (n=0; n< num_row; n++)
				{
					if (!(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness[n] = 
								(float *)calloc(num_column, sizeof(float))))
					{
						syslog(LOG_CRIT,"Error calloc clients[%d]->pat_index->cat[%d]->pat[%d]->pat_fitness[%d]: %s",
									k, l, m, n, strerror(errno));
						return(1);
					}
				}
				index->clients[k]->pat_index->cat[l]->pat[m]->num_fitness = num_row;
				index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_fitness = num_column;
			}
		}
	}
	
	
	/* Put error in pat_fitness, then get fitness */
	p = index->clients[k]->ret->num_pat - 1;
	switch(conf->error_mode)
	{
		/********************************************/
		/*               Average                    */
		/********************************************/
		case 1: 
		{
			switch(conf->pat_group_mode)
			{
				case 1: // group by column (num_elem_output)
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; n+1 ; n--)
								{
									temp = 0.0;
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; o+1 ; o--)
									{
										temp += fabs(index->clients[k]->ret->error[p][o][n]);
									}
									if ((!finite(temp))	|| (temp > SATURATION))
									{
										temp = SATURATION;
									}
									
									cat_error[l] += temp;
									num_pat_cat[l] += index->clients[k]->ret->num_output;
									
									temp /= index->clients[k]->ret->num_output;
									index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness[0][n] = temp;
									
									error_max[l][0][n] = (error_max[l][0][n] > temp) ? error_max[l][0][n] : temp;
									error_min[l][0][n] = (error_min[l][0][n] > temp) ? temp : error_min[l][0][n];
								}
								p -= 1;
							}
						}
					}
					break;
				}
				case 2: // group by row (num_output)
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; n+1 ; n--)
								{
									temp = 0.0;
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; o+1 ; o--)
									{
										temp += fabs(index->clients[k]->ret->error[p][n][o]);
									}
									if ((!finite(temp))	|| (temp > SATURATION))
									{
										temp = SATURATION;
									}
									
									cat_error[l] += temp;
									num_pat_cat[l] += index->clients[k]->ret->num_elem[p];
									
									temp /= index->clients[k]->ret->num_elem[p];
									*(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness[n]) = temp;

									error_max[l][n][0] = (error_max[l][n][0] > temp) ? error_max[l][n][0] : temp;
									error_min[l][n][0] = (error_min[l][n][0] > temp) ? temp : error_min[l][n][0];
								}
								p -= 1;
							}
						}
					}
					break;
				}
				case 3: // No grouping 
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; n+1 ; n--)
								{
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; o+1 ; o--)
									{
										temp = fabs(index->clients[k]->ret->error[p][n][o]);
										if ((!finite(temp))	|| (temp > SATURATION))
										{
											temp = SATURATION;
										}
										cat_error[l] += temp;
										num_pat_cat[l]++;
										
										*(*(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness+n)+o) = temp;

										*(*(error_max[l]+n)+o) = (error_max[l][n][o] > temp) ? error_max[l][n][o] : temp;
										*(*(error_min[l]+n)+o) = (error_min[l][n][o] > temp) ? temp : error_min[l][n][o];
									}
								}
							}
							p -= 1;
						}
					}
					break;
				}
				default: // Whole as one
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								temp = 0.0;
								counter = 0;
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; n+1 ; n--)
								{
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; o+1 ; o--)
									{
										temp += fabs(index->clients[k]->ret->error[p][n][o]);
										counter++;
									}
								}
								if ((!finite(temp))	|| (temp > SATURATION))
								{
									temp = SATURATION;
								}
								
								cat_error[l] += temp;
								num_pat_cat[l] += counter;
								
								temp /= counter;
								*(*(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness)) = temp;

								*(*(error_max[l])) = (error_max[l][0][0] > temp) ? error_max[l][0][0] : temp;
								*(*(error_min[l])) = (error_min[l][0][0] > temp) ? temp : error_min[l][0][0];
								
								p -= 1;
							}
						}
					}
					break;
				}
			}
			break;
		}
		
		/********************************************/
		/*                  Max                     */
		/********************************************/
		case 2: // max
		{
			switch(conf->pat_group_mode)
			{
				case 1: // group by column (num_elem_output)
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; n+1 ; n--)
								{
									temp = 0.0;
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; o+1 ; o--)
									{
										temp = (temp > fabs(index->clients[k]->ret->error[p][o][n])) ? 
														temp : fabs(index->clients[k]->ret->error[p][o][n]);
									}
									if ((!finite(temp))	|| (temp > SATURATION))
									{
										temp = SATURATION;
									}
									
									cat_error[l] += temp;
									num_pat_cat[l]++;
									
									index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness[0][n] = temp;

									error_max[l][0][n] = (error_max[l][0][n] > temp) ? error_max[l][0][n] : temp;
									error_min[l][0][n] = (error_min[l][0][n] > temp) ? temp : error_min[l][0][n];
								}
								p -= 1;
							}
						}
					}
					break;
				}
				case 2: // group by row (num_output)
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; n+1 ; n--)
								{
									temp = 0.0;
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; o+1 ; o--)
									{
										temp = (temp > fabs(index->clients[k]->ret->error[p][n][o])) ? 
														temp : fabs(index->clients[k]->ret->error[p][n][o]);
									}
									if ((!finite(temp))	|| (temp > SATURATION))
									{
										temp = SATURATION;
									}
									
									cat_error[l] += temp;
									num_pat_cat[l]++;
									
									*(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness[n]) = temp;

									error_max[l][n][0] = (error_max[l][n][0] > temp) ? error_max[l][n][0] : temp;
									error_min[l][n][0] = (error_min[l][n][0] > temp) ? temp : error_min[l][n][0];
								}
								p -= 1;
							}
						}
					}
					break;
				}
				case 3: // No grouping 
				{
					temp = 0.0;
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; n+1 ; n--)
								{
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; o+1 ; o--)
									{
										temp = fabs(index->clients[k]->ret->error[p][n][o]);
										if ((!finite(temp))	|| (temp > SATURATION))
										{
											temp = SATURATION;
										}
										
										cat_error[l] += temp;
										num_pat_cat[l]++;
										
										*(*(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness+n)+o) = temp;

										*(*(error_max[l]+n)+o) = (error_max[l][n][o] > temp) ? error_max[l][n][o] : temp;
										*(*(error_min[l]+n)+o) = (error_min[l][n][o] > temp) ? temp : error_min[l][n][o];
									}
								}
							}
							p -= 1;
						}
					}
					break;
				}
				default: // Whole as one
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								temp = 0.0;
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; n+1 ; n--)
								{
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; o+1 ; o--)
									{
										temp = (temp > fabs(index->clients[k]->ret->error[p][n][o])) ? 
														temp : fabs(index->clients[k]->ret->error[p][n][o]);
									}
								}
								if ((!finite(temp))	|| (temp > SATURATION))
								{
									temp = SATURATION;
								}
								
								cat_error[l] += temp;
								num_pat_cat[l]++;
								
								*(*(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness)) = temp;

								*(*(error_max[l])) = (error_max[l][0][0] > temp) ? error_max[l][0][0] : temp;
								*(*(error_min[l])) = (error_min[l][0][0] > temp) ? temp : error_min[l][0][0];

								p -= 1;
							}
						}
					}
					break;
				}
			}
			break;
		}

		/********************************************/
		/*                  RMS                     */
		/********************************************/
		default: 
		{
			switch(conf->pat_group_mode)
			{
				case 1: // group by column (num_elem_output)
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; n+1 ; n--)
								{
									temp = 0.0;
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; o+1 ; o--)
									{
										temp += pow(index->clients[k]->ret->error[p][o][n],2);
									}
									if ((!finite(temp))	|| (temp > SATURATION))
									{
										temp = SATURATION;
									}
									
									cat_error[l]+= temp;
									num_pat_cat[l]+= index->clients[k]->ret->num_output;
									
									temp /= index->clients[k]->ret->num_output;
									temp = pow(temp,0.5);
									index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness[0][n] = temp;
									
									error_max[l][0][n] = (error_max[l][0][n] > temp) ? error_max[l][0][n] : temp;
									error_min[l][0][n] = (error_min[l][0][n] > temp) ? temp : error_min[l][0][n];
								}
								p -= 1;
							}
						}
					}
					break;
				}
				case 2: // group by row (num_output)
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; n+1 ; n--)
								{
									temp = 0.0;
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; o+1 ; o--)
									{
										temp += pow(index->clients[k]->ret->error[p][n][o],2);
									}
									if ((!finite(temp))	|| (temp > SATURATION))
									{
										temp = SATURATION;
									}
									
									cat_error[l]+= temp;
									num_pat_cat[l]+= index->clients[k]->ret->num_elem[p];
									
									temp /= index->clients[k]->ret->num_elem[p];
									temp = pow(temp,0.5);
									*(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness[n]) = temp;

									error_max[l][n][0] = (error_max[l][n][0] > temp) ? error_max[l][n][0] : temp;
									error_min[l][n][0] = (error_min[l][n][0] > temp) ? temp : error_min[l][n][0];
								}
								p -= 1;
							}
						}
					}
					break;
				}
				case 3: // No grouping 
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; n+1 ; n--)
								{
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; o+1 ; o--)
									{
										temp = fabs(index->clients[k]->ret->error[p][n][o]);
										if ((!finite(temp))	|| (temp > SATURATION))
										{
											temp = SATURATION;
										}
										
										cat_error[l]+= temp;
										num_pat_cat[l]++;
								
										*(*(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness+n)+o) = temp;

										*(*(error_max[l]+n)+o) = (error_max[l][n][o] > temp) ? error_max[l][n][o] : temp;
										*(*(error_min[l]+n)+o) = (error_min[l][n][o] > temp) ? temp : error_min[l][n][o];
									}
								}
							}
							p -= 1;
						}
					}
					break;
				}
				default: // Whole as one
				{
					for (k = index->num_client-1; k+1 ; k--)
					{
						for (l = index->clients[k]->pat_index->num_cat-1; l+1; l--)
						{
							for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1 ; m--)
							{
								temp = 0.0;
								counter = 0;
								for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_output-1; n+1 ; n--)
								{
									for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_output-1; o+1 ; o--)
									{
										temp += pow(index->clients[k]->ret->error[p][n][o],2);
										counter++;
									}
								}
								if ((!finite(temp))	|| (temp > SATURATION))
								{
									temp = SATURATION;
								}
								
								cat_error[l]+= temp;
								num_pat_cat[l]+= counter;

								
								temp /= counter;
								temp = pow(temp,0.5);
								*(*(index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness)) = temp;

								*(*(error_max[l])) = (error_max[l][0][0] > temp) ? error_max[l][0][0] : temp;
								*(*(error_min[l])) = (error_min[l][0][0] > temp) ? temp : error_min[l][0][0];

								p -= 1;
							}
						}
					}
					break;
				}
			}
		}
		break;
	}
	
	/* Get error range */
	cat_error_min = 0;
	cat_error_max = 0;
	for (k = 0; k < num_cat; k++)
	{
		cat_error[k] /= num_pat_cat[k];
		if (conf->pat_error_mode == 0) // rms
		{
			cat_error[k] = pow(cat_error[k],0.5);
		}

		cat_error_max = (cat_error_max > cat_error[k]) ? cat_error_max : cat_error[k];
		cat_error_min = (cat_error_min > cat_error[k]) ? cat_error[k] : cat_error_min;

		switch (conf->pat_group_mode)
		{
			case 1: // group by column
			{
				num_column = index->clients[0]->pat_index->cat[k]->pat[0]->num_elem_output;
				num_row = 1;
				break;
			}	
			case 2: // group by row
			{
				num_column = 1;
				num_row = index->clients[0]->pat_index->cat[k]->pat[0]->num_output;
				break;
			}
			case 3: // No grouping
			{
				num_column = index->clients[0]->pat_index->cat[k]->pat[0]->num_elem_output;
				num_row = index->clients[0]->pat_index->cat[k]->pat[0]->num_output;
				break;
			}
			default: // Everything as one
			{
				num_column = 1;
				num_row = 1;
				break;
			}
		}
		
		for (l = 0; l < num_row; l++)
		{
			for (m = 0; m < num_column; m++)
			{
				range[k][l][m] = (error_max[k][l][m] > error_min[k][l][m]) ? 
									error_max[k][l][m] - error_min[k][l][m] : 1.0;
			}
		}
	}
	cat_error_range = (cat_error_max > cat_error_min) ? (cat_error_max - cat_error_min): 1.0;
	
	/* Set fitness */
	switch (conf->pat_fitness_mode)
	{
		case 1: // radix
		{
			for (k = num_cat-1; k+1; k--)
			{
				cat_error[k] = 1.0 + base*pow( (1.0 - (cat_error_max - cat_error[k])/cat_error_range),(1.0/boost));
			}
		
			for (k = index->num_client-1; k+1 ; k--)
			{
				for (l = index->clients[k]->pat_index->num_cat-1 ; l+1 ; l--)
				{
					for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1; m--)
					{
						index->clients[k]->pat_index->cat[l]->pat[m]->fitness = 0.0;
						for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_fitness-1 ; n+1; n--)
						{
							for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_fitness-1 ; o+1; o--)
							{
								temp = 1.0 + base*pow( (1.0 - 
									(error_max[l][n][o] - 
									index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness[n][o]
									)/range[l][n][o]),(1.0/boost));

								index->clients[k]->pat_index->cat[l]->pat[m]->fitness += temp;
							}
						}
					}
					index->clients[k]->pat_index->cat[l]->cat_fitness = cat_error[l];
				}
			}
			break;
		}
		default :
		{
			for (k = num_cat-1; k+1; k--)
			{
				cat_error[k] = pow(base, (cat_error_max - cat_error[k])*boost/cat_error_range);
			}

			for (k = index->num_client-1; k+1 ; k--)
			{
				for (l = index->clients[k]->pat_index->num_cat-1 ; l+1 ; l--)
				{
					for (m = index->clients[k]->pat_index->cat[l]->num_pat-1; m+1; m--)
					{
						index->clients[k]->pat_index->cat[l]->pat[m]->fitness = 0.0;
						for (n = index->clients[k]->pat_index->cat[l]->pat[m]->num_fitness-1 ; n+1; n--)
						{
							for (o = index->clients[k]->pat_index->cat[l]->pat[m]->num_elem_fitness-1 ; o+1; o--)
							{
								temp = pow(base, (error_max[l][n][o] - 
											index->clients[k]->pat_index->cat[l]->pat[m]->pat_fitness[n][o]
											)*boost/range[l][n][o]);							
								index->clients[k]->pat_index->cat[l]->pat[m]->fitness += temp;
							}
						}
					}
					index->clients[k]->pat_index->cat[l]->cat_fitness = cat_error[l];
				}
			}
			break;
		}		
	}	

	/* Memory clean */
	for (k=0; k< num_cat; k++)
	{
		switch (conf->pat_group_mode)
		{
			case 1: // group by column
			{
				num_row = 1;
				break;
			}	
			case 2: // group by row
			{
				num_row = index->clients[0]->pat_index->cat[k]->pat[0]->num_output;
				break;
			}
			case 3: // No grouping
			{
				num_row = index->clients[0]->pat_index->cat[k]->pat[0]->num_output;
				break;
			}
			default: // Everything as one
			{
				num_row = 1;
				break;
			}
		}
		for (l = 0; l < num_row; l++)
		{
			free(error_max[k][l]);
			free(error_min[k][l]);
			free(range[k][l]);
		}
		free(error_max[k]);
		free(error_min[k]);
		free(range);
	}
	free(cat_error);
	free(num_pat_cat);
	free(error_max);
	free(error_min);
	free(range);
	
	return(0);
}
