/*  ga_service_server.c */
/*  Copyright 2004-2006 Oswaldo Morizaki Hirakata */

/*  This file is part of ga-nn-ag-2.

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* 

In case this server is a master server, and multi niche/hosts are set, it
calls to remotes command servers.

After a "cycle", a new (master provider) thread is created to provide
service to a new generation

        |-----------|
        |  service  |
        |-----------|
        /           \
|-------|           |--------|
|startup|           |provider|
|-------|           |--------|

- "startup" is fork process. It starts the clients, and waits for their exit status.

- provider requires two at least two threads:
		- listen thread, who adds connection descriptors
		- service thread, who provides service, modifies common neural network data and
			discard old data and descriptors.
*/

#include "my_header.h"
#include "aux_prot.h"


extern int ga_errno;

/* Global config structure */
struct ga_server_config conf;

/* global alarm time */
int time_wait;

/* global exit_flag for sigterm */
int exit_flag = 0;

/* global num_provider */
int num_provider;

/* global curr_gen for service provider */
int current_generation;

/* global generations for service provider */
int generations;

/* global pat_generation for co-evolution */
int pat_generation;

/* global client index for service provider */
struct ga_service_client_index *client_index = NULL;

/* global base pattern index for service provider */
struct ga_pat_index * pat_index = NULL;

/* global pattern index for service provider */
int num_pat_index;

/* global mutex for client_index in service_provider */
pthread_mutex_t index_mutex = PTHREAD_MUTEX_INITIALIZER;

/* global conditional variable for client_index */
pthread_cond_t index_cond = PTHREAD_COND_INITIALIZER;

/* global mutex for current_generation variable */
pthread_mutex_t generation_mutex = PTHREAD_MUTEX_INITIALIZER;

/* global conditional variable for current_generation variable */
pthread_cond_t generation_cond = PTHREAD_COND_INITIALIZER;

/* global mutex for pat_generation variable */
pthread_mutex_t pat_mutex = PTHREAD_MUTEX_INITIALIZER;

/* global conditional variable for pat_generation variable */
pthread_cond_t pat_cond = PTHREAD_COND_INITIALIZER;

int ga_service_server(struct ga_server_config * conf)
{
int k,l;
int connfd;
int listenfd;
int retry;
int status;
int ret;
int temp_fd;

char empty_char;


pid_t startup_client_pid;
pid_t temp_pid;

pthread_t tid;

pthread_t listen_tid;
pthread_t provider_tid;

fd_set read_fd;
fd_set write_fd;

int * connptr = NULL;
int * num_local = NULL;

char char_temp[BUFFSIZE];

char ** char_vector = NULL;

FILE * remote_config_file = NULL;

struct sigaction sig_term;
struct sigaction sig_chld;

struct io_connection remote_con;

struct ga_server_config remote_conf;

struct io_block * remote_block = NULL;
struct io_block * nn_block = NULL;

struct ga_client_info * client_info = NULL;

struct ga_service_listen_thread_param listen_param;
struct ga_service_provider_thread_param provider_param;

struct sockaddr_in server_addr;

syslog(LOG_INFO,"Starting ga_service_server with pid: %d", getpid());

/* Set generations */
generations = conf->generations;

/* Set basic remote_con parameters */
bzero(&remote_con,sizeof(remote_con)); //set parameters to zero
remote_con.socket = 1; //connection via socket
sprintf(remote_con.port,"%d\0", conf->niche_port);

/***** SIG_CHLD Handler for all children *****/
/* ========================================= */
sigemptyset(&sig_chld.sa_mask);
sig_chld.sa_sigaction = ga_service_sig_chld;
sig_chld.sa_flags = 0;
sig_chld.sa_flags |= SA_RESTART;
sig_chld.sa_flags |= SA_NOMASK;
if (sigaction(SIGCHLD, &sig_chld, NULL) < 0)
{
	syslog(LOG_CRIT,"Couldn't set SIGCHLD handler");
	return(-1);
}

/********************* NICHE CODE ***********************/
/* ==================================================== */
/* Check if remote niches are set and is a master server */
if ((conf->master == 1) && (conf->num_niches > 0))
{
	/* Start remote slaves */
	/***********************/
	for (k=0; k< conf->num_niches; k++)
	{
		/* Calloc io_block */
		if (!(remote_block = va_calloc_io_block(0,0,0,remote_block)))
		{
			syslog(LOG_CRIT,"Error va_calloc_io_block() in ga_service_server()");
			return(-1);
		}
				
		/* Check if niches_config[k] == SAME */
		if (!strncmp(conf->niches_config[k],"SAME",4) )
		{
			strcpy(char_temp,"READFILE=SAME\0");
			remote_block = va_insert_io_block(-1,char_temp,remote_block);
		}
		/* Read new config from inet */
		else
		{
			sprintf(char_temp,"READINET=%s\0",conf->niches_config[k]);
			remote_block = va_insert_io_block(-1,char_temp,remote_block);
			
			if (!(remote_config_file = fopen(conf->niches_config[k],"r")))
			{
				syslog(LOG_CRIT,"Error couldn't open %s",conf->niches_config[k]);
				return(-2);
			}
			
			/* Append the content of file to the io_block */
			while (!feof(remote_config_file) )
			{
				fgets(char_temp,BUFFSIZE,remote_config_file);
				remote_block = va_insert_io_block(-1,char_temp,remote_block);
			}
		}
		
		/* Connect to remote servers */
		strcpy(remote_con.ip,conf->niches[k]);
		retry = 0;
		while (1)
		{
			if (va_io_connect(&remote_con) < 0)
			{
				retry += 1;
			}
			else
			{
				break;
			}
			
			if (retry == MAX_RETRY)
			{
				syslog(LOG_ERR,"Error va_io_connect(), couldn't reach %s",remote_con.ip);
				break;
			}
		}
		
		/* Now send the block to the hosts */
		remote_block->connfd = remote_con.connfd;
		
		retry = 0;
		while (1)
		{
			if (va_dwrite_io_block(remote_block) <0 )
			{
				retry += 1;
			}
			else
			{
				break;
			}
			
			if (retry >= MAX_RETRY)
			{
				syslog(LOG_ERR,"Error va_dwrite_io_block() in ga_service_server()");
				break;
			}
		}
		/* Free allocated memory */
		remote_block = (struct io_block *)va_free_io_block(remote_block);
		va_io_close(&remote_con);
	}
}

/********************* HOSTS CODE ***********************/
/* ==================================================== */
/* Check if remote hosts are used and is a master server or master niche server */
if (((conf->master == 1) || (conf->master_niche == 1)) && (conf->num_hosts > 0) )
{
	/* Start remote slaves */
	/***********************/
	for (k=0; k< conf->num_hosts; k++)
	{
		/* Calloc io_block */
		if (!(remote_block = va_calloc_io_block(0,0,0,remote_block)))
		{
			syslog(LOG_CRIT,"Error va_calloc_io_block() in ga_service_server()");
			return(-1);
		}
				
		/* Check if hosts_config[k] == SAME */
		if (!strncmp(conf->hosts_config[k],"SAME",4))
		{
			strcpy(char_temp,"READFILE=SAME\0");
			remote_block = va_insert_io_block(-1,char_temp,remote_block);
		}
		/* Read new config from inet */
		else
		{
			sprintf(char_temp,"READINET=%s\0",conf->hosts_config[k]);
			remote_block = va_insert_io_block(-1,char_temp,remote_block);
			
			if (!(remote_config_file = fopen(conf->hosts_config[k],"r")))
			{
				syslog(LOG_CRIT,"Error couldn't open %s",conf->hosts_config[k]);
				return(-2);
			}
			
			/* Append the content of file to the io_block */
			while (!feof(remote_config_file) )
			{
				fgets(char_temp,BUFFSIZE,remote_config_file);
				remote_block = va_insert_io_block(-1,char_temp,remote_block);
			}
		}
		
		/* Connect to remote servers */
		strcpy(remote_con.ip,conf->hosts[k]);
		retry = 0;
		while (1)
		{
			if (va_io_connect(&remote_con) < 0)
			{
				retry += 1;
			}
			else
			{
				break;
			}
			
			if (retry == MAX_RETRY)
			{
				syslog(LOG_ERR,"Error va_io_connect(), couldn't reach %s",remote_con.ip);
				break;
			}
		}
		
		/* Now send the block to the hosts */
		remote_block->connfd = remote_con.connfd;
		
		retry = 0;
		while (1)
		{
			if (va_dwrite_io_block(remote_block) <0 )
			{
				retry += 1;
			}
			else
			{
				break;
			}
			
			if (retry >= MAX_RETRY)
			{
				syslog(LOG_ERR,"Error va_dwrite_io_block() in ga_service_server()");
				break;
			}
		}
		/* Free allocated memory */
		remote_block = va_free_io_block(remote_block);
		va_io_close(&remote_con);
	}
}

/********************* STARTUP CODE ***********************/
/* ====================================================== */
/* Start local clients */
if ((startup_client_pid = fork()) < 0)
{
	syslog(LOG_CRIT,"Error fork() startup_client_pid");
	return(-10);
}
else if (startup_client_pid == 0) 
{
	setsid();
	syslog(LOG_INFO,"startup_client_pid = %d",getpid());
	/* Set exit_flag to zero */
	exit_flag = 0;
	
	/* Memory reserve for client_info vector */
	if (!(client_info = (struct ga_client_info *)calloc(conf->num_local, 
												sizeof(struct ga_client_info)) ))
	{
		syslog(LOG_CRIT,"Error calloc() client_info");
		return(-1);
	}
	
	/* Memory reserve for parameter vector */
	if (!(char_vector = (char **)calloc(GA_CLIENT_NUM_PARAM, sizeof(char *)) ))
	{
		syslog(LOG_CRIT,"Error calloc() char_vector");
		return(-1);
	}
	for (k = 0; k< GA_CLIENT_NUM_PARAM -1; k++)
	{
		if (!(char_vector[k] = (char *)calloc(BUFFSIZE, sizeof(char)) ))
		{
			syslog(LOG_CRIT,"Error calloc() char_vector[%d]",k);
			for (l = 0; l< k; l++)
			{
				free(char_vector[l]);
			}
			free(char_vector);
			return(-1);
		}
	}
	
	/* Set general parameters for all clients */
	char_vector[GA_CLIENT_NUM_PARAM -1] = NULL;
	strcpy(char_vector[0],"./ga_client\0");
	sprintf(char_vector[2],"server_ip=%s\0",conf->service_ip);	
	sprintf(char_vector[3],"server_port=%d\0",conf->service_port);
	sprintf(char_vector[4],"init_preffix=%s\0",conf->init_preffix);
	sprintf(char_vector[5],"result_preffix=%s\0",conf->result_preffix);
	
	/* Set sigaction for SIGTERM for startup */
	sigemptyset(&sig_term.sa_mask);
	sig_term.sa_handler = ga_service_sig_term;
	sig_term.sa_flags = 0;
	sig_term.sa_flags |= SA_RESTART;
	sig_term.sa_flags |= SA_NOMASK;
	if (sigaction(SIGTERM, &sig_term, NULL) < 0)
	{
		syslog(LOG_CRIT,"Couldn't set SIGTERM handler");
		return(-1);
	}
	
	while (exit_flag == 0)
	{
		/* Spawn clients */
		for (k = 0; k< conf->num_local; k++)
		{
			if (client_info[k].is_running == 0)
			{
				/* Set the client id */
				sprintf(char_vector[1],"client_id=%d\0",k);
		
				if ((client_info[k].pid = fork()) < 0)
				{
					syslog(LOG_ERR,"Couldn't fork for client %d", k);
				}
				else if (client_info[k].pid == 0)  // exec
				{
					syslog(LOG_INFO,"Starting client %d",k);
					execvp(char_vector[0],char_vector);			
				}
				
				client_info[k].is_running = 1;
			}
		}
		
		/* Send sigterm to clients */
		/* Wait for clients termination, so they don't become zombies */
		for (k = 0; k < conf->num_local; k++)
		{
			temp_pid = waitpid(-1, &status, 0);	//wait for any child
		
			/* Check if normally terminated	*/
			if (WIFEXITED(status)) //Normally terminated
			{
				syslog(LOG_INFO,"client %d exited: %d",temp_pid, WEXITSTATUS(status));
				
				/* Set exit flag if exit status == 0 */
				if (WEXITSTATUS(status) == 0)
				{
					exit_flag = 1;
				}
	
				/* Else break to respawn */
				else // Exited with error
				{
					for (l = 0; l< conf->num_local; l++)
					{
						if (temp_pid == client_info[l].pid)
						{
							client_info[l].is_running = 0;
							break;
						}
					}
					break; //Stop waiting 
				}
			}
		}
		
	}
	return(0);
}




/******************** LISTEN CODE ***********************/
/* ==================================================== */
/********************************************************/
/* Check if is a master server or a master niche server */
if ((conf->master == 1) || (conf->master_niche == 1))
{
	/* Allocate client_index before calling listen thread */
	if (!(client_index = (struct ga_service_client_index *)ga_calloc_client_index(
																		0, 0, 0, client_index) ))
	{
		syslog(LOG_CRIT,"Error calloc client_index in service server: %s",strerror(errno) );
		return(-1);
	}																		

	/* Make socket and wait for connections */
	if ( (listenfd = socket(PF_INET, SOCK_STREAM, 0)) < 0)
	{
		syslog(LOG_CRIT,"Error socket() in ga_service_server(): %s",strerror(errno));
		return(-1);
	}
	
	
	bzero(&server_addr,sizeof(server_addr));
	server_addr.sin_family = PF_INET;
	if ((ret=inet_pton(PF_INET, conf->service_ip, &server_addr.sin_addr)) < 1)
	{
		syslog(LOG_CRIT,"Error inet_pton() in ga_service_server() %d: %s",ret,strerror(errno));
		return(-1);
	}
	
	syslog(LOG_INFO,"conf->service_ip=%s, conf->service_port = %d",conf->service_ip,conf->service_port);
	
	server_addr.sin_port = htons((uint16_t)(conf->service_port));
	if ( (bind(listenfd, (struct sockaddr *) &server_addr, sizeof(server_addr)) ) < 0)
	{
		syslog(LOG_CRIT,"Error bind() socket in ga_service_server(): %s",strerror(errno));
		return(-1);
	}
	if ( (listen(listenfd,LISTENQ)) <0 )
	{
		syslog(LOG_CRIT,"Error listen() in ga_service_server(): %s",strerror(errno));
		return(-1);
	}

	syslog(LOG_INFO,"ga_service_server ready to accept connections");

	/* Set initial state for shared threads vars */
	current_generation = 0;
	pat_generation = 0;
	exit_flag = 0;
	num_provider = 0;
	time_wait = conf->time_wait;

	/* Set parameters for listen thread */
	listen_param.conf = conf;
	listen_param.listenfd = &listenfd;

	/* Create listen thread */
	if (pthread_create(&listen_tid, NULL, ga_service_listen_t, (void *)(&listen_param)) )
	{
		syslog(LOG_CRIT,"Error pthread_create() listen thread: %s",strerror(errno));
		return(-1);
	}

	/* Pause / Sleep until a certain time had passed, then start provider */
	while (!exit_flag)
	{
		syslog(LOG_INFO,"ga_sevice_provider_t sleeping for %d seconds",conf->time_wait);
		sleep(conf->time_wait);

		/* Set parameters for provider thread */
		provider_param.conf = conf;

		/* Create provider thread */
		if (pthread_create(&provider_tid, NULL, ga_service_provider_t, (void *)(&provider_param)) )
		{
			syslog(LOG_CRIT,"Error pthread_create() listen thread: %s",strerror(errno));
			return(-1);
		}	
	}
	
	/* FIXME 14/10/2005 */
	/* Send SIGTERM Signal to everybody */
	if ( kill(startup_client_pid, SIGTERM) < 0)
	{
		syslog(LOG_ERR,"Couldn't send SIGTERM signal to process %d_ %s", 
							startup_client_pid, strerror(errno));
	}

}

return(0);
}
