// -*- C++ -*-

/* 
 * Gnome Crystal
 * view.cc 
 *
 * Copyright (C) 2000-2002
 *
 * Developed by Jean Brfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <GL/gl.h>
#include <GL/glu.h>
#include <gtkgl/gtkglarea.h>
#include <math.h>
#include <gnome-xml/parserInternals.h>
#include <gnome-xml/xmlmemory.h>
#include "view.h"
#ifdef HAVE_PNG
#include <png.h>
#endif
#ifdef HAVE_LIBJPEG
extern "C" {
#include <jpeglib.h>
}
#endif
#include "document.h"
#include "globals.h"
#include "print.h"

guint FoV;
gdouble Psi, Theta, Phi;
gdouble Red, Green, Blue;

typedef struct
{
	unsigned glList;
} WidgetData;

/* Attribute list for gtkglarea widget. Specifies a
     list of Boolean attributes and enum/integer
     attribute/value pairs. The last attribute must be
     GDK_GL_NONE. See glXChooseVisual manpage for further
     explanation.
*/
static int attrlist[] = {
    GDK_GL_RGBA,
    GDK_GL_BUFFER_SIZE,1,
    GDK_GL_RED_SIZE,1,
    GDK_GL_GREEN_SIZE,1,
    GDK_GL_BLUE_SIZE,1,
    GDK_GL_DEPTH_SIZE,1,
    GDK_GL_DOUBLEBUFFER,
    GDK_GL_NONE
  };
extern GtkWidget *vbox1;

extern "C" {
	
	gint on_init(GtkWidget *widget, void *data) 
	{
		((gcView*)data)->Init(widget);
		return TRUE;
	}
	
	gint on_reshape(GtkWidget *widget, GdkEventConfigure *event, void *data) 
	{
		((gcView*)data)->Reshape(widget);
		return TRUE;
	}
	
	gint on_draw(GtkWidget *widget, GdkEventExpose *event, void *data) 
	{
		/* Draw only last expose. */
		if (event->count > 0) return TRUE;

		((gcView*)data)->Draw(widget);
		return TRUE;
	}

	gint on_motion(GtkWidget *widget, GdkEventMotion *event, void *data) 
	{
		((gcView*)data)->OnMotion(widget, event);
		return TRUE;
	}

	gint on_pressed(GtkWidget *widget, GdkEventButton *event, void *data) 
	{
		return ((gcView*)data)->OnPressed(widget, event);
	}

}//end of extern "C" section

static void on_destroyed(GtkWidget *widget, gcView *pView)
{
	pView->OnDestroyed(widget);
}

gcView::gcView(gcDocument *pDoc)
{
	m_pDoc = pDoc;
	m_fAngle = FoV;
	SetRotation(Psi, Theta, Phi);
	m_nGLList = 0;
	m_fBlue = Blue;
	m_fRed = Red;
	m_fGreen = Green;
	m_fAlpha = 1.0;
	
	m_bInit = false;
	m_bLocked = false;
}

gcView::gcView(gcView*pView)
{
	m_pDoc = pView->m_pDoc;
	m_fAngle = pView->m_fAngle;
	SetRotation(pView->m_psi, pView->m_theta, pView->m_phi);
	m_nGLList = 0;
	m_fBlue = pView->m_fBlue;
	m_fRed = pView->m_fRed;
	m_fGreen = pView->m_fGreen;
	m_fAlpha = pView->m_fAlpha;
	m_pDoc->AddView(this);

	m_bInit = false;
	m_bLocked = false;
}

gcView::~gcView()
{
	if (!IsEmbedded())
	{
		gtk_widget_destroy(GTK_WIDGET(m_pMenu));
	}
	gcDialog *dialog;
	while (!m_Dialogs.empty())
	{
		dialog = m_Dialogs.front();
		m_Dialogs.pop_front();
		dialog->Destroy();
	}
}

void gcView::SetDocument(gcDocument *pDoc)
{
	m_pDoc = pDoc;
}

void gcView::Init(GtkWidget *widget)
{
	if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
	{
	    	glEnable(GL_LIGHTING);
		glEnable(GL_LIGHT0);
		glEnable(GL_DEPTH_TEST);
		glEnable(GL_CULL_FACE);
		glEnable(GL_COLOR_MATERIAL);
		float shiny = 25.0, spec[4] = {1.0, 1.0, 1.0, 1.0};
		glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &shiny);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, spec);
		Update(widget);
		m_bInit = true;
    }
}

void gcView::Reshape(GtkWidget *widget)
{
	float fAspect;
	if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
    {
		if (widget->allocation.height)
		{
			fAspect = (GLfloat)widget->allocation.width/(GLfloat)widget->allocation.height;
			if (fAspect == 0.0) fAspect = 1.0;
		}
		else	// don't divide by zero, not that we should ever run into that...
			fAspect = 1.0f;
		double x = m_pDoc->GetMaxDist();
		if (x == 0) x = 1;
		m_fRadius = (float) (x / sin(m_fAngle / 180 * 1.570796326794897)) ;
		glViewport(0,0, widget->allocation.width, widget->allocation.height);
		if (fAspect > 1.0)
		{
			m_height = x * (1 - tan(m_fAngle / 180 * 1.570796326794897));
			m_width = m_height * fAspect;
		}
		else
		{
			m_width = x * (1 - tan(m_fAngle / 180 * 1.570796326794897));
			m_height = m_width / fAspect;
		}
		m_near = m_fRadius - x;
		m_far = m_fRadius + x;
	    glMatrixMode(GL_PROJECTION);
	    glLoadIdentity();
		glFrustum(- m_width, m_width, - m_height, m_height, m_near , m_far);
	    glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		glTranslatef(0, 0, -m_fRadius);
	}
}

void gcView::Draw(GtkWidget *widget)
{
	if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
    {
		glClearColor(m_fRed, m_fGreen, m_fBlue, m_fAlpha);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		WidgetData* pData = (WidgetData*)gtk_object_get_data(GTK_OBJECT(widget), "gldata");
		m_nGLList = pData->glList;
		if  (m_nGLList)
		{
			glPushMatrix();
			glRotated(m_psi, 0.0, 1.0, 0.0);
			glRotated(m_theta, 0.0, 0.0, 1.0);
			glRotated(m_phi, 0.0, 1.0, 0.0);
			glCallList(m_nGLList);
			glPopMatrix();
		}
	/* Swap backbuffer to front */
		gtk_gl_area_swapbuffers(GTK_GL_AREA(widget));
    }
}

void gcView::Update()
{
	std::list<GtkWidget*>::iterator i;
	for (i = m_Widgets.begin(); i!= m_Widgets.end(); i++) Update(*i);
}

void gcView::Update(GtkWidget* widget)
{
	Reshape(widget);
	if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
 	{
		WidgetData* pData = (WidgetData*)gtk_object_get_data(GTK_OBJECT(widget), "gldata");
		m_nGLList = pData->glList;
		if (m_nGLList) glDeleteLists(m_nGLList,1);
		pData->glList = m_nGLList = glGenLists(1);
		glNewList(m_nGLList, GL_COMPILE);
		m_pDoc->Draw();
		glEndList();
	}
	Draw(widget);
}

bool gcView::OnPressed(GtkWidget *widget, GdkEventButton *event)
{
  if (event->button == 1) {
    // beginning of drag, reset mouse position
    m_lastx = event->x;
    m_lasty = event->y;
    return true;
  }
  return false;
}

void gcView::OnMotion(GtkWidget *widget, GdkEventMotion *event)
{
	gint x, y;
	GdkModifierType state;
	GdkRectangle area;
	area.x = 0;
	area.y = 0;
	area.width  = widget->allocation.width;
	area.height = widget->allocation.height;
	
	if (event->is_hint)
		gdk_window_get_pointer(event->window, &x, &y, &state);
	else
	{
	    x = (gint) event->x;
	    y = (gint) event->y;
	    state = (GdkModifierType) event->state;
	}
	if (state & GDK_BUTTON1_MASK)
	{
		if ((x == m_lastx) && (y == m_lasty)) return;
			GetDocument()->SetDirty();
		Rotate(x - m_lastx, y - m_lasty);
		m_lastx = x;
		m_lasty = y;
		std::list<GtkWidget*>::iterator i;
		for (i = m_Widgets.begin(); i!= m_Widgets.end(); i++)
		{
			gtk_widget_draw(*i, &area);
		}
	}
}

void gcView::Rotate(gdouble x, gdouble y)
{
	gdouble z = sqrt(x*x + y*y);
	gcMatrix Mat(0, (y > 0) ? - acos(x/z) : acos(x/z), z * 0.00349065850398866, rotation);
	m_Euler = Mat * m_Euler;
	m_Euler.Euler(m_psi, m_theta, m_phi);
	m_psi /= 0.0174532925199433;
	m_theta /= 0.0174532925199433;
	m_phi /= 0.0174532925199433;
}

void gcView::SetBackgroundColor(float red, float green, float blue, float alpha)
{
	m_fRed = red ;
	m_fGreen = green ;
	m_fBlue = blue ;
	m_fAlpha = alpha ;
}

void gcView::GetBackgroundColor(double *red, double *green, double *blue, double *alpha)
{
	*red = m_fRed ;
	*green = m_fGreen ;
	*blue = m_fBlue ;
	*alpha = m_fAlpha ;
}

void gcView::GetRotation(double *psi, double *theta, double *phi)
{
	*psi = m_psi;
	*theta = m_theta;
	*phi = m_phi;
}

void gcView::SetRotation(double psi, double theta, double phi)
{
	m_psi = psi;
	m_theta = theta;
	m_phi = phi;
	gcMatrix m(m_psi/90*1.570796326794897, m_theta/90*1.570796326794897, m_phi/90*1.570796326794897, euler);
	m_Euler = m;
}

xmlNodePtr gcView::Save(xmlDocPtr xml)
{
	xmlNodePtr parent, child;
	gchar buf[256];
	parent = xmlNewDocNode(xml, NULL, (xmlChar*)"view", NULL);
	if (!parent) return NULL;
	
	g_snprintf(buf, sizeof(buf) - 1, "%g %g %g", m_psi, m_theta, m_phi);
	child = xmlNewDocNode(xml, NULL, (xmlChar*)"orientation", (xmlChar*)buf);
	if (child) xmlAddChild(parent, child);
	else {xmlFreeNode(parent); return NULL;}
	
	g_snprintf(buf, sizeof(buf) - 1, "%g", m_fAngle);
	child = xmlNewDocNode(xml, NULL, (xmlChar*)"fov", (xmlChar*)buf);
	if (child) xmlAddChild(parent, child);
	else {xmlFreeNode(parent); return NULL;}
	
	g_snprintf(buf, sizeof(buf) - 1, "%g %g %g %g", m_fBlue, m_fRed, m_fGreen, m_fAlpha);
	child = xmlNewDocNode(xml, NULL, (xmlChar*)"background", (xmlChar*)buf);
	if (child) xmlAddChild(parent, child);
	else {xmlFreeNode(parent); return NULL;}
	
	return parent;
}

bool gcView::Load(xmlNodePtr node)
{
	char *txt;
	xmlNodePtr child = node->childs;
	while(child)
	{
		if (!strcmp((gchar*)child->name, "orientation"))
		{
			txt = (char*)xmlNodeGetContent(child);
			sscanf(txt, "%lg %lg %lg", &m_psi, &m_theta, &m_phi);
			gcMatrix m(m_psi/90*1.570796326794897, m_theta/90*1.570796326794897, m_phi/90*1.570796326794897, euler);
			m_Euler = m;
		}
		else if (!strcmp((gchar*)child->name, "background"))
		{
			txt = (char*)xmlNodeGetContent(child);
			sscanf(txt, "%g %g %g %g", &m_fBlue, &m_fRed, &m_fGreen, &m_fAlpha);
		}
		else if (!strcmp((gchar*)child->name, "fov"))
		{
			txt = (char*)xmlNodeGetContent(child);
			int result = sscanf(txt, "%lg", &m_fAngle);
		}
		child = child->next;
	}
	return true;
}

#ifdef HAVE_PNG
void gcView::ExportPNG(const gchar* filename)
{
	FILE *fp;
	png_structp png_ptr;
	png_infop info_ptr;
	unsigned char *tmp = NULL, **lines = NULL;
	fp = fopen(filename, "wb");
	try
	{
		if (!fp) throw (int) 0;
		if (!(png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL)))
			throw (int) 1;
		if (!(info_ptr = png_create_info_struct(png_ptr))) throw (int) 2;
		if (setjmp(png_jmpbuf(png_ptr))) throw (int) 3;
		png_init_io(png_ptr, fp);
		png_set_compression_level(png_ptr, Z_BEST_COMPRESSION);
		png_set_IHDR(png_ptr, info_ptr, m_pWidget->allocation.width, m_pWidget->allocation.height,
						8, PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT,
						PNG_FILTER_TYPE_DEFAULT);
		png_write_info(png_ptr, info_ptr);
		unsigned LineWidth, n = sizeof(int);
		if (m_pWidget->allocation.width & (n - 1))
		{
			LineWidth = ((~(n - 1)) & (m_pWidget->allocation.width * 3)) + n;
		}
		else LineWidth = m_pWidget->allocation.width * 3;
		unsigned size = LineWidth * m_pWidget->allocation.height;
		tmp = new unsigned char[size];
		if (!tmp) throw (int) 4;
		
		if (gtk_gl_area_make_current(GTK_GL_AREA(m_pWidget)))
		{
			glPixelStorei(GL_PACK_ALIGNMENT, n);
			glReadBuffer(GL_BACK_LEFT);
			glReadPixels(0, 0, m_pWidget->allocation.width, m_pWidget->allocation.height , GL_RGB,
							GL_UNSIGNED_BYTE, tmp);
		}
		else throw (int) 6;
		lines = new (unsigned char*)[m_pWidget->allocation.height];
		if (!lines) throw (int) 5;
		for (int i = 0, j = m_pWidget->allocation.height - 1; j >= 0; i++, j--)
		{
			lines[i] = tmp + LineWidth * j;
		}
		png_write_image(png_ptr, lines);
		png_write_end(png_ptr, info_ptr);
		delete [] tmp;
		delete [] lines;
		png_destroy_write_struct(&png_ptr, &info_ptr);
		fclose(fp);
	}
	catch (int num)
	{
		if (tmp) delete [] tmp;
		if (lines) delete [] lines;
		if (num > 0) fclose(fp);
		switch(num)
		{
		case 0: break;
		case 1: break;
		case 2:
			png_destroy_write_struct(&png_ptr, NULL);
			break;
		default:
			png_destroy_write_struct(&png_ptr, &info_ptr);
			break;
		}
	}
}
#endif

#ifdef HAVE_LIBJPEG
void gcView::ExportJPG(const gchar* filename)
{
	FILE *fp;
	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;
	unsigned char *tmp = NULL, **lines = NULL;
	fp = fopen(filename, "wb");
	try
	{
		if (!fp) throw (int) 0;
		cinfo.err = jpeg_std_error(&jerr);
		jpeg_create_compress(&cinfo);
		jpeg_stdio_dest(&cinfo, fp);
		cinfo.image_width = m_pWidget->allocation.width;
		cinfo.image_height = m_pWidget->allocation.height;
		cinfo.input_components = 3;
		cinfo.in_color_space = JCS_RGB;
		jpeg_set_defaults(&cinfo);
		unsigned LineWidth, n = sizeof(int);
		if (m_pWidget->allocation.width & (n - 1))
		{
			LineWidth = ((~(n - 1)) & (m_pWidget->allocation.width * 3)) + n;
		}
		else LineWidth = m_pWidget->allocation.width * 3;
		unsigned size = LineWidth * m_pWidget->allocation.height;
		tmp = new unsigned char[size];
		if (!tmp) throw (int) 2;
		if (gtk_gl_area_make_current(GTK_GL_AREA(m_pWidget)))
		{
			glPixelStorei(GL_PACK_ALIGNMENT, n);
			glReadBuffer(GL_BACK_LEFT);
			glReadPixels(0, 0, m_pWidget->allocation.width, m_pWidget->allocation.height , GL_RGB,
							GL_UNSIGNED_BYTE, tmp);
		}
		else throw (int) 4;
		lines = new (unsigned char*)[m_pWidget->allocation.height];
		if (!lines) throw (int) 3;
		for (int i = 0, j = m_pWidget->allocation.height - 1; j >= 0; i++, j--)
		{
			lines[i] = tmp + LineWidth * j;
		}
		jpeg_start_compress(&cinfo, true);
		jpeg_write_scanlines(&cinfo, lines, m_pWidget->allocation.height);
		jpeg_finish_compress(&cinfo);
		delete [] tmp;
		delete [] lines;
		fclose(fp);
		jpeg_destroy_compress(&cinfo);
	}
	catch (int num)
	{
		if (tmp) delete [] tmp;
		if (lines) delete [] lines;
		if (num > 0) fclose(fp);
		switch(num)
		{
		case 0: break;
		case 1: break;
		default:
			jpeg_destroy_compress(&cinfo);
			break;
		}
	}
}
#endif

void gcView::Print(GnomePrintContext *pc, gdouble width, gdouble height)
{
	double hstep, vstep, dxStep, dyStep;
	hstep = m_width;
	vstep = m_height;
	dxStep = m_near;
	dyStep = m_far;
	gcPrintWindow* box = new gcPrintWindow();
	while (!box->IsReady())
		if (gtk_events_pending()) gtk_main_iteration();
	GtkWidget* widget = box->GetWidget();
	WidgetData* pData = new WidgetData;
	pData->glList = 0;
	gtk_object_set_data(GTK_OBJECT(widget), "gldata", pData);
	Update(widget);
	m_width = hstep;
	m_height = vstep;
	m_near = dxStep;
	m_far = dyStep;
	unsigned char *tmp;
	unsigned LineWidth, s = sizeof(int);
	if (widget->allocation.width & (s - 1))
	{
		LineWidth = ((~(s - 1)) & (widget->allocation.width * 3)) + s;
	}
	else LineWidth = widget->allocation.width * 3;
	unsigned size = LineWidth * widget->allocation.height;
	int i, j;
	//hstep, vstep: size of the printing window in gnome-print units
	hstep = (double) widget->allocation.width *72 / PrintResolution;
	vstep = (double) widget->allocation.height *72 / PrintResolution;
	tmp = new unsigned char[size];
	if (!tmp) return;
	double Width, Height;//size of print area
	int n, m, imax, jmax;
	if (IsEmbedded())
	{
		Width = width;
		Height = height;
	}
	else
	{
		Width = m_pWidget->allocation.width;
		Height = m_pWidget->allocation.height;
	}
	imax = int(Width / hstep);
	jmax = int(Height / vstep);
	dxStep = hstep / Width * 2; 
	dyStep = vstep / Height *2;
	double matrix [6] = {hstep, 0, 0, - vstep,
						0, (height + Height )/ 2};
	for (j = 0; j <= jmax; j++)
	{
		matrix[4] = (width - Width)/ 2;
		matrix[3] = (j < jmax) ? - vstep : - Height + vstep * jmax;
		for (i = 0; i <= imax; i++)
		{
			if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
			{
				glMatrixMode(GL_PROJECTION);
				glLoadIdentity();
				glFrustum(m_width * ( -1 + i * dxStep), m_width * ( -1 + (i + 1)* dxStep),
							m_height * ( 1 - (j + 1)* dyStep), m_height * ( 1 - j* dyStep), m_near , m_far);
				glMatrixMode(GL_MODELVIEW);
				glLoadIdentity();
				glTranslatef(0, 0, -m_fRadius);
				glClearColor(m_fRed, m_fGreen, m_fBlue, m_fAlpha);
				glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
				if  (pData->glList)
				{
					glPushMatrix();
					glRotated(m_psi, 0.0, 1.0, 0.0);
					glRotated(m_theta, 0.0, 0.0, 1.0);
					glRotated(m_phi, 0.0, 1.0, 0.0);
					glCallList(pData->glList);
					glPopMatrix();
				}
				glPixelStorei(GL_PACK_ALIGNMENT, s);
				glReadBuffer(GL_BACK_LEFT);
				glReadPixels(0, 0, widget->allocation.width, widget->allocation.height, GL_RGB,
									GL_UNSIGNED_BYTE, tmp);
			}
			gnome_print_gsave(pc);
			matrix[0] = (i < imax) ? hstep : Width - imax * hstep;
			gnome_print_concat (pc, matrix);
			m = (i < imax) ? widget->allocation.width : (int(Width) * PrintResolution / 72)  % widget->allocation.width;
			n = (j < jmax) ? widget->allocation.height: (int(Height) * PrintResolution / 72) % widget->allocation.height;
			gnome_print_rgbimage(pc, (char *) tmp + (widget->allocation.height - n) * LineWidth, m, n, LineWidth);
			gnome_print_grestore(pc);
			matrix[4] += hstep;
		}
		matrix[5] -= vstep;
	}
	delete pData;
	delete box;
}

GtkWidget* gcView::CreateNewWidget()
{
/* Create new OpenGL widget. */
	m_pWidget = GTK_WIDGET(gtk_gl_area_new(attrlist));
	WidgetData* pData = new WidgetData;
	pData->glList = 0;
	gtk_object_set_data(GTK_OBJECT(m_pWidget), "gldata", pData);
	m_Widgets.push_back(m_pWidget);
// Events for widget must be set before X Window is created
	gtk_widget_set_events(GTK_WIDGET(m_pWidget),
			GDK_EXPOSURE_MASK|
			GDK_BUTTON_MOTION_MASK|
			GDK_POINTER_MOTION_HINT_MASK|
			GDK_BUTTON_PRESS_MASK);

// Connect signal handlers
// Do initialization when widget has been realized.
	gtk_signal_connect(GTK_OBJECT(m_pWidget), "realize",
		     GTK_SIGNAL_FUNC(on_init), this);
// When window is resized viewport needs to be resized also.
	gtk_signal_connect(GTK_OBJECT(m_pWidget), "configure_event",
		     GTK_SIGNAL_FUNC(on_reshape), this);
// Redraw image when exposed. 
	gtk_signal_connect(GTK_OBJECT(m_pWidget), "expose_event",
		     GTK_SIGNAL_FUNC(on_draw), this);
// When moving mouse 
  gtk_signal_connect (GTK_OBJECT(m_pWidget), "motion_notify_event",
		      GTK_SIGNAL_FUNC(on_motion), this);
// When a mouse button is pressed
  gtk_signal_connect (GTK_OBJECT(m_pWidget), "button_press_event",
		      GTK_SIGNAL_FUNC(on_pressed), this);
// When a widget is destroyed
	gtk_signal_connect (GTK_OBJECT(m_pWidget), "destroy", GTK_SIGNAL_FUNC(on_destroyed), this);

/* set minimum size */
	gtk_widget_set_usize(GTK_WIDGET(m_pWidget), 100,100);
		    
/* put glarea into window and show it all */

	gtk_widget_show(GTK_WIDGET(m_pWidget));
//	if (IsEmbedded()) return m_pWidget;
//	gtk_container_add(GTK_CONTAINER(vbox1),GTK_WIDGET(m_pWidget));
//	gtk_widget_show(GTK_WIDGET(vbox1));
//	return vbox1;
	return m_pWidget;
}

void gcView::OnDestroyed(GtkWidget *widget)
{
	delete (WidgetData*) gtk_object_get_data(GTK_OBJECT(widget), "gldata");
	m_Widgets.remove(widget);
}

void gcView::SetMenu(GtkMenuItem* item)
{
	m_pMenu = item;
	m_pMenuLabel = NULL;
	GList* l;
	for (l = gtk_container_children(GTK_CONTAINER(item)); l != NULL; l = g_list_next(l))
		if (GTK_IS_LABEL(l->data))
		{
			m_pMenuLabel = (GtkLabel*)(l->data);
			break;
		}
}

void gcView::NotifyDialog(gcDialog* dialog)
{
	m_Dialogs.push_front(dialog);
}

void gcView::RemoveDialog(gcDialog* dialog)
{
	m_Dialogs.remove(dialog);
}
