// -*- C++ -*-

/* 
 * Gnome Crystal
 * gcrystal-bonobo.cc 
 *
 * Copyright (C) 2001-2005
 *
 * Developed by Jean Brfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */
#include "config.h"
#include "gcrystal-bonobo.h"
#include "view.h"
#include "view-settings.h"
#include "filesel.h"
#include "prefs.h"
#include "globals.h"
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprintui/gnome-print-dialog.h>
#include <libgnomeprintui/gnome-print-job-preview.h>

/*******************************************************************************
 * Persist Stream implementation
 ******************************************************************************/
#define PERSIST_STREAM_TYPE         (persist_stream_get_type ())
#define PERSIST_STREAM(o)           (G_TYPE_CHECK_INSTANCE_CAST ((o), PERSIST_STREAM_TYPE, PersistStream))
#define PERSIST_STREAM_CLASS(k)     (G_TYPE_CHECK_CLASS_CAST((k), PERSIST_STREAM_TYPE, PersistStreamClass))
#define PERSIST_STREAM_IS_OBJECT(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), PERSIST_STREAM_TYPE))
#define PERSIST_STREAM_IS_CLASS(k)  (G_TYPE_CHECK_CLASS_TYPE ((k), PERSIST_STREAM_TYPE))
#define PERSIST_STREAM_GET_CLASS(o) (G_TYPE_INSTANCE_GET_CLASS ((o), PERSIST_STREAM_TYPE, PersistStreamClass))

typedef struct {
	BonoboPersist parent;
	gcDocument* pDoc;
} PersistStream;

typedef struct {
	BonoboPersistClass parent_class;

	POA_Bonobo_PersistStream__epv epv;
} PersistStreamClass;

GType persist_stream_get_type (void);

static void on_load(gpointer p, Bonobo_Stream stream, const Bonobo_Persist_ContentType type, CORBA_Environment  *ev)
{
	PersistStream *ps = (PersistStream*)bonobo_object(p);
	if (!ps->pDoc) return;
	CORBA_long l = Bonobo_Stream_seek(stream, 0, Bonobo_Stream_SeekEnd, ev);
//	CORBA_long l = bonobo_stream_client_get_length (stream, ev);//Seeme it does not always work!
	Bonobo_Stream_seek(stream, 0, Bonobo_Stream_SeekSet, ev);
	Bonobo_Stream_iobuf*  buffer;
	Bonobo_Stream_read(stream, l, &buffer, ev);
	char buf[l+1];
	memcpy(buf, buffer->_buffer, l);
	buf[l] = 0;
	xmlDocPtr xml = xmlParseMemory(buf, l);
	CORBA_free (buffer);
	if (xml) ps->pDoc->ParseXMLTree(xml->children);
}

static void on_save(gpointer p, Bonobo_Stream stream, const Bonobo_Persist_ContentType type, CORBA_Environment  *ev)
{
	PersistStream *ps = (PersistStream*)bonobo_object(p);
	if (!ps->pDoc) return;
	xmlDocPtr xml = ps->pDoc->BuildXMLTree();
	if (xml)
	{
		char* tmp;
		int size;
		xmlDocDumpMemory(xml, (xmlChar**)(&tmp), &size);
		if (tmp)
		{
			bonobo_stream_client_write (stream, tmp, size, ev);
			xmlFree((xmlChar*)(tmp));
		}
		xmlFreeDoc(xml);
	}
}

static PersistStream* persist_stream_new()
{
	PersistStream* ps = (PersistStream*) g_object_new (PERSIST_STREAM_TYPE, NULL);
	return ps;
}

static void persist_stream_init(PersistStream *ps)
{
}

static void persist_stream_class_init(PersistStreamClass *klass)
{
	POA_Bonobo_PersistStream__epv* epv = &(klass->epv);
	epv->load = (void(*)(void*, CORBA_Object_type*, const CORBA_char*, CORBA_Environment*)) on_load;
	epv->save = (void(*)(void*, CORBA_Object_type*, const CORBA_char*, CORBA_Environment*)) on_save;
}

 BONOBO_TYPE_FUNC_FULL (
	PersistStream,                /* Glib class name */
	Bonobo_PersistStream,  /* CORBA interface name */
	BONOBO_TYPE_PERSIST,  /* parent type */
	persist_stream);               /* local prefix ie. 'echo'_class_init */

#define GC_BONOBO_CONTROL_TYPE           (gc_bonobo_control_get_type ())
#define GC_BONOBO_CONTROL(o)             (G_TYPE_CHECK_INSTANCE_CAST ((o), GC_BONOBO_CONTROL_TYPE, GCBonoboControl))
#define GC_BONOBO_CONTROL_CLASS(k)       (G_TYPE_CHECK_CLASS_CAST((k), GC_BONOBO_CONTROL_TYPE, GCBonoboControlClass))

#define GC_BONOBO_IS_CONTROL(o)          (G_TYPE_CHECK_INSTANCE_TYPE ((o), GC_BONOBO_CONTROL_TYPE))
#define GC_BONOBO_IS_CONTROL_CLASS(k)    (G_TYPE_CHECK_CLASS_TYPE ((k), GC_BONOBO_CONTROL_TYPE))
#define GC_BONOBO_CONTROL_GET_CLASS(o)   (G_TYPE_INSTANCE_GET_CLASS ((o), GC_BONOBO_CONTROL_TYPE, GCBonoboControlClass))

typedef struct _GCBonoboControl         GCBonoboControl;
typedef struct _GCBonoboControlClass    GCBonoboControlClass;

struct _GCBonoboControl {
	BonoboControl control;
	BonoboUIComponent *uic;
	PersistStream *ps;
};

struct _GCBonoboControlClass {
	BonoboControlClass parent_class;
};

static GType gc_bonobo_control_get_type(void);
static GCBonoboControl *gc_bonobo_control_new(gcDocument *pDoc);
static GCBonoboControl *gc_bonobo_control_construct(GCBonoboControl *control, gcDocument *pDoc);

static GObjectClass *gc_bonobo_control_parent_class;

static void
gc_bonobo_control_destroy (BonoboObject *object)
{
	GCBonoboControl *control;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GC_BONOBO_IS_CONTROL (object));

	control = GC_BONOBO_CONTROL (object);
	RemoveDocument(control->ps->pDoc);

	BONOBO_OBJECT_CLASS (gc_bonobo_control_parent_class)->destroy (object);
}

static void
gc_bonobo_control_finalize (GObject *object)
{
	g_return_if_fail (object != NULL);
	g_return_if_fail (GC_BONOBO_IS_CONTROL (object));
	
	G_OBJECT_CLASS (gc_bonobo_control_parent_class)->finalize (object);
}

void do_save_as(const gchar* filename, gcView* pView);

void on_save_as_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	new gcFileSel(_("Save model as..."), do_save_as, true, ".gcrystal", (gcView*)control->ps->pDoc->GetView());
}

void on_print_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	GnomePrintConfig* config = gnome_print_config_default();
	GnomePrintContext *pc;
	GnomePrintJob *gpj = gnome_print_job_new(config);
	int do_preview = 0, copies = 1, collate = 0;
	GnomePrintDialog *gpd;
	gpd = GNOME_PRINT_DIALOG (gnome_print_dialog_new(gpj, (const guchar*)"Print test", GNOME_PRINT_DIALOG_COPIES));
	gnome_print_dialog_set_copies(gpd, copies, collate);
	switch (gtk_dialog_run(GTK_DIALOG(gpd)))
	{
	case GNOME_PRINT_DIALOG_RESPONSE_PRINT:
		do_preview = 0;
		break;
	case GNOME_PRINT_DIALOG_RESPONSE_PREVIEW:
		do_preview = 1;
		break;
	case GNOME_PRINT_DIALOG_RESPONSE_CANCEL:
		gtk_widget_destroy (GTK_WIDGET(gpd));
		return;
	}
	gtk_widget_destroy(GTK_WIDGET(gpd));
	pc = gnome_print_job_get_context (gpj);
	gnome_print_beginpage(pc, (const guchar*)"");
	gdouble width, height;
	gnome_print_job_get_page_size_from_config(config, &width, &height);
	((gcView*)control->ps->pDoc->GetView())->Print(pc, width, height);
	gnome_print_showpage(pc);
	g_object_unref(pc);
	gnome_print_job_close(gpj);
	if (do_preview)
	{
		gtk_widget_show (gnome_print_job_preview_new (gpj, (const guchar*)_("Preview")));
	}
	else
	{
		gnome_print_job_print(gpj);
	}
	g_object_unref (gpj);
	gnome_print_config_unref(config);
}

void do_export_vrml(const gchar* filename, gcView* pView);

void on_export_vrml_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	new gcFileSel(_("Export model as a VRML file..."), do_export_vrml, true, ".wrl", (gcView*)control->ps->pDoc->GetView());
}

void do_export_vrml(const gchar* filename, gcView* pView);

#ifdef HAVE_LIBPNG
void do_export_png(const gchar* filename, gcView* pView);
#endif

void on_export_png_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
#ifdef HAVE_LIBPNG
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	new gcFileSel(_("Export model as a PNG file..."), do_export_png, true, ".png", (gcView*)control->ps->pDoc->GetView());
#endif
}

#ifdef HAVE_LIBJPEG
void do_export_jpeg(const gchar* filename, gcView* pView);
#endif

void on_export_jpeg_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
#ifdef HAVE_LIBJPEG
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	new gcFileSel(_("Export model as a Jpeg file..."), do_export_jpeg, true, ".jpg", (gcView*)control->ps->pDoc->GetView());
#endif
}

void on_preferences_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	new gcPrefsDlg();
}

void on_lattice_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	control->ps->pDoc->Define(0);
}

void on_atoms_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	control->ps->pDoc->Define(1);
}

void on_lines_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	control->ps->pDoc->Define(2);
}

void on_size_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	control->ps->pDoc->Define(3);
}

void on_cleavages_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	control->ps->pDoc->Define(4);
}

void on_view_settings_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	g_return_if_fail (GC_BONOBO_IS_CONTROL (user_data));
	GCBonoboControl* control = GC_BONOBO_CONTROL (user_data);
	new gcViewSettingsDlg((gcView*)control->ps->pDoc->GetView());
}

void on_about();

void on_about_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	on_about();
}


static BonoboUIVerb gc_verbs[] = {
	BONOBO_UI_VERB ("SaveAs", on_save_as_cb),
	BONOBO_UI_VERB ("Print", on_print_cb),
	BONOBO_UI_VERB ("Vrml", on_export_vrml_cb),
	BONOBO_UI_VERB ("Png", on_export_png_cb),
	BONOBO_UI_VERB ("Jpeg", on_export_jpeg_cb),
	BONOBO_UI_VERB ("Preferences", on_preferences_cb),
	BONOBO_UI_VERB ("Lattice", on_lattice_cb),
	BONOBO_UI_VERB ("Atoms", on_atoms_cb),
	BONOBO_UI_VERB ("Lines", on_lines_cb),
	BONOBO_UI_VERB ("Size", on_size_cb),
	BONOBO_UI_VERB ("Cleavages", on_cleavages_cb),
	BONOBO_UI_VERB ("ViewSettings", on_view_settings_cb),
	BONOBO_UI_VERB ("HelpAbout", on_about_cb),
	BONOBO_UI_VERB_END
};

static void
gc_bonobo_control_activate (BonoboControl *object, gboolean state)
{
	GCBonoboControl *control;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GC_BONOBO_IS_CONTROL (object));

	control = GC_BONOBO_CONTROL (object);

	if (state) {
		Bonobo_UIContainer ui_container;

		ui_container = bonobo_control_get_remote_ui_container (BONOBO_CONTROL (control), NULL);
		if (ui_container != CORBA_OBJECT_NIL) {
			bonobo_ui_component_set_container (control->uic, ui_container, NULL);
			bonobo_ui_util_set_ui(control->uic, PREFIX,
								"gcrystal.xml", "gcrystal", NULL);
			bonobo_ui_component_add_verb_list_with_data(control->uic, gc_verbs, control);
			if (bonobo_ui_component_get_container(control->uic) != CORBA_OBJECT_NIL)
			{
				bonobo_ui_component_set_prop(control->uic, "/menu/Help/BuiltMenuItems/Helpgcrystal0", "label", _("Gnome Crystal help contents"), NULL);
				bonobo_ui_component_set_prop(control->uic, "/menu/Help/BuiltMenuItems/Helpgcrystal0", "tip", _("View help for Gnome Crystal"), NULL);
			}
		}
	}
	else
		bonobo_ui_component_unset_container (control->uic, NULL);

	if (BONOBO_CONTROL_CLASS (gc_bonobo_control_parent_class)->activate)
		BONOBO_CONTROL_CLASS (gc_bonobo_control_parent_class)->activate (object, state);
}

void gc_bonobo_control_class_init (GCBonoboControlClass *klass)
{
	GObjectClass *gobject_class = (GObjectClass *)klass;
	BonoboObjectClass *bonobo_object_class = (BonoboObjectClass *)klass;
	BonoboControlClass *control_class = (BonoboControlClass *)klass;

	gc_bonobo_control_parent_class = (GObjectClass *)g_type_class_peek_parent ((GObjectClass *)klass);

	bonobo_object_class->destroy = gc_bonobo_control_destroy;
	gobject_class->finalize = gc_bonobo_control_finalize;
	control_class->activate = gc_bonobo_control_activate;
}

static void
gc_bonobo_control_init (GCBonoboControl *control)
{
}

BONOBO_TYPE_FUNC (GCBonoboControl, BONOBO_TYPE_CONTROL, gc_bonobo_control);

GCBonoboControl* gc_bonobo_control_construct(GCBonoboControl *control, gcDocument *pDoc)
{
	GtkWidget             *widget;
	
	g_return_val_if_fail (pDoc != NULL, NULL);
	g_return_val_if_fail (control != NULL, NULL);
	g_return_val_if_fail (GC_BONOBO_IS_CONTROL (control), NULL);

	widget = pDoc->GetView()->CreateNewWidget();

	bonobo_control_construct (BONOBO_CONTROL (control), widget);
	
	control->ps = persist_stream_new();
	control->ps->pDoc = pDoc;
	pDoc->SetBonoboPersist((BonoboPersist*)control->ps);
	bonobo_object_add_interface((BonoboObject*)control, (BonoboObject*)control->ps);

	bonobo_control_set_automerge((BonoboControl*)control, true);
	control->uic = bonobo_control_get_ui_component((BonoboControl*)control);
	g_signal_connect (control, "activate", G_CALLBACK (gc_bonobo_control_activate),NULL);

	return control;
}

GCBonoboControl *
gc_bonobo_control_new (gcDocument *pDoc)
{
	GCBonoboControl *control;

	control = (GCBonoboControl*) g_object_new (GC_BONOBO_CONTROL_TYPE, NULL);

	return gc_bonobo_control_construct (control, pDoc);
}

BonoboObject * gcrystal_factory(BonoboGenericFactory * fact, const char *component_id, void* data)
{
	g_return_val_if_fail (fact != NULL, NULL);
	g_return_val_if_fail (component_id != NULL, NULL);

	BonoboObject *retval;
	gcDocument* pDoc= GetNewDocument();
	if (!pDoc) return NULL;
	if (!strcmp (component_id, "OAFIID:gcrystal_control"))
	{
		retval = (BonoboObject*) gc_bonobo_control_new(pDoc);
		if (!retval) RemoveDocument(pDoc);
	}
	else if (!strcmp (component_id, "OAFIID:gcrystal"))
	{
		g_warning ("GCrystal Item is not implemented yet");
		return NULL;
	}
	else
	{
		g_warning ("Unknown IID `%s' requested", component_id);
		RemoveDocument(pDoc);
		return NULL;
	}

	return retval;
}
