/*
   Copyright (C)  2000    Daniel A. Atkinson  <DanAtk@aol.com>
   Copyright (C)  2004    Ivano Primi  <ivano.primi@tin.it>    

   This file is part of the HPA Library.

   The HPA Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The HPA Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the HPA Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA.
*/

/*
    Test:  cxsqrt  cxexp   cxexp10  cxexp2   
           cxsin   cxcos   cxsinh   cxcosh   
	   cxasin  cxacos  cxasinh  cxacosh

    Uses:  All the tested functions + atocx  xpr_asprint

    Input file:  data/cxgen.dat
*/
#include<stdio.h>
#include<stdlib.h>
#include"cxpre.h"

#define DECD 32

int
getline (char *buffer, int size, FILE * fp)
{
  if ((fgets (buffer, size, fp)))
    {
      int i;

      for (i = 0; buffer[i] != '\0'; i++);
      if (i > 0 && buffer[i - 1] == '\n')
	buffer[--i] = '\0';
      return i;
    }
  else
    return -1;
}

typedef struct cxpr (*pfun) (struct cxpr);
extern int xErrNo;

int
main (int na, char **av)
{
  struct cxpr z, w;
  FILE *fp;
  char num[1024], *re, *im;
  const char *nlist[] = {
    "cxsqrt",
    "cxexp", "cxexp10", "cxexp2",
    "cxsin", "cxcos",
    "cxsinh", "cxcosh",
    "cxasin", "cxacos",
    "cxasinh", "cxacosh",
    NULL
  };
  pfun flist[] = {
    cxsqrt,
    cxexp, cxexp10, cxexp2,
    cxsin, cxcos,
    cxsinh, cxcosh,
    cxasin, cxacos,
    cxasinh, cxacosh
  };
  int i;

  if (na != 2)
    {
      printf ("para: input_file\n");
      exit (-1);
    }
  fp = fopen (*++av, "r");
  for (i = 0; nlist[i] != NULL; i++)
    {
      printf ("Test of %s\n", nlist[i]);
      while (getline (num, 1024, fp) != -1)
	{
	  printf ("*string in : \t%s\n", num);
	  /* convert ascii string to extended precision complex */
	  z = atocx (num);
	  re = xpr_asprint (z.re, 1, 0, DECD);
	  im = xpr_asprint (z.im, 1, 0, DECD);
	  printf (" z =  \t\t%s  \t%s\n", re, im);
	  free ((void *) re);
	  free ((void *) im);
	  printf (" %s(z) = \t", nlist[i]);
	  xErrNo = 0;
	  w = (*flist[i]) (z);
	  switch (xErrNo)
	    {
	    case XENONE:
	      re = xpr_asprint (w.re, 1, 0, DECD);
	      im = xpr_asprint (w.im, 1, 0, DECD);
	      printf ("%s  \t%s\n", re, im);
	      free ((void *) re);
	      free ((void *) im);
	      break;
	    case XEDIV:
	      printf ("*** Division by zero\n");
	      break;
	    case XEDOM:
	      printf ("*** Out of domain\n");
	      break;
	    case XEBADEXP:
	      printf ("*** Bad exponent\n");
	      break;
	    case XFPOFLOW:
	      printf ("*** Floating point overflow\n");
	      break;
	    default:
	      printf ("*** Unknown error\n");
	    }
	}
      printf ("\n");
      rewind (fp);
    }
  fclose (fp);
  return 0;
}

/*  Test output

Test of cxsqrt
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	4.80687008353668738925973242765574e-786  	0.00000000000000000000000000000000e-4181
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxsqrt(z) = 	3.31032803300781190751057277475666e-10  	1.47616790580115548109180558337055e-10
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxsqrt(z) = 	3.41306314034768480724313750533058e-786  	3.41306314034768480724313750533057e-786
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxsqrt(z) = 	1.97213223610005568752377271783964e-9  	2.08452046204037195810696043721115e-9
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxsqrt(z) = 	1.39520607796841252149383276668821e-812  	-1.39520607796841252149383276668820e-812
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxsqrt(z) = 	1.00546065158709980740186179544393e-9  	-3.64718399890791845287440189982148e-9
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxsqrt(z) = 	4.52697470724102897543998858407573e-809  	-4.52697470724102897543998858407573e-809
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxsqrt(z) = 	8.10277606351975187994110264557389e-10  	-5.07326127215508645069227601001414e-10
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	1.00000000000000000004930000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxsqrt(z) = 	1.00000000000000000022835000000000e+0  	1.21499999999999999972255474999997e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxsqrt(z) = 	1.00000000000000000000000000000000e+0  	1.72529999999999999999999999999735e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxsqrt(z) = 	9.99999999999999999367999999999989e-1  	2.33385000000000000147499319999645e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	9.99999999999999999876999999999989e-1  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxsqrt(z) = 	9.99999999999999999668999999999988e-1  	-2.53915000000000000084045864999659e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxsqrt(z) = 	1.00000000000000000000000000000000e+0  	-3.35009999999999999999999999999485e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxsqrt(z) = 	1.00000000000000037114999999999993e+0  	-2.66604999999999901049554249999632e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	0.00000000000000000000000000000000e+0  	9.99999999999999999994054999999987e-1
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxsqrt(z) = 	1.16720000000000000077558163258713e-15  	9.99999999999999999883160000000668e-1
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxsqrt(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxsqrt(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000327000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000371500000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxsqrt(z) = 	3.49915000000000057294990948630519e-18  	-1.00000000000000000042511500000000e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxsqrt(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxsqrt(z) = 	3.36605000000000000807917297267311e-16  	-9.99999999999999999882150000000044e-1
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxsqrt(z) = 	7.07106781186547524400844362104850e-1  	7.07106781186547524400844362104849e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxsqrt(z) = 	7.07106781186547524400844362104850e-1  	7.07106781186547524400844362104849e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxsqrt(z) = 	7.07106781186547526359812988704059e-1  	7.07106781186547522825198321586865e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxsqrt(z) = 	7.07106781186547524401976086508138e-1  	7.07106781186547524401976086508137e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxsqrt(z) = 	7.07106781186547527240782831695366e-1  	7.07106781186547527287127317241112e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxsqrt(z) = 	7.07106781186547524400844362104850e-1  	7.07106781186547524400844362104849e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxsqrt(z) = 	7.07106781186547500201670410913862e-1  	7.07106781186547548590401661071682e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxsqrt(z) = 	7.07106781186547485859041730445276e-1  	7.07106781186547485859041730445275e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxsqrt(z) = 	7.07106781186547677516933587441493e-1  	7.07106781186547371134636367122316e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxsqrt(z) = 	7.07106781186547524400844362104850e-1  	-7.07106781186547524400844362104849e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxsqrt(z) = 	7.07106781186547524400844362104850e-1  	-7.07106781186547524400844362104849e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxsqrt(z) = 	7.07106781186547536059797747003932e-1  	-7.07106781186547512498292691086982e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxsqrt(z) = 	7.07106781186547524399783692038158e-1  	-7.07106781186547524399783692038157e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxsqrt(z) = 	7.07106781186547497688553160053861e-1  	-7.07106781186547551104106517666847e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxsqrt(z) = 	7.07106781186547524400844362104850e-1  	-7.07106781186547524400844362104849e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxsqrt(z) = 	7.07106781186547312856843540909262e-1  	-7.07106781186547736576440031324135e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxsqrt(z) = 	7.07106781186547524403376157934887e-1  	-7.07106781186547524403376157934886e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxsqrt(z) = 	7.07106781186547524738120154595208e-1  	-7.07106781186547524125546478985490e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	1.25999999999999999999999999999860e+938  	0.00000000000000000000000000000000e-3994
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	0.00000000000000000000000000000000e-3950  	8.02440028911818431351101012278342e+982
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxsqrt(z) = 	4.13581914498204333654435916566134e+1007  	4.13581914498204333654435916566134e+1007
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxsqrt(z) = 	4.77084898105148576709866612478005e+1005  	-4.77084898105148576709866612478005e+1005
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxsqrt(z) = 	2.22221736110579425944916771232458e+936  	2.22221736110579425895625913408589e+936
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxsqrt(z) = 	0.00000000000000000000000000000000e-3959  	1.81565415208954373156192962020952e+973
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxsqrt(z) = 	0.00000000000000000000000000000000e-3777  	-2.46955461571514955632770265987165e+1155
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxsqrt(z) = 	4.39880665635578941161279968456743e+971  	-4.39880665635578941161279968456742e+971
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	3.16383889335409922259821807882355e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxsqrt(z) = 	3.28100382260412736730225005128782e+0  	9.31120875043751709814268828825224e-1
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxsqrt(z) = 	2.99999073434735794748613754490320e+0  	2.99999073434735794748613754490319e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxsqrt(z) = 	2.40860140326375557782765587524819e+0  	4.33400316333573209803012622188013e+0
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	0.00000000000000000000000000000000e-4932  	4.24119996304112027657752635923691e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxsqrt(z) = 	3.08319554921145952753935646036273e+0  	-5.34834711892256985257997174751881e+0
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxsqrt(z) = 	4.68975480075158087150261312100916e+0  	-4.68975480075158087150261312100915e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxsqrt(z) = 	5.57052635825947399967677015886770e+0  	-3.87924769970964314457327421416987e+0
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	3.16229327631562359370497262934710e+1  	0.00000000000000000000000000000000e-4931
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxsqrt(z) = 	3.28100382260412736730225005128782e+1  	9.31120875043751709814268828825223e+0
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxsqrt(z) = 	9.48680367993878115321114414864742e+0  	9.48680367993878115321114414864741e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxsqrt(z) = 	2.88794597151134791399249534244342e+0  	3.61464037206243883220930751393885e+1
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxsqrt(z) = 	0.00000000000000000000000000000000e-4931  	4.24119996304227678889544174424495e+1
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxsqrt(z) = 	9.74993593958394472530663367045063e+0  	-1.69129675670501614974702131794034e+1
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxsqrt(z) = 	5.57052635825947399967677015886769e+1  	-3.87924769970964314457327421416987e+1

Test of cxexp
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxexp(z) = 	1.00000000000000000008779200000000e+0  	9.77320000000000000085800877439977e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxexp(z) = 	1.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxexp(z) = 	9.99999999999999999544080000000000e-1  	8.22189999999999999625147135199982e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxexp(z) = 	1.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxexp(z) = 	9.99999999999999987709000000000001e-1  	-7.33419999999999990985534779999984e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxexp(z) = 	1.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxexp(z) = 	1.00000000000000000039917000000000e+0  	-8.22150000000000000328177615499983e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	2.71828182845904523536028747135266e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	2.71828182845904523562831005963872e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxexp(z) = 	2.71828182845904523660172678240990e+0  	6.60542484315547992494219608125592e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxexp(z) = 	2.71828182845904523536028747135266e+0  	9.37970327728078148913420794863509e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxexp(z) = 	2.71828182845904523192437924018036e+0  	1.26881240906982854290534249793704e-1296
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	2.71828182845904523469159014155167e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxexp(z) = 	2.71828182845904523356078490091271e+0  	-1.38042506094635694095917339622863e-1144
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxexp(z) = 	2.71828182845904523536028747135266e+0  	-1.82130319070412948859609981155291e-1300
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxexp(z) = 	2.71828182845904725314088873650268e+0  	-1.44941505375264858584725328318829e-1343
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	3.67879441171442321595523770161461e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	3.67879441171442321599897856717000e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxexp(z) = 	3.67879441171442321681489837973411e-1  	8.58777767470614955733269877766675e-16
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxexp(z) = 	3.67879441171442321595523770161461e-1  	1.67756703968589413070774794431130e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxexp(z) = 	3.67879441171442321593117838616200e-1  	2.73452146211556506486596351799699e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	3.67879441171442321592790425913558e-1  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxexp(z) = 	3.67879441171442321282741632894266e-1  	-2.57453069315010479703301076948388e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxexp(z) = 	3.67879441171442321595523770161461e-1  	-1.83975772770955962145278246409984e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxexp(z) = 	3.67879441171442321682232954445496e-1  	-2.47660118591026685379696047262285e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxexp(z) = 	5.40302305868139717400936607442977e-1  	8.41470984807896506652502321630299e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxexp(z) = 	5.40302305868139717400936607442977e-1  	8.41470984807896506652502321630299e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxexp(z) = 	5.40302305868139719645584322921687e-1  	8.41470984807896511151661213400650e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxexp(z) = 	5.40302305868139717398243058820608e-1  	8.41470984807896506654231829311383e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxexp(z) = 	5.40302305868139710551208471941247e-1  	8.41470984807896510972773574965787e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxexp(z) = 	5.40302305868139717400936607442977e-1  	8.41470984807896506652502321630299e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxexp(z) = 	5.40302305868139680438413217667850e-1  	8.41470984807896449061611777896505e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxexp(z) = 	5.40302305868139809131943803591080e-1  	8.41470984807896447752699948764642e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxexp(z) = 	5.40302305868139951687167007124024e-1  	8.41470984807896871138759149508035e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxexp(z) = 	5.40302305868139717400936607442977e-1  	-8.41470984807896506652502321630299e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxexp(z) = 	5.40302305868139717400936607442977e-1  	-8.41470984807896506652502321630299e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxexp(z) = 	5.40302305868139735694236495541603e-1  	-8.41470984807896534505022862042631e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxexp(z) = 	5.40302305868139717403461043947679e-1  	-8.41470984807896506650881399591235e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxexp(z) = 	5.40302305868139676596704862862872e-1  	-8.41470984807896443080043538113346e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxexp(z) = 	5.40302305868139717400936607442977e-1  	-8.41470984807896506652502321630299e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxexp(z) = 	5.40302305868139392883975563737451e-1  	-8.41470984807896002900447517819108e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxexp(z) = 	5.40302305868139717394910833720768e-1  	-8.41470984807896506656371426442621e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxexp(z) = 	5.40302305868139717795250966221194e-1  	-8.41470984807896507428834547588571e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	*** Floating point overflow
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxexp(z) = 	1.00000000000000000000000000000000e+0  	2.83842519004155504485943795581093e-1980
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxexp(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxexp(z) = 	*** Floating point overflow
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxexp(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxexp(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxexp(z) = 	*** Floating point overflow
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	2.22450889770625148990826385426241e+4  	0.00000000000000000000000000000000e-4928
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxexp(z) = 	1.95930806494526388608248274089212e+4  	-3.42711997615723811485035732318001e+3
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxexp(z) = 	6.60233203647946541284590641061578e-1  	-7.51060661199059972394391176909120e-1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxexp(z) = 	-1.01614475900983895842324835933604e-6  	2.06433000349417459488910275060840e-6
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	1.54172761790381731294557683363252e-8  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxexp(z) = 	3.41253505572734944565348150908327e-11  	-5.07598771158903316802465353119493e-9
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxexp(z) = 	9.99985938958289652348443900980463e-1  	-5.30300723248623553824633481538502e-3
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxexp(z) = 	6.30689795492672846225471031331820e+6  	6.03527525685216287786917117278458e+6
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	1.98962500978175844131444704584726e+434  	0.00000000000000000000000000000000e-4498
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxexp(z) = 	2.74841735730998784743501562132872e+428  	7.31765758354908945475907577516725e+429
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxexp(z) = 	-5.99350481814068473887456037147628e-1  	-8.00486726903853150675215925051745e-1
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxexp(z) = 	2.12972538452346545352403898745893e-565  	1.53134087818770966059830225644912e-564
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxexp(z) = 	6.32070068796820849259460646011324e-782  	0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxexp(z) = 	-1.13329237256952474193494313481499e-83  	-7.55070745440985131391942856127386e-85
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxexp(z) = 	7.38197567410941705583203885227343e+693  	1.01327175699758070286737955702451e+694

Test of cxexp10
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxexp10(z) = 	1.00000000000000000020214855048413e+0  	2.25036246308494072835937091846859e-19
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxexp10(z) = 	1.00000000000000000000000000000000e+0  	5.36456274965752763462511649111366e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxexp10(z) = 	9.99999999999999998950205404402156e-1  	1.89316243760877441895532069853636e-17
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxexp10(z) = 	1.00000000000000000000000000000000e+0  	-8.96442428404441865701884433135993e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxexp10(z) = 	9.99999999999999971698926622010185e-1  	-1.68876195890369293777869917580333e-17
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxexp10(z) = 	1.00000000000000000000000000000000e+0  	-9.43760552065469504508454157529674e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxexp10(z) = 	1.00000000000000000091912289157043e+0  	-1.89307033420505466085535595398946e-18
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	9.99999999999999999999999999999997e+0  	0.00000000000000000000000000000000e-4931
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	1.00000000000000000022703489016921e+1  	0.00000000000000000000000000000000e-4931
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxexp10(z) = 	1.00000000000000000105159061197038e+1  	5.59528177597553101804766502182769e-17
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxexp10(z) = 	9.99999999999999999999999999999997e+0  	7.94530012188525403727248131352165e-1297
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxexp10(z) = 	9.99999999999999997089532442455468e+0  	1.07477764385683070080097240728192e-1295
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	9.99999999999999999433564067123406e+0  	0.00000000000000000000000000000000e-4931
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxexp10(z) = 	9.99999999999999998475688668437883e+0  	-1.16932178777516621793244614908262e-1143
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxexp10(z) = 	9.99999999999999999999999999999997e+0  	-1.54277806400787048920573463446524e-1299
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxexp10(z) = 	1.00000000000000170920891452948157e+1  	-1.22776139743535719767595863523300e-1342
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	1.00000000000000000000000000000000e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	1.00000000000000000002737773675576e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxexp10(z) = 	1.00000000000000000053806808451646e-1  	5.37515464108530024766379841357540e-16
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxexp10(z) = 	1.00000000000000000000000000000000e-1  	1.05000182825621477236904428324938e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxexp10(z) = 	9.99999999999999999984941093491820e-2  	1.71155755132433403781847922294412e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	9.99999999999999999982891792759056e-2  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxexp10(z) = 	9.99999999999999998042273076383675e-2  	-1.61141812563002298789159430998247e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxexp10(z) = 	1.00000000000000000000000000000000e-1  	-1.15151819983613625848602949050335e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxexp10(z) = 	1.00000000000000000054271930641755e-1  	-1.55012331045452149577903387512083e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxexp10(z) = 	-6.68201510190312946242330696656144e-1  	7.43980336957493187658416406875515e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxexp10(z) = 	-6.68201510190312946242330696656144e-1  	7.43980336957493187658416406875515e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxexp10(z) = 	-6.68201510190312954861944312854254e-1  	7.43980336957493195387509479771221e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxexp10(z) = 	-6.68201510190312946247814259440923e-1  	7.43980336957493187653491377607943e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxexp10(z) = 	-6.68201510190312960014167136586991e-1  	7.43980336957493175086477106589826e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxexp10(z) = 	-6.68201510190312946242330696656144e-1  	7.43980336957493187658416406875515e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxexp10(z) = 	-6.68201510190312840930184713380925e-1  	7.43980336957493070449985263427863e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxexp10(z) = 	-6.68201510190312759495103231060895e-1  	7.43980336957493355384326915037365e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxexp10(z) = 	-6.68201510190313612534667768733681e-1  	7.43980336957493930244640315586529e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxexp10(z) = 	-6.68201510190312946242330696656144e-1  	-7.43980336957493187658416406875515e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxexp10(z) = 	-6.68201510190312946242330696656144e-1  	-7.43980336957493187658416406875515e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxexp10(z) = 	-6.68201510190312996919560576524779e-1  	-7.43980336957493245269934099598631e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxexp10(z) = 	-6.68201510190312946237191414612090e-1  	-7.43980336957493187663032222445554e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxexp10(z) = 	-6.68201510190312829993766024688024e-1  	-7.43980336957493058270434955293546e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxexp10(z) = 	-6.68201510190312946242330696656144e-1  	-7.43980336957493187658416406875515e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxexp10(z) = 	-6.68201510190312025802682182729126e-1  	-7.43980336957492159756381756103538e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxexp10(z) = 	-6.68201510190312946254598048453025e-1  	-7.43980336957493187647398557895481e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxexp10(z) = 	-6.68201510190312947725378613831328e-1  	-7.43980336957493189007615551137261e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	*** Floating point overflow
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxexp10(z) = 	1.00000000000000000000000000000000e+0  	1.41921259502077752242971897790546e-1980
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxexp10(z) = 	1.00000000000000000000000000000000e+0  	1.16261895784102094637442578670016e-1976
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxexp10(z) = 	*** Floating point overflow
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxexp10(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxexp10(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxexp10(z) = 	*** Floating point overflow
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	1.02300214210602772667042371555189e+10  	0.00000000000000000000000000000000e-4922
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxexp10(z) = 	5.39776816905205848127501696911686e+8  	7.88834014642425333506731778082652e+9
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxexp10(z) = 	-8.22177110380382648034321932968674e-1  	-5.69231762260824731770033876865686e-1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxexp10(z) = 	-6.06870906165474106146704909051578e-14  	-8.46770152386625390558177070000126e-14
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	1.02854399626657472202098046666973e-18  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxexp10(z) = 	6.82886465204743494398954284710201e-20  	-4.10301705185449023479569049433982e-20
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxexp10(z) = 	7.28800682857863065575437815414495e-1  	-6.84725904772057313724776841800516e-1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxexp10(z) = 	5.05811301871035274218793897323071e+15  	8.15755915363860915404618842962165e+15
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	1.02300214210602772667042371554985e+1000  	0.00000000000000000000000000000000e-3933
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxexp10(z) = 	5.38274956839246681411052947761090e+989  	-3.29191800495003791222464021783762e+989
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxexp10(z) = 	9.74239221219182479259964689569664e-1  	-2.25517049994542149362489313655356e-1
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxexp10(z) = 	-5.98160467422596449662705672504067e-1299  	-3.88118213838712417291367127065964e-1300
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxexp10(z) = 	1.66835070429935578916674080173015e-1799  	0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxexp10(z) = 	6.63101491061755487181508792093527e-192  	7.87979382633999212885048948368404e-192
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxexp10(z) = 	8.32719171070893291571958779657420e+1597  	1.43607808099727822504321040428683e+1598

Test of cxexp2
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxexp2(z) = 	1.00000000000000000006085277727572e+0  	6.77426602504845749840872587106363e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxexp2(z) = 	1.00000000000000000000000000000000e+0  	1.61489430126856058188026739657026e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxexp2(z) = 	9.99999999999999999683980337439110e-1  	5.69898680384581433769654889272654e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxexp2(z) = 	1.00000000000000000000000000000000e+0  	-2.69856060335597907862316809525579e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxexp2(z) = 	9.99999999999999991480528003737713e-1  	-5.08368005166275084501759398710042e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxexp2(z) = 	1.00000000000000000000000000000000e+0  	-2.84100234896104783970840929621519e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxexp2(z) = 	1.00000000000000000027668356006411e+0  	-5.69870954497359036295051313124291e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	2.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	2.00000000000000000013668862400642e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxexp2(z) = 	2.00000000000000000063312063472345e+0  	3.36869529752133420483414336276497e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxexp2(z) = 	2.00000000000000000000000000000000e+0  	4.78354732248029456935020231659982e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxexp2(z) = 	1.99999999999999999824772392754445e+0  	6.47080618939931343586430931905988e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	1.99999999999999999965897158716450e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxexp2(z) = 	1.99999999999999999908227313293862e+0  	-7.04001865407514052639665263306356e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxexp2(z) = 	2.00000000000000000000000000000000e+0  	-9.28844947837549112431467732037386e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxexp2(z) = 	2.00000000000000102904630425929507e+0  	-7.39186016292737257197043792060230e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	5.00000000000000000000000000000000e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	5.00000000000000000004120759988433e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxexp2(z) = 	5.00000000000000000080987316575972e-1  	8.09041389149568165282837514378917e-16
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxexp2(z) = 	5.00000000000000000000000000000000e-1  	1.58041022903570330273676009852885e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxexp2(z) = 	4.99999999999999999997733408719569e-1  	2.57615081126909273696840674067500e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	4.99999999999999999997424958224220e-1  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxexp2(z) = 	4.99999999999999999705332736336259e-1  	-2.42542595685633262801792051794859e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxexp2(z) = 	5.00000000000000000000000000000000e-1  	-1.73320759351833764674469474738286e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxexp2(z) = 	5.00000000000000000081687395228937e-1  	-2.33316806712380390914505502650377e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxexp2(z) = 	7.69238901363972126578329993661271e-1  	6.38961276313634801150032911464702e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxexp2(z) = 	7.69238901363972126578329993661271e-1  	6.38961276313634801150032911464702e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxexp2(z) = 	7.69238901363972129003522767736768e-1  	6.38961276313634803652973614798203e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxexp2(z) = 	7.69238901363972126576912289304141e-1  	6.38961276313634801151739671142581e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxexp2(z) = 	7.69238901363972122956782230295898e-1  	6.38961276313634805438877893084714e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxexp2(z) = 	7.69238901363972126578329993661271e-1  	6.38961276313634801150032911464702e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxexp2(z) = 	7.69238901363972090096700035086920e-1  	6.38961276313634770834645064283946e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxexp2(z) = 	7.69238901363972174859414473060151e-1  	6.38961276313634743024932442365430e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxexp2(z) = 	7.69238901363972357700754068466128e-1  	6.38961276313634992938466470286928e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxexp2(z) = 	7.69238901363972126578329993661271e-1  	-6.38961276313634801150032911464702e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxexp2(z) = 	7.69238901363972126578329993661271e-1  	-6.38961276313634801150032911464702e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxexp2(z) = 	7.69238901363972144497523488711669e-1  	-6.38961276313634815724024843655008e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxexp2(z) = 	7.69238901363972126579658688678059e-1  	-6.38961276313634801148433309215160e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxexp2(z) = 	7.69238901363972086305843220851833e-1  	-6.38961276313634767686553231246202e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxexp2(z) = 	7.69238901363972126578329993661271e-1  	-6.38961276313634801150032911464702e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxexp2(z) = 	7.69238901363971806675827811656538e-1  	-6.38961276313634536230792951487796e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxexp2(z) = 	7.69238901363972126575158428243772e-1  	-6.38961276313634801153851126414185e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxexp2(z) = 	7.69238901363972127001423148813187e-1  	-6.38961276313634801580451201809133e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	*** Floating point overflow
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxexp2(z) = 	1.00000000000000000000000000000000e+0  	3.78456692005540672647925060774791e-1980
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxexp2(z) = 	1.00000000000000000000000000000000e+0  	1.55015861045469459516590104893354e-1976
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxexp2(z) = 	*** Floating point overflow
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxexp2(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxexp2(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxexp2(z) = 	*** Floating point overflow
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	1.03103424996204782550186303665942e+3  	0.00000000000000000000000000000000e-4929
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxexp2(z) = 	-4.38250710137595261495730989618205e+2  	-8.47494732940394230875076206975899e+2
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxexp2(z) = 	9.95970829266824219801666012162950e-1  	-8.96777968593925714006297418289207e-2
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxexp2(z) = 	-4.05373935503018511191976817578037e-5  	1.16746204174773153142490620867597e-4
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	3.84715363078116814090792911299804e-6  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxexp2(z) = 	-1.15013362558510984113567865964458e-6  	1.36009287777713808795643557259065e-6
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxexp2(z) = 	6.00999313669123594064541541321832e-1  	7.99249538610590872490245789988174e-1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxexp2(z) = 	7.23042312370386816332539321950916e+3  	6.43273608096782470961336471460165e+4
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	1.07886921204899575773181620858027e+301  	0.00000000000000000000000000000000e-4631
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxexp2(z) = 	-7.51555748998334844885382480597850e+297  	5.14757606566339806230086152253298e+297
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxexp2(z) = 	6.23187627555010888622215367313682e-1  	-7.82072362932201455639525573169428e-1
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxexp2(z) = 	1.53947843524298543022531151811004e-391  	3.12752663435862087361066199811748e-392
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxexp2(z) = 	3.26552460340562779582428289203326e-542  	0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxexp2(z) = 	-2.38283113873786444394878126024755e-58  	-2.15762075979745502429710835193899e-58
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxexp2(z) = 	2.54046216483845074585757837180057e+480  	1.26961117515424148042701379788451e+481

Test of cxsin
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	2.31059999999999999999999999999571e-1571  	-0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxsin(z) = 	8.77919999999999999999999999999979e-20  	9.77320000000000435661642035976737e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxsin(z) = 	-4.55919999999999999999999999999990e-19  	8.22189999999999994626368727154255e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxsin(z) = 	-1.22909999999999999999999999999998e-17  	-7.33420000000000023242731814120121e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxsin(z) = 	3.99169999999999999999999999999991e-19  	-8.22150000000000474780953325262615e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	8.41470984807896506652502321630299e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	8.41470984807896506705776128988898e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxsin(z) = 	8.41470984807896506899258384720279e-1  	1.31293460325957942363145429406539e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxsin(z) = 	8.41470984807896506652502321630299e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxsin(z) = 	8.41470984807896505969560207012958e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	8.41470984807896506519587954386724e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxsin(z) = 	8.41470984807896506294822195145578e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxsin(z) = 	8.41470984807896506652502321630299e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxsin(z) = 	8.41470984807896907718903967550180e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	-8.41470984807896506652502321630299e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	-8.41470984807896506646078127213512e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxsin(z) = 	-8.41470984807896506526244478797310e-1  	1.26128170281858535696470183711544e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxsin(z) = 	-8.41470984807896506652502321630299e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxsin(z) = 	-8.41470984807896506656035898710677e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	-8.41470984807896506656516767762899e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxsin(z) = 	-8.41470984807896507111883551148568e-1  	-3.78119762715700240112551757358578e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxsin(z) = 	-8.41470984807896506652502321630299e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxsin(z) = 	-8.41470984807896506525153068137355e-1  	-3.63736915333490339154100698485726e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxsin(z) = 	0.00000000000000000000000000000000e-4932  	1.17520119364380145688238185059560e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxsin(z) = 	8.65637374518655454850535495130045e-2198  	1.17520119364380145688238185059560e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxsin(z) = 	7.71339716925095907866206146231247e-18  	1.17520119364380145771888586272894e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxsin(z) = 	0.00000000000000000000000000000000e-4932  	1.17520119364380145688732125170764e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxsin(z) = 	-1.01135047886425893108967344557442e-19  	1.17520119364380146937840313939292e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxsin(z) = 	-1.09555638910612677784374343262225e-2199  	1.17520119364380145688238185059560e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxsin(z) = 	-1.05596094001676762247706306833708e-16  	1.17520119364380145686139595396207e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxsin(z) = 	0.00000000000000000000000000000000e-4932  	1.17520119364380128866702639328454e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxsin(z) = 	6.68601408259096976668587945399110e-16  	1.17520119364380145655478583182417e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-1.17520119364380145688238185059560e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxsin(z) = 	4.28390045837407977781036158433505e-2111  	-1.17520119364380145688238185059560e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxsin(z) = 	5.14169898326787379291720593829059e-17  	-1.17520119364380145635079057190171e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-1.17520119364380145687775256550490e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxsin(z) = 	-1.16565854234578330268865889370555e-16  	-1.17520119364380145686267825396960e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxsin(z) = 	-3.26454139101512973774785713126513e-2203  	-1.17520119364380145688238185059560e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxsin(z) = 	-9.24660208800338530006327989908131e-16  	-1.17520119364380145826067690441871e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-1.17520119364380145689343185102151e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxsin(z) = 	1.33678618474679383782242987938066e-18  	-1.17520119364380145701763286823715e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxsin(z) = 	*** Floating point overflow
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxsin(z) = 	*** Floating point overflow
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxsin(z) = 	*** Floating point overflow
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxsin(z) = 	*** Floating point overflow
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxsin(z) = 	*** Floating point overflow
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxsin(z) = 	*** Floating point overflow
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	-5.52281568927395393436697194096161e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxsin(z) = 	-1.02625214869005894708757718375029e+2  	-2.00427727559693760190195910315082e+2
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxsin(z) = 	0.00000000000000000000000000000000e-4925  	3.28263344824292563018151429158893e+7
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxsin(z) = 	-2.35732730870935451632357362242745e+8  	5.33807087808961721577120916001420e+8
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	7.59001915865750446530127491912761e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxsin(z) = 	-2.59410794197346998744688834324899e+13  	-1.01948038245910216678222291390340e+14
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxsin(z) = 	0.00000000000000000000000000000000e+0  	-6.34661327363566563440466783499043e+18
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxsin(z) = 	-7.96840358427977252789933862119803e+17  	2.83244427693939887760139250144444e+18
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	8.32393482306133544452486030530852e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxsin(z) = 	-2.23033528650720365450796320194638e+264  	-1.10986538562753422285426020845881e+265
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxsin(z) = 	0.00000000000000000000000000000000e-4854  	7.43864554819823472373588101667504e+77
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxsin(z) = 	1.58840269896674801672087287066395e+90  	-1.72393324174111194540637050328997e+90
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxsin(z) = 	-9.76780028762991890570092483008109e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxsin(z) = 	-5.14569496276266759180821833330915e+142  	6.76980354925894720314138245903140e+142
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxsin(z) = 	3.53258435774745451731370926157239e+1876  	3.09990974021556001905656125944681e+1876

Test of cxcos
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	-8.58008774399999990781923893368530e-39
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	3.74852864799999986979389318910869e-36
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	-9.01446522000000036146478711062011e-35
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	3.28177615500000206966958121716892e-37
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	5.40302305868139717317967568340919e-1  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxcos(z) = 	5.40302305868139717016636808681211e-1  	-2.04477449308318825460049816289819e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxcos(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxcos(z) = 	5.40302305868139718464555932240179e-1  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	5.40302305868139717607938469705740e-1  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxcos(z) = 	5.40302305868139717957990399385825e-1  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxcos(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxcos(z) = 	5.40302305868139092777024584541252e-1  	0.00000000000000000000000000000000e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	5.40302305868139717410941697452366e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxcos(z) = 	5.40302305868139717597571547174381e-1  	1.96432986693555360517009030624522e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxcos(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxcos(z) = 	5.40302305868139717395433387202334e-1  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	5.40302305868139717394684478025855e-1  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxcos(z) = 	5.40302305868139716685492732029760e-1  	-5.88886639298110236290903904433041e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxcos(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxcos(z) = 	5.40302305868139717599271318562343e-1  	-5.66486681682524006978035833079525e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxcos(z) = 	1.54308063481524377847790562075706e+0  	0.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxcos(z) = 	1.54308063481524377847790562075706e+0  	-6.59264365610299741281878570545405e-2198
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxcos(z) = 	1.54308063481524377911498218783137e+0  	-5.87447820666727034669939476202304e-18
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxcos(z) = 	1.54308063481524377848166743977791e+0  	0.00000000000000000000000000000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxcos(z) = 	1.54308063481524378799480240700393e+0  	7.70238614326083921045299201589498e-20
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxcos(z) = 	1.54308063481524377847790562075706e+0  	8.34369343463226158357353466283692e-2200
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxcos(z) = 	1.54308063481524377846192288452347e+0  	8.04213680834326212959390479215307e-17
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxcos(z) = 	1.54308063481524365036607396244728e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxcos(z) = 	1.54308063481524377822841040734630e+0  	-5.09202925193922733110623153071118e-16
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxcos(z) = 	1.54308063481524377847790562075706e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxcos(z) = 	1.54308063481524377847790562075706e+0  	3.26259355379392160459686849361535e-2111
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxcos(z) = 	1.54308063481524377807304880954674e+0  	3.91588789734051083270646926463360e-17
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxcos(z) = 	1.54308063481524377847437998428573e+0  	0.00000000000000000000000000000000e-4932
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxcos(z) = 	1.54308063481524377846289947671538e+0  	-8.87758733690464058528635779831157e-17
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxcos(z) = 	1.54308063481524377847790562075706e+0  	-2.48625564527282636218036704311357e-2203
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxcos(z) = 	1.54308063481524377952760707893136e+0  	-7.04215811267175147833545421434894e-16
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxcos(z) = 	1.54308063481524377848632123650474e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxcos(z) = 	1.54308063481524377858091200537994e+0  	1.01808854606556164022894553008251e-18
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	1.35205836825448901526582344714312e-1840
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	1.00000000000000000000000000000000e+0  	1.32747618005984383186408840733776e-1930
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxcos(z) = 	*** Floating point overflow
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxcos(z) = 	*** Floating point overflow
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxcos(z) = 	*** Floating point overflow
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxcos(z) = 	*** Floating point overflow
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxcos(z) = 	*** Floating point overflow
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxcos(z) = 	*** Floating point overflow
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	-8.33657644733792752528934953528207e-1  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxcos(z) = 	-2.00429704037766277444420939018444e+2  	1.02624202860891380195229886632928e+2
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxcos(z) = 	3.28263344824292715334883675834717e+7  	0.00000000000000000000000000000000e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxcos(z) = 	5.33807087808961722360932735600971e+8  	2.35732730870935451286220892827328e+8
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	6.51088390091637466139685731472280e-1  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxcos(z) = 	1.01948038245910216678222291394946e+14  	-2.59410794197346998744688834313179e+13
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxcos(z) = 	6.34661327363566563440466783499043e+18  	0.00000000000000000000000000000000e-4913
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxcos(z) = 	-2.83244427693939887760139250144444e+18  	-7.96840358427977252789933862119803e+17
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	5.54185068920363950523834025890113e-1  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxcos(z) = 	-1.10986538562753422285426020845881e+265  	2.23033528650720365450796320194638e+264
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxcos(z) = 	7.43864554819823472373588101667504e+77  	-0.00000000000000000000000000000000e-4854
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxcos(z) = 	-1.72393324174111194540637050328997e+90  	-1.58840269896674801672087287066395e+90
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxcos(z) = 	-2.14244662500069613632002680691563e-1  	0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxcos(z) = 	-6.76980354925894720314138245903140e+142  	-5.14569496276266759180821833330915e+142
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxcos(z) = 	-3.09990974021556001905656125944681e+1876  	3.53258435774745451731370926157239e+1876

Test of cxsinh
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxsinh(z) = 	8.77920000000000795093312100726045e-20  	9.77319999999999999999999999999977e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxsinh(z) = 	-4.55919999999999882056122642566678e-19  	8.22189999999999999999999999999982e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxsinh(z) = 	-1.22910000000000001189707920112774e-17  	-7.33419999999999999999999999999984e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxsinh(z) = 	3.99169999999999779581069919279131e-19  	-8.22149999999999999999999999999983e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	1.17520119364380145688238185059560e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	1.17520119364380145703452960118838e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxsinh(z) = 	1.17520119364380145758710677651572e+0  	3.74968594260104238300552661432278e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxsinh(z) = 	1.17520119364380145688238185059560e+0  	5.32455403849348018201586113497614e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxsinh(z) = 	1.17520119364380145493192792818910e+0  	7.20263747912711337786766498897059e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	1.17520119364380145650278401443101e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxsinh(z) = 	1.17520119364380145586086247034787e+0  	-7.83622638778225247629351607912679e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxsinh(z) = 	1.17520119364380145688238185059560e+0  	-1.03389488693890963645576632401805e-1300
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxsinh(z) = 	1.17520119364380260231113707395138e+0  	-8.22786025289836600268931883987673e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	-1.17520119364380145688238185059560e+0  	0.00000000000000000000000000000000e-4932
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	-1.17520119364380145686403462184760e+0  	0.00000000000000000000000000000000e-4932
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxsinh(z) = 	-1.17520119364380145652179476784873e+0  	3.60216743391270507583774758383768e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxsinh(z) = 	-1.17520119364380145688238185059560e+0  	7.03660200282099315423709742120598e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxsinh(z) = 	-1.17520119364380145689247359794729e+0  	1.14700269747086700542390982662430e-1531
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	-1.17520119364380145689384693971228e+0  	0.00000000000000000000000000000000e-4932
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxsinh(z) = 	-1.17520119364380145819435529873456e+0  	-1.07989412066275205419145674566178e-17
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxsinh(z) = 	-1.17520119364380145688238185059560e+0  	-7.71691539309833783129243645128432e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxsinh(z) = 	-1.17520119364380145651867774496934e+0  	-1.03881731416397026392263509195320e-15
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	8.41470984807896506652502321630299e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	8.41470984807896506652502321630299e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxsinh(z) = 	2.70080913634307003532580723758278e-18  	8.41470984807896506945400201641418e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	8.41470984807896506654231829311383e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxsinh(z) = 	-3.54119534289038350137389205443908e-20  	8.41470984807896511027924424781081e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	8.41470984807896506652502321630299e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxsinh(z) = 	-3.69739673951685375113577757696710e-17  	8.41470984807896506645154210270480e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	8.41470984807896447752699948764642e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxsinh(z) = 	2.34107586109606258167452474624405e-16  	8.41470984807896506537796142094558e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-8.41470984807896506652502321630299e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-8.41470984807896506652502321630299e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxsinh(z) = 	1.80034131338322831402675722893607e-17  	-8.41470984807896506466368177258712e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-8.41470984807896506650881399591235e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxsinh(z) = 	-4.08149764875851425012296305004826e-17  	-8.41470984807896506645603201486656e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-8.41470984807896506652502321630299e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxsinh(z) = 	-3.23765350745365362209132743164365e-16  	-8.41470984807896507135105744254931e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-8.41470984807896506656371426442621e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxsinh(z) = 	4.68069290596627923256974484513139e-19  	-8.41470984807896506699859818739643e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	*** Floating point overflow
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	*** Floating point overflow
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxsinh(z) = 	*** Floating point overflow
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxsinh(z) = 	*** Floating point overflow
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxsinh(z) = 	*** Floating point overflow
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxsinh(z) = 	*** Floating point overflow
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	1.11225444660543862392007852998622e+4  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxsinh(z) = 	9.79654029996469303157644787139314e+3  	-1.71355999240979423053221479712797e+3
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-7.51060661199059972394391176909120e-1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxsinh(z) = 	9.59712052186242553428012009364705e+4  	1.94968518658159144343534986982302e+5
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	-3.24311502365000140312671130167038e+7  	0.00000000000000000000000000000000e-4925
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxsinh(z) = 	-6.62195725664570053196911826454023e+5  	-9.84985446669268485763568965184188e+7
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-5.30300723248623553824633481538502e-3
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxsinh(z) = 	3.15344897746332284810215104375303e+6  	3.01763762842612103969392131813045e+6
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	9.94812504890879220657223522923629e+433  	0.00000000000000000000000000000000e-4498
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxsinh(z) = 	1.37420867865499392371750781066436e+428  	3.65882879177454472737953788758363e+429
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxsinh(z) = 	0.00000000000000000000000000000000e+0  	-8.00486726903853150675215925051745e-1
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxsinh(z) = 	-4.45481734065219720614116306403713e+562  	3.20315658918935587831360092758868e+563
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxsinh(z) = 	-7.91051537928028640236426723074910e+780  	0.00000000000000000000000000000000e-4151
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxsinh(z) = 	4.39242590351548073281216347256289e+82  	-2.92651074121498313189782184800059e+81
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxsinh(z) = 	3.69098783705470852791601942613671e+693  	5.06635878498790351433689778512256e+693

Test of cxcosh
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	8.58008774400001034830638773132455e-39
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	-3.74852864799999936865051004682201e-36
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	9.01446521999999996055008059679076e-35
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	-3.28177615499999789347472169811323e-37
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	1.54308063481524377847790562075706e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	1.54308063481524377859378045845034e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxcosh(z) = 	1.54308063481524377901462000589418e+0  	2.85573890055443754193666946693314e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxcosh(z) = 	1.54308063481524377847790562075706e+0  	4.05514923878730130711834681365895e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxcosh(z) = 	1.54308063481524377699245131199127e+0  	5.48548661157117205118575999039982e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	1.54308063481524377818880612712066e+0  	0.00000000000000000000000000000000e-4932
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxcosh(z) = 	1.54308063481524377769992243056484e+0  	-5.96802422168131693329821788315951e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxcosh(z) = 	1.54308063481524377847790562075706e+0  	-7.87408303765219852140333487534852e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxcosh(z) = 	1.54308063481524465082975166255131e+0  	-6.26629028462811985578321399200619e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	1.54308063481524377847790562075706e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	1.54308063481524377846393247856460e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxcosh(z) = 	1.54308063481524377820328460582214e+0  	-2.74338966644209012010447770607101e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxcosh(z) = 	1.54308063481524377847790562075706e+0  	-5.35903496313509902352934947689468e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxcosh(z) = 	1.54308063481524377848559143656349e+0  	-8.73550551259310498937313474824599e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	1.54308063481524377848663736562583e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxcosh(z) = 	1.54308063481524377947709693162883e+0  	8.22441051347741574488155668713390e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxcosh(z) = 	1.54308063481524377847790562075706e+0  	5.87715766538877820983965398718448e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxcosh(z) = 	1.54308063481524377820091069941483e+0  	7.91157195572943578542939044690913e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxcosh(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxcosh(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxcosh(z) = 	5.40302305868139716944775186578616e-1  	4.20626101175923248764690016786621e-18
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxcosh(z) = 	5.40302305868139717398243058820608e-1  	0.00000000000000000000000000000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxcosh(z) = 	5.40302305868139710586620425370151e-1  	-5.51508498152943630831680298911900e-20
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxcosh(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxcosh(z) = 	5.40302305868139717412380612836387e-1  	-5.75835424323739743615614781552633e-17
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxcosh(z) = 	5.40302305868139809131943803591080e-1  	0.00000000000000000000000000000000e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxcosh(z) = 	5.40302305868139717579580897517766e-1  	3.64600963007413477050238761820125e-16
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxcosh(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxcosh(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxcosh(z) = 	5.40302305868139717690823361709320e-1  	-2.80386546847839190529871521251528e-17
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxcosh(z) = 	5.40302305868139717403461043947679e-1  	0.00000000000000000000000000000000e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxcosh(z) = 	5.40302305868139717411681350448014e-1  	6.35655596633733101492526947480102e-17
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxcosh(z) = 	5.40302305868139717400936607442977e-1  	0.00000000000000000000000000000000e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxcosh(z) = 	5.40302305868139716649326309102813e-1  	5.04234658226435823715738328121071e-16
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxcosh(z) = 	5.40302305868139717394910833720768e-1  	0.00000000000000000000000000000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxcosh(z) = 	5.40302305868139717327181675624566e-1  	-7.28974728848928647775528612355083e-19
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	*** Floating point overflow
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	*** Floating point overflow
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	2.83842519004155504485943795581093e-1980
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxcosh(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxcosh(z) = 	*** Floating point overflow
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxcosh(z) = 	*** Floating point overflow
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxcosh(z) = 	*** Floating point overflow
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxcosh(z) = 	*** Floating point overflow
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	1.11225445110081286598818532427618e+4  	0.00000000000000000000000000000000e-4928
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxcosh(z) = 	9.79654034948794582924837953752801e+3  	-1.71355998374744388431814252605203e+3
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxcosh(z) = 	6.60233203647946541284590641061578e-1  	0.00000000000000000000000000000000e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxcosh(z) = 	-9.59712052196404001018110398948937e+4  	-1.94968518656094814340040812387413e+5
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	3.24311502365000294485432920548770e+7  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxcosh(z) = 	6.62195725664570087322262383727517e+5  	9.84985446669268435003691849293856e+7
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxcosh(z) = 	9.99985938958289652348443900980463e-1  	0.00000000000000000000000000000000e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxcosh(z) = 	3.15344897746340561415255926956517e+6  	3.01763762842604183817524985465413e+6
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	9.94812504890879220657223522923629e+433  	0.00000000000000000000000000000000e-4498
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxcosh(z) = 	1.37420867865499392371750781066436e+428  	3.65882879177454472737953788758363e+429
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxcosh(z) = 	-5.99350481814068473887456037147628e-1  	0.00000000000000000000000000000000e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxcosh(z) = 	4.45481734065219720614116306403713e+562  	-3.20315658918935587831360092758868e+563
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxcosh(z) = 	7.91051537928028640236426723074910e+780  	-0.00000000000000000000000000000000e-4151
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxcosh(z) = 	-4.39242590351548073281216347256289e+82  	2.92651074121498313189782184800059e+81
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxcosh(z) = 	3.69098783705470852791601942613671e+693  	5.06635878498790351433689778512256e+693

Test of cxasin
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	2.31059999999999999999999999999571e-1571  	-0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxasin(z) = 	8.77919999999999999999999999999979e-20  	9.77320000000013917171252746627708e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxasin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxasin(z) = 	-4.55919999999999999999999999999990e-19  	8.22190000000000167960063722005481e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxasin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxasin(z) = 	-1.22909999999999999999999999999997e-17  	-7.33420000000000003983432370247762e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxasin(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxasin(z) = 	3.99169999999999999999999999999991e-19  	-8.22149999999999897001970009091859e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	1.57079632679489661923132169163975e+0  	-4.44072066223490349345309743370226e-10
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxasin(z) = 	1.57079632537509233202740775562489e+0  	1.71150349516517594894946878196929e-9
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxasin(z) = 	1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxasin(z) = 	1.57079632520492806608316320181204e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	1.57079632609346950256128025709158e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxasin(z) = 	1.57079632564424463013989871050699e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxasin(z) = 	1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxasin(z) = 	1.57079632679489661923132169163975e+0  	-3.85305073934927544691643733742842e-8
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	-1.57079632664068896702153708483507e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxasin(z) = 	-1.57079627847684935223168121510817e+0  	4.83132107367748142468600464941897e-8
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	1.14367827643963392649213782191763e-10
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	1.21901599661363420920613616271139e-10
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxasin(z) = 	-1.57079632430501139466212660058374e+0  	-2.81069180657146950450949369938233e-9
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxasin(z) = 	-1.57079630084406346765619673646478e+0  	-2.59417490015783367563912075301276e-8
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxasin(z) = 	0.00000000000000000000000000000000e-4932  	8.81373587019543025232609324979791e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxasin(z) = 	3.96672762110029430238385670252546e-2198  	8.81373587019543025232609324979791e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxasin(z) = 	3.53461466711719510926444340733131e-18  	8.81373587019543025615931911061019e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxasin(z) = 	0.00000000000000000000000000000000e-4932  	8.81373587019543025234872773786369e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxasin(z) = 	-4.63444855457475111091046011377039e-20  	8.81373587019543030958830749706572e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxasin(z) = 	-5.02031672506825011374111480205894e-2200  	8.81373587019543025232609324979791e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxasin(z) = 	-4.83887312501578201901276247600597e-17  	8.81373587019543025222992672755636e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxasin(z) = 	0.00000000000000000000000000000000e-4932  	8.81373587019542948149004061660646e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxasin(z) = 	3.06382297220319176880164334506344e-16  	8.81373587019543025082490555333908e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxasin(z) = 	0.00000000000000000000000000000000e+0  	-8.81373587019543025232609324979791e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxasin(z) = 	1.96306984593009323724162411807058e-2111  	-8.81373587019543025232609324979791e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxasin(z) = 	2.35615050559169500646190042355772e-17  	-8.81373587019543024989011038861008e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxasin(z) = 	0.00000000000000000000000000000000e+0  	-8.81373587019543025230487984846410e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxasin(z) = 	-5.34155533576129865411052155581737e-17  	-8.81373587019543025223580278490803e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxasin(z) = 	-1.49595510627825994262242633246512e-2203  	-8.81373587019543025232609324979791e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxasin(z) = 	-4.23719596490414872857482676713472e-16  	-8.81373587019543025864204173003491e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxasin(z) = 	0.00000000000000000000000000000000e+0  	-8.81373587019543025237672916639869e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxasin(z) = 	6.12573675609717985836849438001442e-19  	-8.81373587019543025294587234350793e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	1.57079632679489661923132169163975e+0  	-4.32080500507931642178572338596806e+3
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	4.52713524369265702346500523521065e+3
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxasin(z) = 	-0.00000000000000000000000000000000e+0  	4.64163204246955746623187397808978e+3
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxasin(z) = 	0.00000000000000000000000000000000e+0  	-4.63270737982417576297363093306680e+3
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxasin(z) = 	2.21809345415886194502100946691642e-19  	4.31342259946339938398139043056709e+3
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	4.48271662978128796495011433394440e+3
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	-5.32147278763051846646398439231689e+3
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxasin(z) = 	2.03868317010775472234424662135973e-89  	-4.47596921153490526493129157770164e+3
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	1.57079632679489661923132169163975e+0  	-2.99421498124804669738168202192493e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxasin(z) = 	1.01609836063110273315267887583313e+0  	3.14607971441311583548055894215930e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxasin(z) = 	0.00000000000000000000000000000000e-4930  	3.58428348426065560396839560061604e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxasin(z) = 	-5.55938769862396875219973536274315e-1  	3.89546451524207868004792854982204e+0
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	3.58206610831763153148102511357533e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxasin(z) = 	-5.24760921333366899063474379292684e-1  	-4.33373351111893455618437386238840e+0
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxasin(z) = 	0.00000000000000000000000000000000e+0  	-4.47718414073456115123640995665135e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxasin(z) = 	3.54124369936612999514667669083657e-1  	-4.52360099327847577931228182181974e+0
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	1.57079632679489661923132169163975e+0  	-7.60091208604125417408124323995500e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxasin(z) = 	1.01775405459294542019592406808788e+0  	7.75207481180884383469861818404337e+0
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxasin(z) = 	-0.00000000000000000000000000000000e+0  	5.88610557039425744526142607748437e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxasin(z) = 	-1.41134337937270567676768859603560e+0  	7.87466501849215791282473106996163e+0
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxasin(z) = 	-1.57079632679489661923132169163975e+0  	8.18800976799369186272650771187112e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxasin(z) = 	-5.24907758561879139728403516216131e-1  	-6.63623535715674539207629079569489e+0
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxasin(z) = 	3.54200946507630265711812801105975e-1  	-9.12868177857066149797369711607171e+0

Test of cxacos
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxacos(z) = 	1.57079632679489661914352969163975e+0  	-9.77319999999998509731697648740884e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxacos(z) = 	1.57079632679489661968724169163975e+0  	-8.22190000000000004256018449090434e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxacos(z) = 	1.57079632679489663152232169163975e+0  	7.33420000000000186946777087035168e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxacos(z) = 	1.57079632679489661883215169163975e+0  	8.22150000000001534042422738242334e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	0.00000000000000000000000000000000e+0  	4.44072066223490349345311091521187e-10
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxacos(z) = 	1.41980428720391393601486321191917e-9  	-1.71150349516517594894946743381833e-9
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxacos(z) = 	1.85757906964952638594199068087148e-649  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxacos(z) = 	1.58996855314815848982770778995538e-9  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	7.01427116670041434548172246648149e-10  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxacos(z) = 	1.15065198909142298113276533593908e-9  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxacos(z) = 	8.18547494040510761125339709507446e-651  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxacos(z) = 	-5.33210020544852243086059693983464e-1344  	3.85305073934927544691643753965106e-8
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	3.14159265343558558625285877647482e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxacos(z) = 	3.14159260527174597146300290674792e+0  	-4.83132107367748142468600460127072e-8
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	-1.14367827643963392649212145151311e-10
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	-1.21901599661363420920612845899161e-10
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxacos(z) = 	3.14159265109990801389344829222350e+0  	2.81069180657146950450949514382979e-9
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxacos(z) = 	3.14159262763896008688751842810453e+0  	2.59417490015783367563912088782786e-8
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-8.81373587019543025232609324979791e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-8.81373587019543025232609324979791e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxacos(z) = 	1.57079632679489661569670702452256e+0  	-8.81373587019543025615931911061019e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-8.81373587019543025234872773786370e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxacos(z) = 	1.57079632679489661927766617718550e+0  	-8.81373587019543030958830749706572e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-8.81373587019543025232609324979791e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxacos(z) = 	1.57079632679489666762005294179757e+0  	-8.81373587019543025222992672755638e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-8.81373587019542948149004061660645e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxacos(z) = 	1.57079632679489631284902447132057e+0  	-8.81373587019543025082490555333902e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	8.81373587019543025232609324979791e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	8.81373587019543025232609324979791e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxacos(z) = 	1.57079632679489659566981663572280e+0  	8.81373587019543024989011038861008e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	8.81373587019543025230487984846411e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxacos(z) = 	1.57079632679489667264687504925274e+0  	8.81373587019543025223580278490800e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	8.81373587019543025232609324979791e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxacos(z) = 	1.57079632679489704295091818205462e+0  	8.81373587019543025864204173003496e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	8.81373587019543025237672916639868e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxacos(z) = 	1.57079632679489661861874801603003e+0  	8.81373587019543025294587234350791e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	-0.00000000000000000000000000000000e+0  	4.32080500507931642178572338596806e+3
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	-4.52713524369265702346500523521065e+3
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-4.64163204246955746623187397808978e+3
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	4.63270737982417576297363093306680e+3
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxacos(z) = 	1.57079632679489661900951234622387e+0  	-4.31342259946339938398139043056709e+3
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	-4.48271662978128796495011433394440e+3
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	5.32147278763051846646398439231689e+3
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	4.47596921153490526493129157770164e+3
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	0.00000000000000000000000000000000e-4930  	2.99421498124804669738168202192491e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxacos(z) = 	5.54697966163793886078642815806552e-1  	-3.14607971441311583548055894215930e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-3.58428348426065560396839560061615e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxacos(z) = 	2.12673509665729349445129522791405e+0  	-3.89546451524207868004792854982317e+0
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	-3.58206610831763153148102511357538e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxacos(z) = 	2.09555724812826351829479607093162e+0  	4.33373351111893455618437386238427e+0
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	4.47718414073456115123640995665048e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxacos(z) = 	1.21667195685828361971665402255801e+0  	4.52360099327847577931228182181538e+0
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	-0.00000000000000000000000000000000e+0  	7.60091208604125417408124323995500e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxacos(z) = 	5.53042272201951199035397623551874e-1  	-7.75207481180884383469861818404337e+0
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxacos(z) = 	1.57079632679489661923132169163975e+0  	-5.88610557039425744526142607748437e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxacos(z) = 	2.98213970616760229599901028767535e+0  	-7.87466501849215791282473106996163e+0
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxacos(z) = 	3.14159265358979323846264338327950e+0  	-8.18800976799369186272650771187112e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxacos(z) = 	2.09570408535677575895972520785588e+0  	6.63623535715674539207629079569489e+0
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxacos(z) = 	1.21659538028726635351950889053378e+0  	9.12868177857066149797369711607171e+0

Test of cxasinh
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxasinh(z) = 	8.77919999999997906198395519872265e-20  	9.77319999999999999999999999999977e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxasinh(z) = 	-4.55920000000001519096575371717153e-19  	8.22189999999999999999999999999983e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxasinh(z) = 	-1.22910000000000018523077419597896e-17  	-7.33419999999999999999999999999985e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxasinh(z) = 	3.99169999999999875877567138640924e-19  	-8.22149999999999999999999999999983e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	8.81373587019543025232609324979791e-1  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	8.81373587019543025302330053604786e-1  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxasinh(z) = 	8.81373587019543025555544991947688e-1  	1.71826947828331048390168496454875e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxasinh(z) = 	8.81373587019543025232609324979791e-1  	2.43994265916230088769755355587525e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxasinh(z) = 	8.81373587019543024338826353559979e-1  	3.30056232254444788173177661683987e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	8.81373587019543025058661056807885e-1  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxasinh(z) = 	8.81373587019543024764504635834280e-1  	-3.59090036689964429435339594551602e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxasinh(z) = 	8.81373587019543025232609324979791e-1  	-4.73775685530610572299053739496762e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxasinh(z) = 	8.81373587019543550117972999753922e-1  	-3.77036406796478865548711839804170e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	-8.81373587019543025232609324979791e-1  	0.00000000000000000000000000000000e-4932
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	-8.81373587019543025224201825351467e-1  	0.00000000000000000000000000000000e-4932
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxasinh(z) = 	-8.81373587019543025067372612353071e-1  	1.65067007000187654115419536987621e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxasinh(z) = 	-8.81373587019543025232609324979791e-1  	3.22447763288877536602029037563052e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxasinh(z) = 	-8.81373587019543025237233803328750e-1  	5.25606612591584505835916897615650e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	-8.81373587019543025237863128364007e-1  	0.00000000000000000000000000000000e-4932
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxasinh(z) = 	-8.81373587019543025833812723548029e-1  	-4.94854538677781553791072822721821e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxasinh(z) = 	-8.81373587019543025232609324979791e-1  	-3.53622687057830043857813463799483e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxasinh(z) = 	-8.81373587019543025065944256654190e-1  	-4.76031356162595658957992728336358e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxasinh(z) = 	2.36010782800638093852159651971599e-9  	1.57079632467690016979685746513326e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxasinh(z) = 	8.00124990235852843665287462042112e-11  	1.57079632679489661923132169163975e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxasinh(z) = 	-4.02448322393867884084498881723795e-9  	1.57079632677861105004068747822618e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxasinh(z) = 	-8.27154165506131657960320060232053e-9  	1.57079631852171093586759370570069e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	1.57079631202921478600736260875249e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxasinh(z) = 	2.08105202244089168069341852591527e-8  	1.57079630597417732304627030260469e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	-1.57079632679489661923132169163975e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	-1.57079632679489661923132169163975e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxasinh(z) = 	5.74267192298863782704632905735702e-9  	-1.57079632099254530143936373321590e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	-1.57079632671743659099694132135162e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxasinh(z) = 	-8.69069802025117878767270805891998e-9  	-1.57079631810272945107366984070417e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxasinh(z) = 	0.00000000000000000000000000000000e+0  	-1.57079632679489661923132169163975e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxasinh(z) = 	-2.44974258996200517643194024979537e-8  	-1.57079630233395927604319963640679e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxasinh(z) = 	-1.19674558699830883884380319897279e-10  	-1.57079632679489661923132169163975e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxasinh(z) = 	9.78970254694527441912631278357824e-10  	-1.57079632590997698470134441011724e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	4.32080500507931642178572338596806e+3  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	-4.52713524369265702346500523521065e+3  	-0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxasinh(z) = 	4.64163204246955746623187397808978e+3  	1.57079632679489661923132169163975e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxasinh(z) = 	-4.63270737982417576297363093306680e+3  	-1.57079632679489661923132169163975e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxasinh(z) = 	4.31342259946339938398139043056709e+3  	1.57079632679489661900951234622386e+0
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxasinh(z) = 	-4.48271662978128796495011433394440e+3  	2.29694230419219802220469574713341e-112
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxasinh(z) = 	-5.32147278763051846646398439231689e+3  	-8.92173741945004673127059865217177e-113
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxasinh(z) = 	4.47596921153490526493129157770164e+3  	-1.57079632679489661923132169163975e+0
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	2.99920522287400229987703918761449e+0  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxasinh(z) = 	3.14773583068738672478544942275635e+0  	5.51394454415403526120667838719062e-1
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxasinh(z) = 	3.58274025225608460734116035719557e+0  	1.57079632679489661923132169163975e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxasinh(z) = 	-3.89509861575484661807235467509158e+0  	1.01411563834406850059043506368195e+0
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	-3.58361141923479880308323219075935e+0  	0.00000000000000000000000000000000e-4930
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxasinh(z) = 	-4.33356216970929866957829400430197e+0  	-1.04573682759625091865211734392921e+0
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxasinh(z) = 	-4.47692573063071298420136726954508e+0  	-1.57079632679489661923132169163975e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxasinh(z) = 	4.52342216762501430051403043720851e+0  	-1.21651874729859515850004155165870e+0
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	7.60091258603137777740254109913007e+0  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxasinh(z) = 	7.75207497742461114306109632532273e+0  	5.53041941850037215897781302718926e-1
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxasinh(z) = 	5.88609013810483618939403066521643e+0  	1.57079632679489661923132169163975e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxasinh(z) = 	-7.87466529310059149372261788509553e+0  	1.59452856753218017167586694660060e-1
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxasinh(z) = 	-8.18800992252447606468828394309875e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxasinh(z) = 	-6.63623364374225938023389939022563e+0  	-1.04588558246700779702317717632289e+0
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxasinh(z) = 	9.12868176068809557897436968006321e+0  	-1.21659536496631130983792157635872e+0

Test of cxacosh
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxacosh(z) = 	9.77319999999998509731697648740884e-20  	1.57079632679489661914352969163975e+0
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxacosh(z) = 	-8.22190000000000167960063722005481e-18  	-1.57079632679489661968724169163975e+0
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxacosh(z) = 	-7.33420000000000186946777087035168e-18  	1.57079632679489663152232169163975e+0
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxacosh(z) = 	8.22149999999999897001970009091859e-19  	-1.57079632679489661883215169163975e+0
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	4.44072066223490349345309743370226e-10  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxacosh(z) = 	1.71150349516517594894946743381833e-9  	1.41980428720391393601486321191917e-9
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	1.85757906964952638594199068087148e-649
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	1.58996855314815848982770778995538e-9
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	7.01427116670041434548172246648149e-10
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	-1.15065198909142298113276533593908e-9
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	-8.18547494040510761125339709507445e-651
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxacosh(z) = 	3.85305073934927544691643733742842e-8  	-1.38386446499287815404461491133362e-1336
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	3.14159265358979323846264338327950e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	3.14159265343558558625285877647482e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxacosh(z) = 	-4.83132107367748142468600464941897e-8  	-3.14159260527174597146300290674792e+0
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	-3.14159265358979323846264338327950e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxacosh(z) = 	-1.14367827643963392649213782191763e-10  	-3.14159265358979323846264338327950e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	-1.21901599661363420920613616271139e-10  	3.14159265358979323846264338327950e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxacosh(z) = 	-2.81069180657146950450949514382979e-9  	3.14159265109990801389344829222350e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxacosh(z) = 	0.00000000000000000000000000000000e+0  	3.14159265358979323846264338327950e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxacosh(z) = 	-2.59417490015783367563912088782786e-8  	3.14159262763896008688751842810453e+0
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxacosh(z) = 	8.81373587019543025232609324979791e-1  	1.57079632679489661923132169163975e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxacosh(z) = 	8.81373587019543025232609324979791e-1  	1.57079632679489661923132169163975e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxacosh(z) = 	8.81373587019543025615931911061020e-1  	1.57079632679489661569670702452256e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxacosh(z) = 	8.81373587019543025234872773786370e-1  	1.57079632679489661923132169163975e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxacosh(z) = 	-8.81373587019543030958830749706570e-1  	-1.57079632679489661927766617718550e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxacosh(z) = 	-8.81373587019543025232609324979791e-1  	-1.57079632679489661923132169163975e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxacosh(z) = 	-8.81373587019543025222992672755636e-1  	-1.57079632679489666762005294179757e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxacosh(z) = 	8.81373587019542948149004061660645e-1  	1.57079632679489661923132169163975e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxacosh(z) = 	8.81373587019543025082490555333902e-1  	1.57079632679489631284902447132057e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxacosh(z) = 	-8.81373587019543025232609324979791e-1  	1.57079632679489661923132169163975e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxacosh(z) = 	8.81373587019543025232609324979791e-1  	-1.57079632679489661923132169163975e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxacosh(z) = 	8.81373587019543024989011038861008e-1  	-1.57079632679489659566981663572280e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxacosh(z) = 	-8.81373587019543025230487984846411e-1  	1.57079632679489661923132169163975e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxacosh(z) = 	-8.81373587019543025223580278490805e-1  	1.57079632679489667264687504925274e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxacosh(z) = 	-8.81373587019543025232609324979791e-1  	1.57079632679489661923132169163975e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxacosh(z) = 	-8.81373587019543025864204173003492e-1  	1.57079632679489704295091818205462e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxacosh(z) = 	-8.81373587019543025237672916639868e-1  	1.57079632679489661923132169163975e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxacosh(z) = 	8.81373587019543025294587234350793e-1  	-1.57079632679489661861874801603003e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	4.32080500507931642178572338596806e+3  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	-4.52713524369265702346500523521065e+3  	-3.14159265358979323846264338327950e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxacosh(z) = 	4.64163204246955746623187397808978e+3  	1.57079632679489661923132169163975e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxacosh(z) = 	-4.63270737982417576297363093306680e+3  	1.57079632679489661923132169163975e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxacosh(z) = 	4.31342259946339938398139043056709e+3  	1.57079632679489661900951234622386e+0
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxacosh(z) = 	-4.48271662978128796495011433394440e+3  	-3.14159265358979323846264338327950e+0
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxacosh(z) = 	-5.32147278763051846646398439231689e+3  	3.14159265358979323846264338327950e+0
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxacosh(z) = 	4.47596921153490526493129157770164e+3  	-1.57079632679489661923132169163975e+0
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	2.99421498124804669738168202192493e+0  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxacosh(z) = 	3.14607971441311583548055894215930e+0  	5.54697966163793886078642815806551e-1
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxacosh(z) = 	3.58428348426065560396839560061615e+0  	1.57079632679489661923132169163975e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxacosh(z) = 	-3.89546451524207868004792854982544e+0  	-2.12673509665729349445129522791261e+0
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	-3.58206610831763153148102511357533e+0  	3.14159265358979323846264338327950e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxacosh(z) = 	-4.33373351111893455618437386239093e+0  	2.09555724812826351829479607092950e+0
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxacosh(z) = 	-4.47718414073456115123640995665048e+0  	1.57079632679489661923132169163975e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxacosh(z) = 	4.52360099327847577931228182181974e+0  	-1.21667195685828361971665402255609e+0
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	7.60091208604125417408124323995500e+0  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxacosh(z) = 	7.75207481180884383469861818404337e+0  	5.53042272201951199035397623551873e-1
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxacosh(z) = 	5.88610557039425744526142607748437e+0  	1.57079632679489661923132169163975e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxacosh(z) = 	-7.87466501849215791282473106996163e+0  	-2.98213970616760229599901028767535e+0
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxacosh(z) = 	-8.18800976799369186272650771187112e+0  	-3.14159265358979323846264338327950e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxacosh(z) = 	-6.63623535715674539207629079569489e+0  	2.09570408535677575895972520785588e+0
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxacosh(z) = 	9.12868177857066149797369711607171e+0  	-1.21659538028726635351950889053378e+0

*/
