/*
   Copyright (C)  2000    Daniel A. Atkinson  <DanAtk@aol.com>
   Copyright (C)  2004    Ivano Primi  <ivprimi@libero.it>    

   This file is part of the HPA Library.

   The HPA Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The HPA Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the HPA Library; if not, write to the Free
   Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
   02110-1301 USA.
*/

/*
    Test:  cxsum   cxsub    cxmul  cxdiv  
           cxgdiv  cxidiv  cxgmod  cxmod  cxpow

    Uses:  All the tested functions + strtocx  xpr_asprint

    Input file:  data/cxbin.dat
*/

#include<stdio.h>
#include<stdlib.h>
#include<ctype.h>
#include"cxpre.h"

#define DECD 32

int
get_line (char *buffer, int size, FILE * fp)
{
  if ((fgets (buffer, size, fp)))
    {
      int i;

      for (i = 0; buffer[i] != '\0'; i++);
      if (i > 0 && buffer[i - 1] == '\n')
	buffer[--i] = '\0';
      return i;
    }
  else
    return -1;
}

typedef struct cxpr (*binop) (struct cxpr, struct cxpr);
extern int xErrNo;

int
main (int na, char **av)
{
  struct cxpr z1, z2, w;
  FILE *fp;
  char num[1024], *endptr, *re, *im;
  const char *nlist[] = {
    "cxsum", "cxsub", "cxmul", "cxdiv",
    "cxgdiv", "cxidiv", "cxgmod", "cxmod",
    "cxpow", NULL
  };
  binop oplist[] = {
    cxsum, cxsub, cxmul, cxdiv,
    cxgdiv, cxidiv, cxgmod, cxmod,
    cxpow
  };
  int i;

  if (na != 2)
    {
      printf ("para: input_file\n");
      exit (-1);
    }
  fp = fopen (*++av, "r");
  for (i = 0; nlist[i] != NULL; i++)
    {
      printf ("Test of %s\n", nlist[i]);
      while (get_line (num, 1024, fp) != -1)
	{
	  printf ("*string in : \t%s\n", num);
	  z1 = strtocx (num, &endptr);
	  re = xpr_asprint (z1.re, 1, 0, DECD);
	  im = xpr_asprint (z1.im, 1, 0, DECD);
	  printf (" z1 =  \t\t%s  \t%s\n", re, im);
	  free ((void *) re);
	  free ((void *) im);
	  for (; isspace (*endptr) != 0; endptr++);
	  z2 = strtocx (endptr, NULL);
	  re = xpr_asprint (z2.re, 1, 0, DECD);
	  im = xpr_asprint (z2.im, 1, 0, DECD);
	  printf (" z2 =  \t\t%s  \t%s\n", re, im);
	  free ((void *) re);
	  free ((void *) im);
	  xErrNo = 0;
	  w = (*oplist[i]) (z1, z2);
	  switch (xErrNo)
	    {
	    case XENONE:
	      printf ("%s(z1,z2) = ", nlist[i]);
	      re = xpr_asprint (w.re, 1, 0, DECD);
	      im = xpr_asprint (w.im, 1, 0, DECD);
	      printf ("%s  \t%s\n", re, im);
	      free ((void *) re);
	      free ((void *) im);
	      break;
	    case XEDIV:
	      printf ("*** Division by zero\n");
	      break;
	    case XEDOM:
	      printf ("*** Out of domain\n");
	      break;
	    case XEBADEXP:
	      printf ("*** Bad exponent\n");
	      break;
	    case XFPOFLOW:
	      printf ("*** Floating point overflow\n");
	      break;
	    default:
	      printf ("*** Unknown error\n");
	    }
	}
      printf ("\n");
      rewind (fp);
    }
  fclose (fp);
  return 0;
}

/*  Test output

Test of cxsum
*string in : 	128.970870234 +73.6850340129i	-23.723000129 +5.09881720006i
 z1 =  		1.28970870234000000000000000000000e+2  	7.36850340128999999999999999999998e+1
 z2 =  		-2.37230001290000000000000000000000e+1  	5.09881720006000000000000000000000e+0
cxsum(z1,z2) = 1.05247870105000000000000000000000e+2  	7.87838512129599999999999999999998e+1
*string in : 	138.570870234 +75.4850340129i	-23.723000129e2381  +5.09881720006e3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
cxsum(z1,z2) = -2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
*string in : 	138.570870234 +75.4850340129i	-23.723000129e-2381 +5.09881720006e-3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.09881720005999999999999999997939e-3418
cxsum(z1,z2) = 1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
*string in : 	236.570870234e-2382 +375.4850340129e-3417i 	-23.723000129e-2381 +51.09881720006e-3418i
 z1 =  		2.36570870233999999999999999999333e-2380  	3.75485034012899999999999999998482e-3415
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.10988172000599999999999999997935e-3417
cxsum(z1,z2) = -6.59131055999999999999999999998557e-2383  	3.80594915732905999999999999998462e-3415
*string in : 	236.570870234e+2382 +375.4850340129e+3417i	-23.723000129e+2381 +51.09881720006e+3418i
 z1 =  		2.36570870233999999999999999999333e+2384  	3.75485034012899999999999999998482e+3419
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.10988172000599999999999999997933e+3419
cxsum(z1,z2) = 2.34198570221099999999999999999340e+2384  	8.86473206013499999999999999996414e+3419
*string in : 	-7.2100128992 -0.23422221i 	+2.2198112817 -11.9127653i
 z1 =  		-7.21001289920000000000000000000000e+0  	-2.34222209999999999999999999999998e-1
 z2 =  		2.21981128170000000000000000000000e+0  	-1.19127653000000000000000000000000e+1
cxsum(z1,z2) = -4.99020161750000000000000000000000e+0  	-1.21469875100000000000000000000000e+1
*string in : 	-99.888888888134 +3332.1128976i		0.000
 z1 =  		-9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
cxsum(z1,z2) = -9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
*string in : 	+55 -321i 	0
 z1 =  		5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
cxsum(z1,z2) = 5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
*string in : 	+55.3 +4.234i	+1.443e-3210
 z1 =  		5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
 z2 =  		1.44299999999999999999999999999452e-3210  	0.00000000000000000000000000000000e+0
cxsum(z1,z2) = 5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
*string in : 	+56.3 +6.234i	-3.443e-3201
 z1 =  		5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
 z2 =  		-3.44299999999999999999999999998696e-3201  	0.00000000000000000000000000000000e+0
cxsum(z1,z2) = 5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
*string in : 	+0.651119  -1.9872i 	+9.871e-3312i
 z1 =  		6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	9.87099999999999999999999999996133e-3312
cxsum(z1,z2) = 6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
*string in : 	-0.81273 -0.087133i	-7.12289e-3301i
 z1 =  		-8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
 z2 =  		0.00000000000000000000000000000000e+0  	-7.12288999999999999999999999997218e-3301
cxsum(z1,z2) = -8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
*string in : 	0 +0i			-45.768891 -56.98127i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		-4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
cxsum(z1,z2) = -4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
*string in : 	0 -0i			+53.8761 -32.11198i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
cxsum(z1,z2) = 5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
*string in : 	+3.2156e-3678 +6i 	+5.38761 +2.411198i
 z1 =  		3.21559999999999999999999999998601e-3678  	6.00000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxsum(z1,z2) = 5.38761000000000000000000000000000e+0  	8.41119799999999999999999999999998e+0
*string in : 	0.000 -7.1256e-3839i	+5.38761 +2.411198i
 z1 =  		0.00000000000000000000000000000000e+0  	-7.12559999999999999999999999996765e-3839
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxsum(z1,z2) = 5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
cxsum(z1,z2) = 4.40000000000000000000000000000000e+0  	-8.59999999999999999999999999999998e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3000000000000000000082i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000819999999999e+0
cxsum(z1,z2) = 4.40000000000000000000000000000000e+0  	-8.60000000000000000000819999999997e+0
*string in : 	+2.2 -4.3i	+2.20000000000000000000000871 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000870999999e+0  	-4.30000000000000000000000000000000e+0
cxsum(z1,z2) = 4.40000000000000000000000870999999e+0  	-8.59999999999999999999999999999998e+0
*string in : 	+5.5 +7.8i	+5.38761 +2.411198i
 z1 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxsum(z1,z2) = 1.08876100000000000000000000000000e+1  	1.02111980000000000000000000000000e+1
*string in : 	+5.5000000000000001829 +7.8000000000000007615i 		+5.5 +7.8i
 z1 =  		5.50000000000000018289999999999999e+0  	7.80000000000000076149999999999998e+0
 z2 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
cxsum(z1,z2) = 1.10000000000000001829000000000000e+1  	1.56000000000000007615000000000000e+1
*string in : 	-2.60 -6.54i 	-2.60 -6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxsum(z1,z2) = -5.20000000000000000000000000000000e+0  	-1.30799998000000000000000000000000e+1
*string in : 	-2.60 -6.54i	-2.59 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxsum(z1,z2) = -5.19000000000000000000000000000000e+0  	-1.30800000000000000000000000000000e+1
*string in : 	-2.60 -6.54i	-2.61 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxsum(z1,z2) = -5.21000000000000000000000000000000e+0  	-1.30800000000000000000000000000000e+1
*string in : 	-2.60 -6.54i	-2.60 -6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxsum(z1,z2) = -5.20000000000000000000000000000000e+0  	-1.30800000180000000000000000000000e+1
*string in : 	-4.55 -8.71i	-4.550000012 -8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxsum(z1,z2) = -9.10000001199999999999999999999997e+0  	-1.74200000000080000000000000000000e+1
*string in : 	-4.55 -8.71i	-4.54 -8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxsum(z1,z2) = -9.08999999999999999999999999999998e+0  	-1.74000000000000000000000000000000e+1
*string in : 	+2.60 +6.54i	+2.60 +6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxsum(z1,z2) = 5.20000000000000000000000000000000e+0  	1.30799998000000000000000000000000e+1
*string in : 	+2.60 +6.54i	+2.59 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxsum(z1,z2) = 5.19000000000000000000000000000000e+0  	1.30800000000000000000000000000000e+1
*string in : 	+2.60 +6.54i	+2.61 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxsum(z1,z2) = 5.21000000000000000000000000000000e+0  	1.30800000000000000000000000000000e+1
*string in : 	+2.60 +6.54i	+2.60 +6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxsum(z1,z2) = 5.20000000000000000000000000000000e+0  	1.30800000180000000000000000000000e+1
*string in : 	+4.55 +8.71i	+4.550000012 +8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxsum(z1,z2) = 9.10000001199999999999999999999997e+0  	1.74200000000080000000000000000000e+1
*string in : 	+4.55 +8.71i	+4.54 +8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxsum(z1,z2) = 9.08999999999999999999999999999998e+0  	1.74000000000000000000000000000000e+1
*string in : 	+2.60 -6.54i	+2.60 -6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxsum(z1,z2) = 5.20000000000000000000000000000000e+0  	-1.30799998000000000000000000000000e+1
*string in : 	+2.60 -6.54i	+2.59 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxsum(z1,z2) = 5.19000000000000000000000000000000e+0  	-1.30800000000000000000000000000000e+1
*string in : 	+2.60 -6.54i	+2.61 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxsum(z1,z2) = 5.21000000000000000000000000000000e+0  	-1.30800000000000000000000000000000e+1
*string in : 	+2.60 -6.54i	+2.60 -6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxsum(z1,z2) = 5.20000000000000000000000000000000e+0  	-1.30800000180000000000000000000000e+1
*string in : 	+4.55 -8.71i	+4.550000012 -8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxsum(z1,z2) = 9.10000001199999999999999999999997e+0  	-1.74200000000080000000000000000000e+1
*string in : 	+4.55 -8.71i	+4.54 -8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxsum(z1,z2) = 9.08999999999999999999999999999998e+0  	-1.74000000000000000000000000000000e+1
*string in : 	-2.60 +6.54i	-2.60 +6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxsum(z1,z2) = -5.20000000000000000000000000000000e+0  	1.30799998000000000000000000000000e+1
*string in : 	-2.60 +6.54i	-2.59 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxsum(z1,z2) = -5.19000000000000000000000000000000e+0  	1.30800000000000000000000000000000e+1
*string in : 	-2.60 +6.54i	-2.61 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxsum(z1,z2) = -5.21000000000000000000000000000000e+0  	1.30800000000000000000000000000000e+1
*string in : 	-2.60 +6.54i	-2.60 +6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxsum(z1,z2) = -5.20000000000000000000000000000000e+0  	1.30800000180000000000000000000000e+1
*string in : 	-4.55 +8.71i	-4.550000012 +8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxsum(z1,z2) = -9.10000001199999999999999999999997e+0  	1.74200000000080000000000000000000e+1
*string in : 	-4.55 +8.71i	-4.54 +8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxsum(z1,z2) = -9.08999999999999999999999999999998e+0  	1.74000000000000000000000000000000e+1
*string in : 	23.5  -6.5i	12.0 +8.5i
 z1 =  		2.35000000000000000000000000000000e+1  	-6.50000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.49999999999999999999999999999998e+0
cxsum(z1,z2) = 3.54999999999999999999999999999999e+1  	2.00000000000000000000000000000000e+0
*string in : 	23    -6i	12 +8i
 z1 =  		2.30000000000000000000000000000000e+1  	-6.00000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.00000000000000000000000000000000e+0
cxsum(z1,z2) = 3.50000000000000000000000000000000e+1  	2.00000000000000000000000000000000e+0
*string in : 	23.49999999999999999   -6.4999999999999999999i 	11.99999999999999999  +8.4999999999999999999i
 z1 =  		2.34999999999999999899999999999999e+1  	-6.49999999999999999989999999999999e+0
 z2 =  		1.19999999999999999900000000000000e+1  	8.49999999999999999989999999999997e+0
cxsum(z1,z2) = 3.54999999999999999799999999999999e+1  	2.00000000000000000000000000000000e+0
*string in : 	22.9999999999999999999 -5.9999999999999999999i	11.999999999999999999 +7.99999999999999999999i
 z1 =  		2.29999999999999999998999999999999e+1  	-5.99999999999999999989999999999998e+0
 z2 =  		1.19999999999999999990000000000000e+1  	7.99999999999999999998999999999998e+0
cxsum(z1,z2) = 3.49999999999999999988999999999999e+1  	2.00000000000000000009000000000000e+0
*string in : 	23.500001 -6.500001i				12.0 +8.50000000001i
 z1 =  		2.35000010000000000000000000000000e+1  	-6.50000100000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.50000000000999999999999999999998e+0
cxsum(z1,z2) = 3.55000010000000000000000000000000e+1  	1.99999900001000000000000000000000e+0
*string in : 	23.00000000000100 -6.00000000000000002i 	12.00000000000003 +8.0000000000000003i
 z1 =  		2.30000000000010000000000000000000e+1  	-6.00000000000000002000000000000000e+0
 z2 =  		1.20000000000000300000000000000000e+1  	8.00000000000000029999999999999999e+0
cxsum(z1,z2) = 3.50000000000010300000000000000000e+1  	2.00000000000000028000000000000000e+0
*string in : 	+4.55 -6i 	-0.50 +0.5i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999999999999999999999e-1  	5.00000000000000000000000000000000e-1
cxsum(z1,z2) = 4.05000000000000000000000000000000e+0  	-5.50000000000000000000000000000000e+0
*string in : 	+4.55 -6i	-0.49999999999999999 -0.49999999999999978i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999989999999999999990e-1  	-4.99999999999999779999999999999990e-1
cxsum(z1,z2) = 4.05000000000000001000000000000001e+0  	-6.49999999999999977999999999999999e+0
*string in : 	+7.60 -8i	+0.5000000000000023  -0.500000000000000000000123i
 z1 =  		7.60000000000000000000000000000000e+0  	-8.00000000000000000000000000000000e+0
 z2 =  		5.00000000000002299999999999999991e-1  	-5.00000000000000000000122999999986e-1
cxsum(z1,z2) = 8.10000000000000229999999999999997e+0  	-8.50000000000000000000012299999998e+0
*string in : 	+77.60 -788i	+21.5000000000000023 +12.500000000000000000000123i
 z1 =  		7.75999999999999999999999999999998e+1  	-7.87999999999999999999999999999998e+2
 z2 =  		2.15000000000000023000000000000000e+1  	1.25000000000000000000001230000000e+1
cxsum(z1,z2) = 9.91000000000000022999999999999997e+1  	-7.75499999999999999999999876999997e+2
*string in : 	0.00 +0.0i	0.0 +765.21i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2
cxsum(z1,z2) = 0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2

Test of cxsub
*string in : 	128.970870234 +73.6850340129i	-23.723000129 +5.09881720006i
 z1 =  		1.28970870234000000000000000000000e+2  	7.36850340128999999999999999999998e+1
 z2 =  		-2.37230001290000000000000000000000e+1  	5.09881720006000000000000000000000e+0
cxsub(z1,z2) = 1.52693870363000000000000000000000e+2  	6.85862168128399999999999999999998e+1
*string in : 	138.570870234 +75.4850340129i	-23.723000129e2381  +5.09881720006e3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
cxsub(z1,z2) = 2.37230001289999999999999999999331e+2382  	-5.09881720005999999999999999997939e+3418
*string in : 	138.570870234 +75.4850340129i	-23.723000129e-2381 +5.09881720006e-3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.09881720005999999999999999997939e-3418
cxsub(z1,z2) = 1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
*string in : 	236.570870234e-2382 +375.4850340129e-3417i 	-23.723000129e-2381 +51.09881720006e-3418i
 z1 =  		2.36570870233999999999999999999333e-2380  	3.75485034012899999999999999998482e-3415
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.10988172000599999999999999997935e-3417
cxsub(z1,z2) = 4.73800871523999999999999999998665e-2380  	3.70375152292893999999999999998503e-3415
*string in : 	236.570870234e+2382 +375.4850340129e+3417i	-23.723000129e+2381 +51.09881720006e+3418i
 z1 =  		2.36570870233999999999999999999333e+2384  	3.75485034012899999999999999998482e+3419
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.10988172000599999999999999997933e+3419
cxsub(z1,z2) = 2.38943170246899999999999999999326e+2384  	-1.35503137987699999999999999999452e+3419
*string in : 	-7.2100128992 -0.23422221i 	+2.2198112817 -11.9127653i
 z1 =  		-7.21001289920000000000000000000000e+0  	-2.34222209999999999999999999999998e-1
 z2 =  		2.21981128170000000000000000000000e+0  	-1.19127653000000000000000000000000e+1
cxsub(z1,z2) = -9.42982418089999999999999999999998e+0  	1.16785430900000000000000000000000e+1
*string in : 	-99.888888888134 +3332.1128976i		0.000
 z1 =  		-9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
cxsub(z1,z2) = -9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
*string in : 	+55 -321i 	0
 z1 =  		5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
cxsub(z1,z2) = 5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
*string in : 	+55.3 +4.234i	+1.443e-3210
 z1 =  		5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
 z2 =  		1.44299999999999999999999999999452e-3210  	0.00000000000000000000000000000000e+0
cxsub(z1,z2) = 5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
*string in : 	+56.3 +6.234i	-3.443e-3201
 z1 =  		5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
 z2 =  		-3.44299999999999999999999999998696e-3201  	0.00000000000000000000000000000000e+0
cxsub(z1,z2) = 5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
*string in : 	+0.651119  -1.9872i 	+9.871e-3312i
 z1 =  		6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	9.87099999999999999999999999996133e-3312
cxsub(z1,z2) = 6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
*string in : 	-0.81273 -0.087133i	-7.12289e-3301i
 z1 =  		-8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
 z2 =  		0.00000000000000000000000000000000e+0  	-7.12288999999999999999999999997218e-3301
cxsub(z1,z2) = -8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
*string in : 	0 +0i			-45.768891 -56.98127i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		-4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
cxsub(z1,z2) = 4.57688909999999999999999999999999e+1  	5.69812699999999999999999999999999e+1
*string in : 	0 -0i			+53.8761 -32.11198i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
cxsub(z1,z2) = -5.38760999999999999999999999999999e+1  	3.21119800000000000000000000000000e+1
*string in : 	+3.2156e-3678 +6i 	+5.38761 +2.411198i
 z1 =  		3.21559999999999999999999999998601e-3678  	6.00000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxsub(z1,z2) = -5.38761000000000000000000000000000e+0  	3.58880200000000000000000000000000e+0
*string in : 	0.000 -7.1256e-3839i	+5.38761 +2.411198i
 z1 =  		0.00000000000000000000000000000000e+0  	-7.12559999999999999999999999996765e-3839
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxsub(z1,z2) = -5.38761000000000000000000000000000e+0  	-2.41119800000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3000000000000000000082i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000819999999999e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	8.19999999999199072104636850947636e-21
*string in : 	+2.2 -4.3i	+2.20000000000000000000000871 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000870999999e+0  	-4.30000000000000000000000000000000e+0
cxsub(z1,z2) = -8.70999999425334308544899607935334e-24  	0.00000000000000000000000000000000e+0
*string in : 	+5.5 +7.8i	+5.38761 +2.411198i
 z1 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxsub(z1,z2) = 1.12390000000000000000000000000002e-1  	5.38880200000000000000000000000000e+0
*string in : 	+5.5000000000000001829 +7.8000000000000007615i 		+5.5 +7.8i
 z1 =  		5.50000000000000018289999999999999e+0  	7.80000000000000076149999999999998e+0
 z2 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
cxsub(z1,z2) = 1.82899999999999989742838433617051e-16  	7.61499999999999983220147975266869e-16
*string in : 	-2.60 -6.54i 	-2.60 -6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	-2.00000000000000000000000002726343e-7
*string in : 	-2.60 -6.54i	-2.59 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxsub(z1,z2) = -1.00000000000000000000000000000006e-2  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.61 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxsub(z1,z2) = 9.99999999999999999999999999999982e-3  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.60 -6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	1.79999999999999999999999957772134e-8
*string in : 	-4.55 -8.71i	-4.550000012 -8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxsub(z1,z2) = 1.19999999999999999999999946169023e-8  	7.99999999999999999998999082002114e-12
*string in : 	-4.55 -8.71i	-4.54 -8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxsub(z1,z2) = -1.00000000000000000000000000000014e-2  	-1.99999999999999999999999999999996e-2
*string in : 	+2.60 +6.54i	+2.60 +6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	2.00000000000000000000000002726343e-7
*string in : 	+2.60 +6.54i	+2.59 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxsub(z1,z2) = 1.00000000000000000000000000000006e-2  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.61 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxsub(z1,z2) = -9.99999999999999999999999999999982e-3  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.60 +6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	-1.79999999999999999999999957772134e-8
*string in : 	+4.55 +8.71i	+4.550000012 +8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxsub(z1,z2) = -1.19999999999999999999999946169023e-8  	-7.99999999999999999998999082002114e-12
*string in : 	+4.55 +8.71i	+4.54 +8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxsub(z1,z2) = 1.00000000000000000000000000000014e-2  	1.99999999999999999999999999999996e-2
*string in : 	+2.60 -6.54i	+2.60 -6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	-2.00000000000000000000000002726343e-7
*string in : 	+2.60 -6.54i	+2.59 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxsub(z1,z2) = 1.00000000000000000000000000000006e-2  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.61 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxsub(z1,z2) = -9.99999999999999999999999999999982e-3  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.60 -6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	1.79999999999999999999999957772134e-8
*string in : 	+4.55 -8.71i	+4.550000012 -8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxsub(z1,z2) = -1.19999999999999999999999946169023e-8  	7.99999999999999999998999082002114e-12
*string in : 	+4.55 -8.71i	+4.54 -8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxsub(z1,z2) = 1.00000000000000000000000000000014e-2  	-1.99999999999999999999999999999996e-2
*string in : 	-2.60 +6.54i	-2.60 +6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	2.00000000000000000000000002726343e-7
*string in : 	-2.60 +6.54i	-2.59 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxsub(z1,z2) = -1.00000000000000000000000000000006e-2  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.61 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxsub(z1,z2) = 9.99999999999999999999999999999982e-3  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.60 +6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	-1.79999999999999999999999957772134e-8
*string in : 	-4.55 +8.71i	-4.550000012 +8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxsub(z1,z2) = 1.19999999999999999999999946169023e-8  	-7.99999999999999999998999082002114e-12
*string in : 	-4.55 +8.71i	-4.54 +8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxsub(z1,z2) = -1.00000000000000000000000000000014e-2  	1.99999999999999999999999999999996e-2
*string in : 	23.5  -6.5i	12.0 +8.5i
 z1 =  		2.35000000000000000000000000000000e+1  	-6.50000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.49999999999999999999999999999998e+0
cxsub(z1,z2) = 1.15000000000000000000000000000000e+1  	-1.50000000000000000000000000000000e+1
*string in : 	23    -6i	12 +8i
 z1 =  		2.30000000000000000000000000000000e+1  	-6.00000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.00000000000000000000000000000000e+0
cxsub(z1,z2) = 1.10000000000000000000000000000000e+1  	-1.40000000000000000000000000000000e+1
*string in : 	23.49999999999999999   -6.4999999999999999999i 	11.99999999999999999  +8.4999999999999999999i
 z1 =  		2.34999999999999999899999999999999e+1  	-6.49999999999999999989999999999999e+0
 z2 =  		1.19999999999999999900000000000000e+1  	8.49999999999999999989999999999997e+0
cxsub(z1,z2) = 1.15000000000000000000000000000000e+1  	-1.49999999999999999998000000000000e+1
*string in : 	22.9999999999999999999 -5.9999999999999999999i	11.999999999999999999 +7.99999999999999999999i
 z1 =  		2.29999999999999999998999999999999e+1  	-5.99999999999999999989999999999998e+0
 z2 =  		1.19999999999999999990000000000000e+1  	7.99999999999999999998999999999998e+0
cxsub(z1,z2) = 1.10000000000000000009000000000000e+1  	-1.39999999999999999998899999999999e+1
*string in : 	23.500001 -6.500001i				12.0 +8.50000000001i
 z1 =  		2.35000010000000000000000000000000e+1  	-6.50000100000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.50000000000999999999999999999998e+0
cxsub(z1,z2) = 1.15000010000000000000000000000000e+1  	-1.50000010000100000000000000000000e+1
*string in : 	23.00000000000100 -6.00000000000000002i 	12.00000000000003 +8.0000000000000003i
 z1 =  		2.30000000000010000000000000000000e+1  	-6.00000000000000002000000000000000e+0
 z2 =  		1.20000000000000300000000000000000e+1  	8.00000000000000029999999999999999e+0
cxsub(z1,z2) = 1.10000000000009700000000000000000e+1  	-1.40000000000000003200000000000000e+1
*string in : 	+4.55 -6i 	-0.50 +0.5i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999999999999999999999e-1  	5.00000000000000000000000000000000e-1
cxsub(z1,z2) = 5.05000000000000000000000000000000e+0  	-6.50000000000000000000000000000000e+0
*string in : 	+4.55 -6i	-0.49999999999999999 -0.49999999999999978i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999989999999999999990e-1  	-4.99999999999999779999999999999990e-1
cxsub(z1,z2) = 5.04999999999999998999999999999999e+0  	-5.50000000000000022000000000000001e+0
*string in : 	+7.60 -8i	+0.5000000000000023  -0.500000000000000000000123i
 z1 =  		7.60000000000000000000000000000000e+0  	-8.00000000000000000000000000000000e+0
 z2 =  		5.00000000000002299999999999999991e-1  	-5.00000000000000000000122999999986e-1
cxsub(z1,z2) = 7.09999999999999770000000000000001e+0  	-7.49999999999999999999987700000002e+0
*string in : 	+77.60 -788i	+21.5000000000000023 +12.500000000000000000000123i
 z1 =  		7.75999999999999999999999999999998e+1  	-7.87999999999999999999999999999998e+2
 z2 =  		2.15000000000000023000000000000000e+1  	1.25000000000000000000001230000000e+1
cxsub(z1,z2) = 5.60999999999999977000000000000000e+1  	-8.00500000000000000000000122999997e+2
*string in : 	0.00 +0.0i	0.0 +765.21i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2
cxsub(z1,z2) = 0.00000000000000000000000000000000e+0  	-7.65209999999999999999999999999997e+2

Test of cxmul
*string in : 	128.970870234 +73.6850340129i	-23.723000129 +5.09881720006i
 z1 =  		1.28970870234000000000000000000000e+2  	7.36850340128999999999999999999998e+1
 z2 =  		-2.37230001290000000000000000000000e+1  	5.09881720006000000000000000000000e+0
cxmul(z1,z2) = -3.43528249001040490410677399999998e+3  	-1.09043117993757061065005999999999e+3
*string in : 	138.570870234 +75.4850340129i	-23.723000129e2381  +5.09881720006e3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
cxmul(z1,z2) = -3.84884389772088643920773999998443e+3420  	7.06547536576401477014039999997142e+3420
*string in : 	138.570870234 +75.4850340129i	-23.723000129e-2381 +5.09881720006e-3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.09881720005999999999999999997939e-3418
cxmul(z1,z2) = -3.28731677243682426018599999999074e-2378  	-1.79073147162559608766409999999496e-2378
*string in : 	236.570870234e-2382 +375.4850340129e-3417i 	-23.723000129e-2381 +51.09881720006e-3418i
 z1 =  		2.36570870233999999999999999999333e-2380  	3.75485034012899999999999999998482e-3415
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.10988172000599999999999999997935e-3417
cxmul(z1,z2) = -5.61217078507882426018599999996837e-4760  	0.00000000000000000000000000000000e+0
*string in : 	236.570870234e+2382 +375.4850340129e+3417i	-23.723000129e+2381 +51.09881720006e+3418i
 z1 =  		2.36570870233999999999999999999333e+2384  	3.75485034012899999999999999998482e+3419
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.10988172000599999999999999997933e+3419
*** Floating point overflow
*string in : 	-7.2100128992 -0.23422221i 	+2.2198112817 -11.9127653i
 z1 =  		-7.21001289920000000000000000000000e+0  	-2.34222209999999999999999999999998e-1
 z2 =  		2.21981128170000000000000000000000e+0  	-1.19127653000000000000000000000000e+1
cxmul(z1,z2) = -1.87951021906239979046399999999999e+1  	8.53712623739594512029999999999997e+1
*string in : 	-99.888888888134 +3332.1128976i		0.000
 z1 =  		-9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
cxmul(z1,z2) = -0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+55 -321i 	0
 z1 =  		5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
cxmul(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+55.3 +4.234i	+1.443e-3210
 z1 =  		5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
 z2 =  		1.44299999999999999999999999999452e-3210  	0.00000000000000000000000000000000e+0
cxmul(z1,z2) = 7.97978999999999999999999999996970e-3209  	6.10966199999999999999999999997679e-3210
*string in : 	+56.3 +6.234i	-3.443e-3201
 z1 =  		5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
 z2 =  		-3.44299999999999999999999999998696e-3201  	0.00000000000000000000000000000000e+0
cxmul(z1,z2) = -1.93840899999999999999999999999266e-3199  	-2.14636619999999999999999999999187e-3200
*string in : 	+0.651119  -1.9872i 	+9.871e-3312i
 z1 =  		6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	9.87099999999999999999999999996133e-3312
cxmul(z1,z2) = 1.96156511999999999999999999999231e-3311  	6.42719564899999999999999999997477e-3312
*string in : 	-0.81273 -0.087133i	-7.12289e-3301i
 z1 =  		-8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
 z2 =  		0.00000000000000000000000000000000e+0  	-7.12288999999999999999999999997218e-3301
cxmul(z1,z2) = -6.20638774369999999999999999997573e-3302  	5.78898638969999999999999999997736e-3301
*string in : 	0 +0i			-45.768891 -56.98127i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		-4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
cxmul(z1,z2) = 0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	0 -0i			+53.8761 -32.11198i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
cxmul(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.2156e-3678 +6i 	+5.38761 +2.411198i
 z1 =  		3.21559999999999999999999999998601e-3678  	6.00000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxmul(z1,z2) = -1.44671880000000000000000000000000e+1  	3.23256599999999999999999999999999e+1
*string in : 	0.000 -7.1256e-3839i	+5.38761 +2.411198i
 z1 =  		0.00000000000000000000000000000000e+0  	-7.12559999999999999999999999996765e-3839
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxmul(z1,z2) = 1.71812324687999999999999999999220e-3838  	-3.83899538159999999999999999998257e-3838
*string in : 	+2.2 -4.3i	+2.2 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
cxmul(z1,z2) = -1.36500000000000000000000000000000e+1  	-1.89199999999999999999999999999999e+1
*string in : 	+2.2 -4.3i	+2.2 -4.3000000000000000000082i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000819999999999e+0
cxmul(z1,z2) = -1.36500000000000000000352599999999e+1  	-1.89200000000000000000180399999999e+1
*string in : 	+2.2 -4.3i	+2.20000000000000000000000871 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000870999999e+0  	-4.30000000000000000000000000000000e+0
cxmul(z1,z2) = -1.36499999999999999999999808380000e+1  	-1.89200000000000000000000374529999e+1
*string in : 	+5.5 +7.8i	+5.38761 +2.411198i
 z1 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxmul(z1,z2) = 1.08245106000000000000000000000000e+1  	5.52849469999999999999999999999998e+1
*string in : 	+5.5000000000000001829 +7.8000000000000007615i 		+5.5 +7.8i
 z1 =  		5.50000000000000018289999999999999e+0  	7.80000000000000076149999999999998e+0
 z2 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
cxmul(z1,z2) = -3.05900000000000049337499999999999e+1  	8.58000000000000056148699999999995e+1
*string in : 	-2.60 -6.54i 	-2.60 -6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxmul(z1,z2) = -3.60115986919999999999999999999999e+1  	3.40079994799999999999999999999999e+1
*string in : 	-2.60 -6.54i	-2.59 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxmul(z1,z2) = -3.60375999999999999999999999999999e+1  	3.39425999999999999999999999999999e+1
*string in : 	-2.60 -6.54i	-2.61 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxmul(z1,z2) = -3.59855999999999999999999999999999e+1  	3.40733999999999999999999999999999e+1
*string in : 	-2.60 -6.54i	-2.60 -6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxmul(z1,z2) = -3.60116001177199999999999999999999e+1  	3.40080000467999999999999999999999e+1
*string in : 	-4.55 -8.71i	-4.550000012 -8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxmul(z1,z2) = -5.51615999454696799999999999999998e+1  	7.92610001045563999999999999999996e+1
*string in : 	-4.55 -8.71i	-4.54 -8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxmul(z1,z2) = -5.50328999999999999999999999999999e+1  	7.90828999999999999999999999999997e+1
*string in : 	+2.60 +6.54i	+2.60 +6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxmul(z1,z2) = -3.60115986919999999999999999999999e+1  	3.40079994799999999999999999999999e+1
*string in : 	+2.60 +6.54i	+2.59 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxmul(z1,z2) = -3.60375999999999999999999999999999e+1  	3.39425999999999999999999999999999e+1
*string in : 	+2.60 +6.54i	+2.61 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxmul(z1,z2) = -3.59855999999999999999999999999999e+1  	3.40733999999999999999999999999999e+1
*string in : 	+2.60 +6.54i	+2.60 +6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxmul(z1,z2) = -3.60116001177199999999999999999999e+1  	3.40080000467999999999999999999999e+1
*string in : 	+4.55 +8.71i	+4.550000012 +8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxmul(z1,z2) = -5.51615999454696799999999999999998e+1  	7.92610001045563999999999999999996e+1
*string in : 	+4.55 +8.71i	+4.54 +8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxmul(z1,z2) = -5.50328999999999999999999999999999e+1  	7.90828999999999999999999999999997e+1
*string in : 	+2.60 -6.54i	+2.60 -6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxmul(z1,z2) = -3.60115986919999999999999999999999e+1  	-3.40079994799999999999999999999999e+1
*string in : 	+2.60 -6.54i	+2.59 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxmul(z1,z2) = -3.60375999999999999999999999999999e+1  	-3.39425999999999999999999999999999e+1
*string in : 	+2.60 -6.54i	+2.61 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxmul(z1,z2) = -3.59855999999999999999999999999999e+1  	-3.40733999999999999999999999999999e+1
*string in : 	+2.60 -6.54i	+2.60 -6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxmul(z1,z2) = -3.60116001177199999999999999999999e+1  	-3.40080000467999999999999999999999e+1
*string in : 	+4.55 -8.71i	+4.550000012 -8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxmul(z1,z2) = -5.51615999454696799999999999999998e+1  	-7.92610001045563999999999999999996e+1
*string in : 	+4.55 -8.71i	+4.54 -8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxmul(z1,z2) = -5.50328999999999999999999999999999e+1  	-7.90828999999999999999999999999997e+1
*string in : 	-2.60 +6.54i	-2.60 +6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxmul(z1,z2) = -3.60115986919999999999999999999999e+1  	-3.40079994799999999999999999999999e+1
*string in : 	-2.60 +6.54i	-2.59 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxmul(z1,z2) = -3.60375999999999999999999999999999e+1  	-3.39425999999999999999999999999999e+1
*string in : 	-2.60 +6.54i	-2.61 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxmul(z1,z2) = -3.59855999999999999999999999999999e+1  	-3.40733999999999999999999999999999e+1
*string in : 	-2.60 +6.54i	-2.60 +6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxmul(z1,z2) = -3.60116001177199999999999999999999e+1  	-3.40080000467999999999999999999999e+1
*string in : 	-4.55 +8.71i	-4.550000012 +8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxmul(z1,z2) = -5.51615999454696799999999999999998e+1  	-7.92610001045563999999999999999996e+1
*string in : 	-4.55 +8.71i	-4.54 +8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxmul(z1,z2) = -5.50328999999999999999999999999999e+1  	-7.90828999999999999999999999999997e+1
*string in : 	23.5  -6.5i	12.0 +8.5i
 z1 =  		2.35000000000000000000000000000000e+1  	-6.50000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.49999999999999999999999999999998e+0
cxmul(z1,z2) = 3.37249999999999999999999999999999e+2  	1.21749999999999999999999999999999e+2
*string in : 	23    -6i	12 +8i
 z1 =  		2.30000000000000000000000000000000e+1  	-6.00000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.00000000000000000000000000000000e+0
cxmul(z1,z2) = 3.23999999999999999999999999999999e+2  	1.12000000000000000000000000000000e+2
*string in : 	23.49999999999999999   -6.4999999999999999999i 	11.99999999999999999  +8.4999999999999999999i
 z1 =  		2.34999999999999999899999999999999e+1  	-6.49999999999999999989999999999999e+0
 z2 =  		1.19999999999999999900000000000000e+1  	8.49999999999999999989999999999997e+0
cxmul(z1,z2) = 3.37249999999999999643499999999998e+2  	1.21749999999999999978849999999999e+2
*string in : 	22.9999999999999999999 -5.9999999999999999999i	11.999999999999999999 +7.99999999999999999999i
 z1 =  		2.29999999999999999998999999999999e+1  	-5.99999999999999999989999999999998e+0
 z2 =  		1.19999999999999999990000000000000e+1  	7.99999999999999999998999999999998e+0
cxmul(z1,z2) = 3.23999999999999999974939999999998e+2  	1.12000000000000000006169999999999e+2
*string in : 	23.500001 -6.500001i				12.0 +8.50000000001i
 z1 =  		2.35000010000000000000000000000000e+1  	-6.50000100000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.50000000000999999999999999999998e+0
cxmul(z1,z2) = 3.37250020500065000009999999999999e+2  	1.21749996500235000010000000000000e+2
*string in : 	23.00000000000100 -6.00000000000000002i 	12.00000000000003 +8.0000000000000003i
 z1 =  		2.30000000000010000000000000000000e+1  	-6.00000000000000002000000000000000e+0
 z2 =  		1.20000000000000300000000000000000e+1  	8.00000000000000029999999999999999e+0
cxmul(z1,z2) = 3.24000000000012691960000000029999e+2  	1.12000000000007826660000000000299e+2
*string in : 	+4.55 -6i 	-0.50 +0.5i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999999999999999999999e-1  	5.00000000000000000000000000000000e-1
cxmul(z1,z2) = 7.25000000000000000000000000000007e-1  	5.27499999999999999999999999999999e+0
*string in : 	+4.55 -6i	-0.49999999999999999 -0.49999999999999978i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999989999999999999990e-1  	-4.99999999999999779999999999999990e-1
cxmul(z1,z2) = -5.27499999999999863449999999999989e+0  	7.25000000000000940999999999999988e-1
*string in : 	+7.60 -8i	+0.5000000000000023  -0.500000000000000000000123i
 z1 =  		7.60000000000000000000000000000000e+0  	-8.00000000000000000000000000000000e+0
 z2 =  		5.00000000000002299999999999999991e-1  	-5.00000000000000000000122999999986e-1
cxmul(z1,z2) = -1.99999999999982520000983999999962e-1  	-7.80000000000001840000093479999982e+0
*string in : 	+77.60 -788i	+21.5000000000000023 +12.500000000000000000000123i
 z1 =  		7.75999999999999999999999999999998e+1  	-7.87999999999999999999999999999998e+2
 z2 =  		2.15000000000000023000000000000000e+1  	1.25000000000000000000001230000000e+1
cxmul(z1,z2) = 1.15184000000000001784800969239999e+4  	-1.59720000000000018123999904551999e+4
*string in : 	0.00 +0.0i	0.0 +765.21i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2
cxmul(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0

Test of cxdiv
*string in : 	128.970870234 +73.6850340129i	-23.723000129 +5.09881720006i
 z1 =  		1.28970870234000000000000000000000e+2  	7.36850340128999999999999999999998e+1
 z2 =  		-2.37230001290000000000000000000000e+1  	5.09881720006000000000000000000000e+0
cxdiv(z1,z2) = -4.55836731220461252327465299047861e+0  	-4.08579501502729812221131475690346e+0
*string in : 	138.570870234 +75.4850340129i	-23.723000129e2381  +5.09881720006e3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
cxdiv(z1,z2) = 1.48044205256096913551761413448376e-3417  	-2.71770618158990630789913504283425e-3417
*string in : 	138.570870234 +75.4850340129i	-23.723000129e-2381 +5.09881720006e-3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.09881720005999999999999999997939e-3418
cxdiv(z1,z2) = -5.84120345152319499024186849297300e+2381  	-3.18193456149856432857080477234608e+2381
*string in : 	236.570870234e-2382 +375.4850340129e-3417i 	-23.723000129e-2381 +51.09881720006e-3418i
 z1 =  		2.36570870233999999999999999999333e-2380  	3.75485034012899999999999999998482e-3415
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.10988172000599999999999999997935e-3417
cxdiv(z1,z2) = -9.97221552702374054773584577591684e-1  	-1.60426892098951630500328794571315e-1035
*string in : 	236.570870234e+2382 +375.4850340129e+3417i	-23.723000129e+2381 +51.09881720006e+3418i
 z1 =  		2.36570870233999999999999999999333e+2384  	3.75485034012899999999999999998482e+3419
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.10988172000599999999999999997933e+3419
cxdiv(z1,z2) = 7.34821380586592339306922440068566e-1  	-4.66378871494436355550727290549307e-1036
*string in : 	-7.2100128992 -0.23422221i 	+2.2198112817 -11.9127653i
 z1 =  		-7.21001289920000000000000000000000e+0  	-2.34222209999999999999999999999998e-1
 z2 =  		2.21981128170000000000000000000000e+0  	-1.19127653000000000000000000000000e+1
cxdiv(z1,z2) = -8.99924764431869623186377166139236e-2  	-5.88465096741500656605963285995845e-1
*string in : 	-99.888888888134 +3332.1128976i		0.000
 z1 =  		-9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55 -321i 	0
 z1 =  		5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55.3 +4.234i	+1.443e-3210
 z1 =  		5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
 z2 =  		1.44299999999999999999999999999452e-3210  	0.00000000000000000000000000000000e+0
cxdiv(z1,z2) = 3.83229383229383229383229383229383e+3211  	2.93416493416493416493416493416494e+3210
*string in : 	+56.3 +6.234i	-3.443e-3201
 z1 =  		5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
 z2 =  		-3.44299999999999999999999999998696e-3201  	0.00000000000000000000000000000000e+0
cxdiv(z1,z2) = -1.63520185884403136799302933488237e+3202  	-1.81063026430438571013650885855359e+3201
*string in : 	+0.651119  -1.9872i 	+9.871e-3312i
 z1 =  		6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	9.87099999999999999999999999996133e-3312
cxdiv(z1,z2) = -2.01316989160166143247897882686658e+3311  	-6.59628203829399250329247290041532e+3310
*string in : 	-0.81273 -0.087133i	-7.12289e-3301i
 z1 =  		-8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
 z2 =  		0.00000000000000000000000000000000e+0  	-7.12288999999999999999999999997218e-3301
cxdiv(z1,z2) = 1.22328156127639202627023581720341e+3299  	-1.14101158378130225231612449441168e+3300
*string in : 	0 +0i			-45.768891 -56.98127i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		-4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
cxdiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	0 -0i			+53.8761 -32.11198i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
cxdiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.2156e-3678 +6i 	+5.38761 +2.411198i
 z1 =  		3.21559999999999999999999999998601e-3678  	6.00000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxdiv(z1,z2) = 4.15243908279729888307121807067627e-1  	9.27826015402698386255435065463566e-1
*string in : 	0.000 -7.1256e-3839i	+5.38761 +2.411198i
 z1 =  		0.00000000000000000000000000000000e+0  	-7.12559999999999999999999999996765e-3839
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxdiv(z1,z2) = -4.93143665473007215353537858071275e-3840  	-1.10188617589224460351695468373953e-3839
*string in : 	+2.2 -4.3i	+2.2 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
cxdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	1.92592994438723585305597794258493e-34
*string in : 	+2.2 -4.3i	+2.2 -4.3000000000000000000082i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000819999999999e+0
cxdiv(z1,z2) = 9.99999999999999999998488641234463e-1  	7.73253321902505667481917767841456e-22
*string in : 	+2.2 -4.3i	+2.20000000000000000000000871 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000870999999e+0  	-4.30000000000000000000000000000000e+0
cxdiv(z1,z2) = 9.99999999999999999999999178654093e-1  	-1.60535790722337976106855315308670e-24
*string in : 	+5.5 +7.8i	+5.38761 +2.411198i
 z1 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxdiv(z1,z2) = 1.39032426154945570886674049252951e+0  	8.25533570767088837850537261957311e-1
*string in : 	+5.5000000000000001829 +7.8000000000000007615i 		+5.5 +7.8i
 z1 =  		5.50000000000000018289999999999999e+0  	7.80000000000000076149999999999998e+0
 z2 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
cxdiv(z1,z2) = 1.00000000000000007625041168075530e+0  	3.03175979800197604513784650187060e-17
*string in : 	-2.60 -6.54i 	-2.60 -6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxdiv(z1,z2) = 1.00000002640738496397689574873739e+0  	1.04983490834877286917833248325812e-8
*string in : 	-2.60 -6.54i	-2.59 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxdiv(z1,z2) = 1.00052344698937139877565951289115e+0  	-1.32175417393395675398193602628959e-3
*string in : 	-2.60 -6.54i	-2.61 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxdiv(z1,z2) = 9.99473617337955820158640843664349e-1  	1.31898184282334718869305840427394e-3
*string in : 	-2.60 -6.54i	-2.60 -6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxdiv(z1,z2) = 9.99999997623335410839987233238432e-1  	-9.44851363120597654038121619847173e-10
*string in : 	-4.55 -8.71i	-4.550000012 -8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxdiv(z1,z2) = 9.99999999433865538580634668374588e-1  	1.08198486724909510047462709650785e-9
*string in : 	-4.55 -8.71i	-4.54 -8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxdiv(z1,z2) = 1.00228030005919209551461233338569e+0  	4.05710320750418453785953476470013e-5
*string in : 	+2.60 +6.54i	+2.60 +6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxdiv(z1,z2) = 1.00000002640738496397689574873739e+0  	1.04983490834877286917833248325812e-8
*string in : 	+2.60 +6.54i	+2.59 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxdiv(z1,z2) = 1.00052344698937139877565951289115e+0  	-1.32175417393395675398193602628959e-3
*string in : 	+2.60 +6.54i	+2.61 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxdiv(z1,z2) = 9.99473617337955820158640843664349e-1  	1.31898184282334718869305840427394e-3
*string in : 	+2.60 +6.54i	+2.60 +6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxdiv(z1,z2) = 9.99999997623335410839987233238432e-1  	-9.44851363120597654038121619847173e-10
*string in : 	+4.55 +8.71i	+4.550000012 +8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxdiv(z1,z2) = 9.99999999433865538580634668374588e-1  	1.08198486724909510047462709650785e-9
*string in : 	+4.55 +8.71i	+4.54 +8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxdiv(z1,z2) = 1.00228030005919209551461233338569e+0  	4.05710320750418453785953476470013e-5
*string in : 	+2.60 -6.54i	+2.60 -6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxdiv(z1,z2) = 1.00000002640738496397689574873739e+0  	-1.04983490834877286917833248325812e-8
*string in : 	+2.60 -6.54i	+2.59 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxdiv(z1,z2) = 1.00052344698937139877565951289115e+0  	1.32175417393395675398193602628959e-3
*string in : 	+2.60 -6.54i	+2.61 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxdiv(z1,z2) = 9.99473617337955820158640843664349e-1  	-1.31898184282334718869305840427394e-3
*string in : 	+2.60 -6.54i	+2.60 -6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxdiv(z1,z2) = 9.99999997623335410839987233238432e-1  	9.44851363120597654038121619847173e-10
*string in : 	+4.55 -8.71i	+4.550000012 -8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxdiv(z1,z2) = 9.99999999433865538580634668374588e-1  	-1.08198486724909510047462709650785e-9
*string in : 	+4.55 -8.71i	+4.54 -8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxdiv(z1,z2) = 1.00228030005919209551461233338569e+0  	-4.05710320750418453785953476470013e-5
*string in : 	-2.60 +6.54i	-2.60 +6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxdiv(z1,z2) = 1.00000002640738496397689574873739e+0  	-1.04983490834877286917833248325812e-8
*string in : 	-2.60 +6.54i	-2.59 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxdiv(z1,z2) = 1.00052344698937139877565951289115e+0  	1.32175417393395675398193602628959e-3
*string in : 	-2.60 +6.54i	-2.61 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxdiv(z1,z2) = 9.99473617337955820158640843664349e-1  	-1.31898184282334718869305840427394e-3
*string in : 	-2.60 +6.54i	-2.60 +6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxdiv(z1,z2) = 9.99999997623335410839987233238432e-1  	9.44851363120597654038121619847173e-10
*string in : 	-4.55 +8.71i	-4.550000012 +8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxdiv(z1,z2) = 9.99999999433865538580634668374588e-1  	-1.08198486724909510047462709650785e-9
*string in : 	-4.55 +8.71i	-4.54 +8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxdiv(z1,z2) = 1.00228030005919209551461233338569e+0  	-4.05710320750418453785953476470013e-5
*string in : 	23.5  -6.5i	12.0 +8.5i
 z1 =  		2.35000000000000000000000000000000e+1  	-6.50000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.49999999999999999999999999999998e+0
cxdiv(z1,z2) = 1.04855491329479768786127167630058e+0  	-1.28439306358381502890173410404624e+0
*string in : 	23    -6i	12 +8i
 z1 =  		2.30000000000000000000000000000000e+1  	-6.00000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.00000000000000000000000000000000e+0
cxdiv(z1,z2) = 1.09615384615384615384615384615385e+0  	-1.23076923076923076923076923076923e+0
*string in : 	23.49999999999999999   -6.4999999999999999999i 	11.99999999999999999  +8.4999999999999999999i
 z1 =  		2.34999999999999999899999999999999e+1  	-6.49999999999999999989999999999999e+0
 z2 =  		1.19999999999999999900000000000000e+1  	8.49999999999999999989999999999997e+0
cxdiv(z1,z2) = 1.04855491329479768739854669384209e+0  	-1.28439306358381502962722683684720e+0
*string in : 	22.9999999999999999999 -5.9999999999999999999i	11.999999999999999999 +7.99999999999999999999i
 z1 =  		2.29999999999999999998999999999999e+1  	-5.99999999999999999989999999999998e+0
 z2 =  		1.19999999999999999990000000000000e+1  	7.99999999999999999998999999999998e+0
cxdiv(z1,z2) = 1.09615384615384615386126479289941e+0  	-1.23076923076923076933416050295858e+0
*string in : 	23.500001 -6.500001i				12.0 +8.50000000001i
 z1 =  		2.35000010000000000000000000000000e+1  	-6.50000100000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.50000000000999999999999999999998e+0
cxdiv(z1,z2) = 1.04855492947864391052942670834551e+0  	-1.28439315838157989906624278833684e+0
*string in : 	23.00000000000100 -6.00000000000000002i 	12.00000000000003 +8.0000000000000003i
 z1 =  		2.30000000000010000000000000000000e+1  	-6.00000000000000002000000000000000e+0
 z2 =  		1.20000000000000300000000000000000e+1  	8.00000000000000029999999999999999e+0
cxdiv(z1,z2) = 1.09615384615390333436390532538402e+0  	-1.23076923076926584172337278093899e+0
*string in : 	+4.55 -6i 	-0.50 +0.5i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999999999999999999999e-1  	5.00000000000000000000000000000000e-1
cxdiv(z1,z2) = -1.05500000000000000000000000000000e+1  	1.44999999999999999999999999999999e+0
*string in : 	+4.55 -6i	-0.49999999999999999 -0.49999999999999978i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999989999999999999990e-1  	-4.99999999999999779999999999999990e-1
cxdiv(z1,z2) = 1.44999999999999811799999999999903e+0  	1.05500000000000027310000000000004e+1
*string in : 	+7.60 -8i	+0.5000000000000023  -0.500000000000000000000123i
 z1 =  		7.60000000000000000000000000000000e+0  	-8.00000000000000000000000000000000e+0
 z2 =  		5.00000000000002299999999999999991e-1  	-5.00000000000000000000122999999986e-1
cxdiv(z1,z2) = 1.55999999999999631999981304000045e+1  	-4.00000000000034959998031999835036e-1
*string in : 	+77.60 -788i	+21.5000000000000023 +12.500000000000000000000123i
 z1 =  		7.75999999999999999999999999999998e+1  	-7.87999999999999999999999999999998e+2
 z2 =  		2.15000000000000023000000000000000e+1  	1.25000000000000000000001230000000e+1
cxdiv(z1,z2) = -1.32281325788197227376843075217812e+1  	-2.89603880355699255427931184700743e+1
*string in : 	0.00 +0.0i	0.0 +765.21i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2
cxdiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0

Test of cxgdiv
*string in : 	128.970870234 +73.6850340129i	-23.723000129 +5.09881720006i
 z1 =  		1.28970870234000000000000000000000e+2  	7.36850340128999999999999999999998e+1
 z2 =  		-2.37230001290000000000000000000000e+1  	5.09881720006000000000000000000000e+0
cxgdiv(z1,z2) = -5.00000000000000000000000000000000e+0  	-4.00000000000000000000000000000000e+0
*string in : 	138.570870234 +75.4850340129i	-23.723000129e2381  +5.09881720006e3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
*** Floating point overflow
*string in : 	138.570870234 +75.4850340129i	-23.723000129e-2381 +5.09881720006e-3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.09881720005999999999999999997939e-3418
*** Division by zero
*string in : 	236.570870234e-2382 +375.4850340129e-3417i 	-23.723000129e-2381 +51.09881720006e-3418i
 z1 =  		2.36570870233999999999999999999333e-2380  	3.75485034012899999999999999998482e-3415
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.10988172000599999999999999997935e-3417
*** Division by zero
*string in : 	236.570870234e+2382 +375.4850340129e+3417i	-23.723000129e+2381 +51.09881720006e+3418i
 z1 =  		2.36570870233999999999999999999333e+2384  	3.75485034012899999999999999998482e+3419
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.10988172000599999999999999997933e+3419
*** Floating point overflow
*string in : 	-7.2100128992 -0.23422221i 	+2.2198112817 -11.9127653i
 z1 =  		-7.21001289920000000000000000000000e+0  	-2.34222209999999999999999999999998e-1
 z2 =  		2.21981128170000000000000000000000e+0  	-1.19127653000000000000000000000000e+1
cxgdiv(z1,z2) = 0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-99.888888888134 +3332.1128976i		0.000
 z1 =  		-9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55 -321i 	0
 z1 =  		5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55.3 +4.234i	+1.443e-3210
 z1 =  		5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
 z2 =  		1.44299999999999999999999999999452e-3210  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+56.3 +6.234i	-3.443e-3201
 z1 =  		5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
 z2 =  		-3.44299999999999999999999999998696e-3201  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+0.651119  -1.9872i 	+9.871e-3312i
 z1 =  		6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	9.87099999999999999999999999996133e-3312
*** Division by zero
*string in : 	-0.81273 -0.087133i	-7.12289e-3301i
 z1 =  		-8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
 z2 =  		0.00000000000000000000000000000000e+0  	-7.12288999999999999999999999997218e-3301
*** Division by zero
*string in : 	0 +0i			-45.768891 -56.98127i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		-4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
cxgdiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	0 -0i			+53.8761 -32.11198i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
cxgdiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.2156e-3678 +6i 	+5.38761 +2.411198i
 z1 =  		3.21559999999999999999999999998601e-3678  	6.00000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxgdiv(z1,z2) = 0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	0.000 -7.1256e-3839i	+5.38761 +2.411198i
 z1 =  		0.00000000000000000000000000000000e+0  	-7.12559999999999999999999999996765e-3839
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxgdiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3000000000000000000082i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000819999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.20000000000000000000000871 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000870999999e+0  	-4.30000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+5.5 +7.8i	+5.38761 +2.411198i
 z1 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxgdiv(z1,z2) = 2.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+5.5000000000000001829 +7.8000000000000007615i 		+5.5 +7.8i
 z1 =  		5.50000000000000018289999999999999e+0  	7.80000000000000076149999999999998e+0
 z2 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i 	-2.60 -6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.59 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.61 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.60 -6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 -8.71i	-4.550000012 -8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 -8.71i	-4.54 -8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.60 +6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.59 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.61 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.60 +6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 +8.71i	+4.550000012 +8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 +8.71i	+4.54 +8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.60 -6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.59 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.61 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.60 -6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 -8.71i	+4.550000012 -8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 -8.71i	+4.54 -8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.60 +6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.59 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.61 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.60 +6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 +8.71i	-4.550000012 +8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 +8.71i	-4.54 +8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	23.5  -6.5i	12.0 +8.5i
 z1 =  		2.35000000000000000000000000000000e+1  	-6.50000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.49999999999999999999999999999998e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	23    -6i	12 +8i
 z1 =  		2.30000000000000000000000000000000e+1  	-6.00000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.00000000000000000000000000000000e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	23.49999999999999999   -6.4999999999999999999i 	11.99999999999999999  +8.4999999999999999999i
 z1 =  		2.34999999999999999899999999999999e+1  	-6.49999999999999999989999999999999e+0
 z2 =  		1.19999999999999999900000000000000e+1  	8.49999999999999999989999999999997e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	22.9999999999999999999 -5.9999999999999999999i	11.999999999999999999 +7.99999999999999999999i
 z1 =  		2.29999999999999999998999999999999e+1  	-5.99999999999999999989999999999998e+0
 z2 =  		1.19999999999999999990000000000000e+1  	7.99999999999999999998999999999998e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	23.500001 -6.500001i				12.0 +8.50000000001i
 z1 =  		2.35000010000000000000000000000000e+1  	-6.50000100000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.50000000000999999999999999999998e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	23.00000000000100 -6.00000000000000002i 	12.00000000000003 +8.0000000000000003i
 z1 =  		2.30000000000010000000000000000000e+1  	-6.00000000000000002000000000000000e+0
 z2 =  		1.20000000000000300000000000000000e+1  	8.00000000000000029999999999999999e+0
cxgdiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+4.55 -6i 	-0.50 +0.5i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999999999999999999999e-1  	5.00000000000000000000000000000000e-1
cxgdiv(z1,z2) = -6.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+4.55 -6i	-0.49999999999999999 -0.49999999999999978i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999989999999999999990e-1  	-4.99999999999999779999999999999990e-1
*** Division by zero
*string in : 	+7.60 -8i	+0.5000000000000023  -0.500000000000000000000123i
 z1 =  		7.60000000000000000000000000000000e+0  	-8.00000000000000000000000000000000e+0
 z2 =  		5.00000000000002299999999999999991e-1  	-5.00000000000000000000122999999986e-1
cxgdiv(z1,z2) = 8.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+77.60 -788i	+21.5000000000000023 +12.500000000000000000000123i
 z1 =  		7.75999999999999999999999999999998e+1  	-7.87999999999999999999999999999998e+2
 z2 =  		2.15000000000000023000000000000000e+1  	1.25000000000000000000001230000000e+1
cxgdiv(z1,z2) = -1.30000000000000000000000000000000e+1  	-2.80000000000000000000000000000000e+1
*string in : 	0.00 +0.0i	0.0 +765.21i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2
cxgdiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0

Test of cxidiv
*string in : 	128.970870234 +73.6850340129i	-23.723000129 +5.09881720006i
 z1 =  		1.28970870234000000000000000000000e+2  	7.36850340128999999999999999999998e+1
 z2 =  		-2.37230001290000000000000000000000e+1  	5.09881720006000000000000000000000e+0
cxidiv(z1,z2) = -4.00000000000000000000000000000000e+0  	-4.00000000000000000000000000000000e+0
*string in : 	138.570870234 +75.4850340129i	-23.723000129e2381  +5.09881720006e3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
*** Floating point overflow
*string in : 	138.570870234 +75.4850340129i	-23.723000129e-2381 +5.09881720006e-3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.09881720005999999999999999997939e-3418
*** Division by zero
*string in : 	236.570870234e-2382 +375.4850340129e-3417i 	-23.723000129e-2381 +51.09881720006e-3418i
 z1 =  		2.36570870233999999999999999999333e-2380  	3.75485034012899999999999999998482e-3415
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.10988172000599999999999999997935e-3417
*** Division by zero
*string in : 	236.570870234e+2382 +375.4850340129e+3417i	-23.723000129e+2381 +51.09881720006e+3418i
 z1 =  		2.36570870233999999999999999999333e+2384  	3.75485034012899999999999999998482e+3419
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.10988172000599999999999999997933e+3419
*** Floating point overflow
*string in : 	-7.2100128992 -0.23422221i 	+2.2198112817 -11.9127653i
 z1 =  		-7.21001289920000000000000000000000e+0  	-2.34222209999999999999999999999998e-1
 z2 =  		2.21981128170000000000000000000000e+0  	-1.19127653000000000000000000000000e+1
cxidiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-99.888888888134 +3332.1128976i		0.000
 z1 =  		-9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55 -321i 	0
 z1 =  		5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55.3 +4.234i	+1.443e-3210
 z1 =  		5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
 z2 =  		1.44299999999999999999999999999452e-3210  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+56.3 +6.234i	-3.443e-3201
 z1 =  		5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
 z2 =  		-3.44299999999999999999999999998696e-3201  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+0.651119  -1.9872i 	+9.871e-3312i
 z1 =  		6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	9.87099999999999999999999999996133e-3312
*** Division by zero
*string in : 	-0.81273 -0.087133i	-7.12289e-3301i
 z1 =  		-8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
 z2 =  		0.00000000000000000000000000000000e+0  	-7.12288999999999999999999999997218e-3301
*** Division by zero
*string in : 	0 +0i			-45.768891 -56.98127i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		-4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
cxidiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	0 -0i			+53.8761 -32.11198i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
cxidiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.2156e-3678 +6i 	+5.38761 +2.411198i
 z1 =  		3.21559999999999999999999999998601e-3678  	6.00000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxidiv(z1,z2) = 0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	0.000 -7.1256e-3839i	+5.38761 +2.411198i
 z1 =  		0.00000000000000000000000000000000e+0  	-7.12559999999999999999999999996765e-3839
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxidiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3000000000000000000082i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000819999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.20000000000000000000000871 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000870999999e+0  	-4.30000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+5.5 +7.8i	+5.38761 +2.411198i
 z1 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+5.5000000000000001829 +7.8000000000000007615i 		+5.5 +7.8i
 z1 =  		5.50000000000000018289999999999999e+0  	7.80000000000000076149999999999998e+0
 z2 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i 	-2.60 -6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.59 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.61 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.60 -6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 -8.71i	-4.550000012 -8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 -8.71i	-4.54 -8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.60 +6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.59 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.61 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.60 +6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 +8.71i	+4.550000012 +8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 +8.71i	+4.54 +8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.60 -6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.59 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.61 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.60 -6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 -8.71i	+4.550000012 -8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 -8.71i	+4.54 -8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.60 +6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.59 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.61 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.60 +6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 +8.71i	-4.550000012 +8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 +8.71i	-4.54 +8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	23.5  -6.5i	12.0 +8.5i
 z1 =  		2.35000000000000000000000000000000e+1  	-6.50000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.49999999999999999999999999999998e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	23    -6i	12 +8i
 z1 =  		2.30000000000000000000000000000000e+1  	-6.00000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.00000000000000000000000000000000e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	23.49999999999999999   -6.4999999999999999999i 	11.99999999999999999  +8.4999999999999999999i
 z1 =  		2.34999999999999999899999999999999e+1  	-6.49999999999999999989999999999999e+0
 z2 =  		1.19999999999999999900000000000000e+1  	8.49999999999999999989999999999997e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	22.9999999999999999999 -5.9999999999999999999i	11.999999999999999999 +7.99999999999999999999i
 z1 =  		2.29999999999999999998999999999999e+1  	-5.99999999999999999989999999999998e+0
 z2 =  		1.19999999999999999990000000000000e+1  	7.99999999999999999998999999999998e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	23.500001 -6.500001i				12.0 +8.50000000001i
 z1 =  		2.35000010000000000000000000000000e+1  	-6.50000100000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.50000000000999999999999999999998e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	23.00000000000100 -6.00000000000000002i 	12.00000000000003 +8.0000000000000003i
 z1 =  		2.30000000000010000000000000000000e+1  	-6.00000000000000002000000000000000e+0
 z2 =  		1.20000000000000300000000000000000e+1  	8.00000000000000029999999999999999e+0
cxidiv(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+4.55 -6i 	-0.50 +0.5i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999999999999999999999e-1  	5.00000000000000000000000000000000e-1
cxidiv(z1,z2) = -5.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 -6i	-0.49999999999999999 -0.49999999999999978i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999989999999999999990e-1  	-4.99999999999999779999999999999990e-1
*** Division by zero
*string in : 	+7.60 -8i	+0.5000000000000023  -0.500000000000000000000123i
 z1 =  		7.60000000000000000000000000000000e+0  	-8.00000000000000000000000000000000e+0
 z2 =  		5.00000000000002299999999999999991e-1  	-5.00000000000000000000122999999986e-1
cxidiv(z1,z2) = 8.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+77.60 -788i	+21.5000000000000023 +12.500000000000000000000123i
 z1 =  		7.75999999999999999999999999999998e+1  	-7.87999999999999999999999999999998e+2
 z2 =  		2.15000000000000023000000000000000e+1  	1.25000000000000000000001230000000e+1
cxidiv(z1,z2) = -1.30000000000000000000000000000000e+1  	-2.80000000000000000000000000000000e+1
*string in : 	0.00 +0.0i	0.0 +765.21i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2
cxidiv(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0

Test of cxgmod
*string in : 	128.970870234 +73.6850340129i	-23.723000129 +5.09881720006i
 z1 =  		1.28970870234000000000000000000000e+2  	7.36850340128999999999999999999998e+1
 z2 =  		-2.37230001290000000000000000000000e+1  	5.09881720006000000000000000000000e+0
cxgmod(z1,z2) = -1.10000000000000000000000000000000e+1  	3.00000000000000000000000000000000e+0
*string in : 	138.570870234 +75.4850340129i	-23.723000129e2381  +5.09881720006e3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
*** Floating point overflow
*string in : 	138.570870234 +75.4850340129i	-23.723000129e-2381 +5.09881720006e-3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.09881720005999999999999999997939e-3418
*** Division by zero
*string in : 	236.570870234e-2382 +375.4850340129e-3417i 	-23.723000129e-2381 +51.09881720006e-3418i
 z1 =  		2.36570870233999999999999999999333e-2380  	3.75485034012899999999999999998482e-3415
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.10988172000599999999999999997935e-3417
*** Division by zero
*string in : 	236.570870234e+2382 +375.4850340129e+3417i	-23.723000129e+2381 +51.09881720006e+3418i
 z1 =  		2.36570870233999999999999999999333e+2384  	3.75485034012899999999999999998482e+3419
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.10988172000599999999999999997933e+3419
*** Floating point overflow
*string in : 	-7.2100128992 -0.23422221i 	+2.2198112817 -11.9127653i
 z1 =  		-7.21001289920000000000000000000000e+0  	-2.34222209999999999999999999999998e-1
 z2 =  		2.21981128170000000000000000000000e+0  	-1.19127653000000000000000000000000e+1
cxgmod(z1,z2) = 5.00000000000000000000000000000000e+0  	2.00000000000000000000000000000000e+0
*string in : 	-99.888888888134 +3332.1128976i		0.000
 z1 =  		-9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55 -321i 	0
 z1 =  		5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55.3 +4.234i	+1.443e-3210
 z1 =  		5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
 z2 =  		1.44299999999999999999999999999452e-3210  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+56.3 +6.234i	-3.443e-3201
 z1 =  		5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
 z2 =  		-3.44299999999999999999999999998696e-3201  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+0.651119  -1.9872i 	+9.871e-3312i
 z1 =  		6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	9.87099999999999999999999999996133e-3312
*** Division by zero
*string in : 	-0.81273 -0.087133i	-7.12289e-3301i
 z1 =  		-8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
 z2 =  		0.00000000000000000000000000000000e+0  	-7.12288999999999999999999999997218e-3301
*** Division by zero
*string in : 	0 +0i			-45.768891 -56.98127i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		-4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	0 -0i			+53.8761 -32.11198i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.2156e-3678 +6i 	+5.38761 +2.411198i
 z1 =  		3.21559999999999999999999999998601e-3678  	6.00000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxgmod(z1,z2) = 2.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	0.000 -7.1256e-3839i	+5.38761 +2.411198i
 z1 =  		0.00000000000000000000000000000000e+0  	-7.12559999999999999999999999996765e-3839
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3000000000000000000082i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000819999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.20000000000000000000000871 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000870999999e+0  	-4.30000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+5.5 +7.8i	+5.38761 +2.411198i
 z1 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxgmod(z1,z2) = -2.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+5.5000000000000001829 +7.8000000000000007615i 		+5.5 +7.8i
 z1 =  		5.50000000000000018289999999999999e+0  	7.80000000000000076149999999999998e+0
 z2 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i 	-2.60 -6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.59 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.61 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.60 -6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 -8.71i	-4.550000012 -8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 -8.71i	-4.54 -8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.60 +6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.59 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.61 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.60 +6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 +8.71i	+4.550000012 +8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 +8.71i	+4.54 +8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.60 -6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.59 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.61 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.60 -6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 -8.71i	+4.550000012 -8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 -8.71i	+4.54 -8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.60 +6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.59 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.61 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.60 +6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 +8.71i	-4.550000012 +8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 +8.71i	-4.54 +8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	23.5  -6.5i	12.0 +8.5i
 z1 =  		2.35000000000000000000000000000000e+1  	-6.50000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.49999999999999999999999999999998e+0
cxgmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-4.00000000000000000000000000000000e+0
*string in : 	23    -6i	12 +8i
 z1 =  		2.30000000000000000000000000000000e+1  	-6.00000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.00000000000000000000000000000000e+0
cxgmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	23.49999999999999999   -6.4999999999999999999i 	11.99999999999999999  +8.4999999999999999999i
 z1 =  		2.34999999999999999899999999999999e+1  	-6.49999999999999999989999999999999e+0
 z2 =  		1.19999999999999999900000000000000e+1  	8.49999999999999999989999999999997e+0
cxgmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	22.9999999999999999999 -5.9999999999999999999i	11.999999999999999999 +7.99999999999999999999i
 z1 =  		2.29999999999999999998999999999999e+1  	-5.99999999999999999989999999999998e+0
 z2 =  		1.19999999999999999990000000000000e+1  	7.99999999999999999998999999999998e+0
cxgmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	23.500001 -6.500001i				12.0 +8.50000000001i
 z1 =  		2.35000010000000000000000000000000e+1  	-6.50000100000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.50000000000999999999999999999998e+0
cxgmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-4.00000000000000000000000000000000e+0
*string in : 	23.00000000000100 -6.00000000000000002i 	12.00000000000003 +8.0000000000000003i
 z1 =  		2.30000000000010000000000000000000e+1  	-6.00000000000000002000000000000000e+0
 z2 =  		1.20000000000000300000000000000000e+1  	8.00000000000000029999999999999999e+0
cxgmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	+4.55 -6i 	-0.50 +0.5i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999999999999999999999e-1  	5.00000000000000000000000000000000e-1
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+4.55 -6i	-0.49999999999999999 -0.49999999999999978i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999989999999999999990e-1  	-4.99999999999999779999999999999990e-1
*** Division by zero
*string in : 	+7.60 -8i	+0.5000000000000023  -0.500000000000000000000123i
 z1 =  		7.60000000000000000000000000000000e+0  	-8.00000000000000000000000000000000e+0
 z2 =  		5.00000000000002299999999999999991e-1  	-5.00000000000000000000122999999986e-1
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+77.60 -788i	+21.5000000000000023 +12.500000000000000000000123i
 z1 =  		7.75999999999999999999999999999998e+1  	-7.87999999999999999999999999999998e+2
 z2 =  		2.15000000000000023000000000000000e+1  	1.25000000000000000000001230000000e+1
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	-3.00000000000000000000000000000000e+0
*string in : 	0.00 +0.0i	0.0 +765.21i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2
cxgmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0

Test of cxmod
*string in : 	128.970870234 +73.6850340129i	-23.723000129 +5.09881720006i
 z1 =  		1.28970870234000000000000000000000e+2  	7.36850340128999999999999999999998e+1
 z2 =  		-2.37230001290000000000000000000000e+1  	5.09881720006000000000000000000000e+0
cxmod(z1,z2) = 1.30000000000000000000000000000000e+1  	-2.00000000000000000000000000000000e+0
*string in : 	138.570870234 +75.4850340129i	-23.723000129e2381  +5.09881720006e3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
*** Floating point overflow
*string in : 	138.570870234 +75.4850340129i	-23.723000129e-2381 +5.09881720006e-3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.09881720005999999999999999997939e-3418
*** Division by zero
*string in : 	236.570870234e-2382 +375.4850340129e-3417i 	-23.723000129e-2381 +51.09881720006e-3418i
 z1 =  		2.36570870233999999999999999999333e-2380  	3.75485034012899999999999999998482e-3415
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.10988172000599999999999999997935e-3417
*** Division by zero
*string in : 	236.570870234e+2382 +375.4850340129e+3417i	-23.723000129e+2381 +51.09881720006e+3418i
 z1 =  		2.36570870233999999999999999999333e+2384  	3.75485034012899999999999999998482e+3419
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.10988172000599999999999999997933e+3419
*** Floating point overflow
*string in : 	-7.2100128992 -0.23422221i 	+2.2198112817 -11.9127653i
 z1 =  		-7.21001289920000000000000000000000e+0  	-2.34222209999999999999999999999998e-1
 z2 =  		2.21981128170000000000000000000000e+0  	-1.19127653000000000000000000000000e+1
cxmod(z1,z2) = -7.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-99.888888888134 +3332.1128976i		0.000
 z1 =  		-9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55 -321i 	0
 z1 =  		5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+55.3 +4.234i	+1.443e-3210
 z1 =  		5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
 z2 =  		1.44299999999999999999999999999452e-3210  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+56.3 +6.234i	-3.443e-3201
 z1 =  		5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
 z2 =  		-3.44299999999999999999999999998696e-3201  	0.00000000000000000000000000000000e+0
*** Division by zero
*string in : 	+0.651119  -1.9872i 	+9.871e-3312i
 z1 =  		6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	9.87099999999999999999999999996133e-3312
*** Division by zero
*string in : 	-0.81273 -0.087133i	-7.12289e-3301i
 z1 =  		-8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
 z2 =  		0.00000000000000000000000000000000e+0  	-7.12288999999999999999999999997218e-3301
*** Division by zero
*string in : 	0 +0i			-45.768891 -56.98127i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		-4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	0 -0i			+53.8761 -32.11198i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.2156e-3678 +6i 	+5.38761 +2.411198i
 z1 =  		3.21559999999999999999999999998601e-3678  	6.00000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxmod(z1,z2) = 2.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	0.000 -7.1256e-3839i	+5.38761 +2.411198i
 z1 =  		0.00000000000000000000000000000000e+0  	-7.12559999999999999999999999996765e-3839
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3000000000000000000082i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000819999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.20000000000000000000000871 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000870999999e+0  	-4.30000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+5.5 +7.8i	+5.38761 +2.411198i
 z1 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxmod(z1,z2) = 1.00000000000000000000000000000000e+0  	6.00000000000000000000000000000000e+0
*string in : 	+5.5000000000000001829 +7.8000000000000007615i 		+5.5 +7.8i
 z1 =  		5.50000000000000018289999999999999e+0  	7.80000000000000076149999999999998e+0
 z2 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i 	-2.60 -6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.59 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.61 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 -6.54i	-2.60 -6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 -8.71i	-4.550000012 -8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 -8.71i	-4.54 -8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.60 +6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.59 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.61 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 +6.54i	+2.60 +6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 +8.71i	+4.550000012 +8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 +8.71i	+4.54 +8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.60 -6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.59 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.61 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.60 -6.54i	+2.60 -6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 -8.71i	+4.550000012 -8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.55 -8.71i	+4.54 -8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.60 +6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.59 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.61 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.60 +6.54i	-2.60 +6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 +8.71i	-4.550000012 +8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.55 +8.71i	-4.54 +8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	23.5  -6.5i	12.0 +8.5i
 z1 =  		2.35000000000000000000000000000000e+1  	-6.50000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.49999999999999999999999999999998e+0
cxmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-4.00000000000000000000000000000000e+0
*string in : 	23    -6i	12 +8i
 z1 =  		2.30000000000000000000000000000000e+1  	-6.00000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.00000000000000000000000000000000e+0
cxmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	23.49999999999999999   -6.4999999999999999999i 	11.99999999999999999  +8.4999999999999999999i
 z1 =  		2.34999999999999999899999999999999e+1  	-6.49999999999999999989999999999999e+0
 z2 =  		1.19999999999999999900000000000000e+1  	8.49999999999999999989999999999997e+0
cxmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	22.9999999999999999999 -5.9999999999999999999i	11.999999999999999999 +7.99999999999999999999i
 z1 =  		2.29999999999999999998999999999999e+1  	-5.99999999999999999989999999999998e+0
 z2 =  		1.19999999999999999990000000000000e+1  	7.99999999999999999998999999999998e+0
cxmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	23.500001 -6.500001i				12.0 +8.50000000001i
 z1 =  		2.35000010000000000000000000000000e+1  	-6.50000100000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.50000000000999999999999999999998e+0
cxmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-4.00000000000000000000000000000000e+0
*string in : 	23.00000000000100 -6.00000000000000002i 	12.00000000000003 +8.0000000000000003i
 z1 =  		2.30000000000010000000000000000000e+1  	-6.00000000000000002000000000000000e+0
 z2 =  		1.20000000000000300000000000000000e+1  	8.00000000000000029999999999999999e+0
cxmod(z1,z2) = 3.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	+4.55 -6i 	-0.50 +0.5i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999999999999999999999e-1  	5.00000000000000000000000000000000e-1
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+4.55 -6i	-0.49999999999999999 -0.49999999999999978i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999989999999999999990e-1  	-4.99999999999999779999999999999990e-1
*** Division by zero
*string in : 	+7.60 -8i	+0.5000000000000023  -0.500000000000000000000123i
 z1 =  		7.60000000000000000000000000000000e+0  	-8.00000000000000000000000000000000e+0
 z2 =  		5.00000000000002299999999999999991e-1  	-5.00000000000000000000122999999986e-1
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+77.60 -788i	+21.5000000000000023 +12.500000000000000000000123i
 z1 =  		7.75999999999999999999999999999998e+1  	-7.87999999999999999999999999999998e+2
 z2 =  		2.15000000000000023000000000000000e+1  	1.25000000000000000000001230000000e+1
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	-3.00000000000000000000000000000000e+0
*string in : 	0.00 +0.0i	0.0 +765.21i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2
cxmod(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0

Test of cxpow
*string in : 	128.970870234 +73.6850340129i	-23.723000129 +5.09881720006i
 z1 =  		1.28970870234000000000000000000000e+2  	7.36850340128999999999999999999998e+1
 z2 =  		-2.37230001290000000000000000000000e+1  	5.09881720006000000000000000000000e+0
cxpow(z1,z2) = 1.73534457917482724700682594163481e-53  	1.23359817465711837559134500071438e-53
*string in : 	138.570870234 +75.4850340129i	-23.723000129e2381  +5.09881720006e3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.09881720005999999999999999997939e+3418
*** Floating point overflow
*string in : 	138.570870234 +75.4850340129i	-23.723000129e-2381 +5.09881720006e-3418i
 z1 =  		1.38570870234000000000000000000000e+2  	7.54850340128999999999999999999998e+1
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.09881720005999999999999999997939e-3418
cxpow(z1,z2) = 1.00000000000000000000000000000000e+0  	-1.18329259834337495548699946629982e-2380
*string in : 	236.570870234e-2382 +375.4850340129e-3417i 	-23.723000129e-2381 +51.09881720006e-3418i
 z1 =  		2.36570870233999999999999999999333e-2380  	3.75485034012899999999999999998482e-3415
 z2 =  		-2.37230001289999999999999999999332e-2380  	5.10988172000599999999999999997935e-3417
cxpow(z1,z2) = 1.00000000000000000000000000000000e+0  	-2.83750623928016706429118623592944e-3413
*string in : 	236.570870234e+2382 +375.4850340129e+3417i	-23.723000129e+2381 +51.09881720006e+3418i
 z1 =  		2.36570870233999999999999999999333e+2384  	3.75485034012899999999999999998482e+3419
 z2 =  		-2.37230001289999999999999999999331e+2382  	5.10988172000599999999999999997933e+3419
*** Floating point overflow
*string in : 	-7.2100128992 -0.23422221i 	+2.2198112817 -11.9127653i
 z1 =  		-7.21001289920000000000000000000000e+0  	-2.34222209999999999999999999999998e-1
 z2 =  		2.21981128170000000000000000000000e+0  	-1.19127653000000000000000000000000e+1
cxpow(z1,z2) = 3.70497101540665558874409161386099e-15  	5.46079658017740740113534176994966e-15
*string in : 	-99.888888888134 +3332.1128976i		0.000
 z1 =  		-9.98888888881339999999999999999997e+1  	3.33211289759999999999999999999999e+3
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
cxpow(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+55 -321i 	0
 z1 =  		5.50000000000000000000000000000000e+1  	-3.20999999999999999999999999999999e+2
 z2 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
cxpow(z1,z2) = 1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+55.3 +4.234i	+1.443e-3210
 z1 =  		5.53000000000000000000000000000000e+1  	4.23400000000000000000000000000000e+0
 z2 =  		1.44299999999999999999999999999452e-3210  	0.00000000000000000000000000000000e+0
cxpow(z1,z2) = 1.00000000000000000000000000000000e+0  	1.10267005082310990596158897311749e-3211
*string in : 	+56.3 +6.234i	-3.443e-3201
 z1 =  		5.63000000000000000000000000000000e+1  	6.23400000000000000000000000000000e+0
 z2 =  		-3.44299999999999999999999999998696e-3201  	0.00000000000000000000000000000000e+0
cxpow(z1,z2) = 1.00000000000000000000000000000000e+0  	-3.79690613840392320249504846540140e-3202
*string in : 	+0.651119  -1.9872i 	+9.871e-3312i
 z1 =  		6.51118999999999999999999999999996e-1  	-1.98720000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	9.87099999999999999999999999996133e-3312
cxpow(z1,z2) = 1.00000000000000000000000000000000e+0  	7.28198935826007961671153541548936e-3312
*string in : 	-0.81273 -0.087133i	-7.12289e-3301i
 z1 =  		-8.12729999999999999999999999999996e-1  	-8.71329999999999999999999999999994e-2
 z2 =  		0.00000000000000000000000000000000e+0  	-7.12288999999999999999999999997218e-3301
cxpow(z1,z2) = 1.00000000000000000000000000000000e+0  	1.43627438798290972034195052803298e-3301
*string in : 	0 +0i			-45.768891 -56.98127i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		-4.57688909999999999999999999999999e+1  	-5.69812699999999999999999999999999e+1
*** Bad exponent
*string in : 	0 -0i			+53.8761 -32.11198i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		5.38760999999999999999999999999999e+1  	-3.21119800000000000000000000000000e+1
cxpow(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.2156e-3678 +6i 	+5.38761 +2.411198i
 z1 =  		3.21559999999999999999999999998601e-3678  	6.00000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxpow(z1,z2) = 3.44509887251779634594019160106680e+2  	7.58658339037708830668448445251386e+1
*string in : 	0.000 -7.1256e-3839i	+5.38761 +2.411198i
 z1 =  		0.00000000000000000000000000000000e+0  	-7.12559999999999999999999999996765e-3839
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxpow(z1,z2) = 0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.2 -4.3i	+2.2 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
cxpow(z1,z2) = -2.76739389929216342963404142276472e-1  	-6.69838778584700500258458588331995e-2
*string in : 	+2.2 -4.3i	+2.2 -4.3000000000000000000082i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000819999999999e+0
cxpow(z1,z2) = -2.76739389929216342961777767153879e-1  	-6.69838778584700500216690290476872e-2
*string in : 	+2.2 -4.3i	+2.20000000000000000000000871 -4.3i
 z1 =  		2.20000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+0
 z2 =  		2.20000000000000000000000870999999e+0  	-4.30000000000000000000000000000000e+0
cxpow(z1,z2) = -2.76739389929216342963408578884693e-1  	-6.69838778584700500258441313054788e-2
*string in : 	+5.5 +7.8i	+5.38761 +2.411198i
 z1 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
 z2 =  		5.38761000000000000000000000000000e+0  	2.41119800000000000000000000000000e+0
cxpow(z1,z2) = -7.40024720435308749960296967080785e+3  	-1.74008579126700754589706270065287e+4
*string in : 	+5.5000000000000001829 +7.8000000000000007615i 		+5.5 +7.8i
 z1 =  		5.50000000000000018289999999999999e+0  	7.80000000000000076149999999999998e+0
 z2 =  		5.50000000000000000000000000000000e+0  	7.80000000000000000000000000000000e+0
cxpow(z1,z2) = -9.10220412953799428016067586828261e+1  	-1.07141067234889439462445058721742e+2
*string in : 	-2.60 -6.54i 	-2.60 -6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxpow(z1,z2) = 2.90790045783408251170455809613376e-9  	-1.79797601811638606758362653458800e-8
*string in : 	-2.60 -6.54i	-2.59 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxpow(z1,z2) = 2.60728513412017517664508032724102e-9  	-1.83883504302448800274631915456958e-8
*string in : 	-2.60 -6.54i	-2.61 -6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxpow(z1,z2) = 3.19482464929624288165008044682690e-9  	-1.75733841543586942077963185514047e-8
*string in : 	-2.60 -6.54i	-2.60 -6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxpow(z1,z2) = 2.90789157388834649914884381871452e-9  	-1.79797537781107626383852112351530e-8
*string in : 	-4.55 -8.71i	-4.550000012 -8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxpow(z1,z2) = -2.19474651973058877661558026142752e-13  	4.78944449482309885706333435244669e-13
*string in : 	-4.55 -8.71i	-4.54 -8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxpow(z1,z2) = -2.46735904942826562803073134260998e-13  	5.04493143244564924079883287334069e-13
*string in : 	+2.60 +6.54i	+2.60 +6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxpow(z1,z2) = -6.47783878975138001278492098598989e-2  	-1.00432708890096055836050675616163e-2
*string in : 	+2.60 +6.54i	+2.59 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxpow(z1,z2) = -6.36395206224814197268946933259149e-2  	-9.09104423586988130641641715316659e-3
*string in : 	+2.60 +6.54i	+2.61 +6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxpow(z1,z2) = -6.59279964165020783552015486816715e-2  	-1.10280849394446357567340186383634e-2
*string in : 	+2.60 +6.54i	+2.60 +6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxpow(z1,z2) = -6.47783667865464062639748636980897e-2  	-1.00432958340477247768373826657866e-2
*string in : 	+4.55 +8.71i	+4.550000012 +8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxpow(z1,z2) = 2.38926506802513171481500907079951e+0  	-6.67969377257703133274818184112870e-1
*string in : 	+4.55 +8.71i	+4.54 +8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxpow(z1,z2) = 2.34516034578385243560513894365123e+0  	-8.01199029324510717375812715756862e-1
*string in : 	+2.60 -6.54i	+2.60 -6.5399998i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.53999979999999999999999999999999e+0
cxpow(z1,z2) = -6.47783878975138001278492098598989e-2  	1.00432708890096055836050675616163e-2
*string in : 	+2.60 -6.54i	+2.59 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.59000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxpow(z1,z2) = -6.36395206224814197268946933259149e-2  	9.09104423586988130641641715316659e-3
*string in : 	+2.60 -6.54i	+2.61 -6.54i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.61000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
cxpow(z1,z2) = -6.59279964165020783552015486816715e-2  	1.10280849394446357567340186383634e-2
*string in : 	+2.60 -6.54i	+2.60 -6.540000018i
 z1 =  		2.60000000000000000000000000000000e+0  	-6.54000000000000000000000000000000e+0
 z2 =  		2.60000000000000000000000000000000e+0  	-6.54000001799999999999999999999999e+0
cxpow(z1,z2) = -6.47783667865464062639748636980897e-2  	1.00432958340477247768373826657866e-2
*string in : 	+4.55 -8.71i	+4.550000012 -8.710000000008i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.55000001199999999999999999999999e+0  	-8.71000000000799999999999999999997e+0
cxpow(z1,z2) = 2.38926506802513171481500907079951e+0  	6.67969377257703133274818184112870e-1
*string in : 	+4.55 -8.71i	+4.54 -8.69i
 z1 =  		4.55000000000000000000000000000000e+0  	-8.70999999999999999999999999999999e+0
 z2 =  		4.54000000000000000000000000000000e+0  	-8.68999999999999999999999999999999e+0
cxpow(z1,z2) = 2.34516034578385243560513894365123e+0  	8.01199029324510717375812715756862e-1
*string in : 	-2.60 +6.54i	-2.60 +6.5399998i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.53999979999999999999999999999999e+0
cxpow(z1,z2) = 2.90790045783408251170455809613376e-9  	1.79797601811638606758362653458800e-8
*string in : 	-2.60 +6.54i	-2.59 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.59000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxpow(z1,z2) = 2.60728513412017517664508032724102e-9  	1.83883504302448800274631915456958e-8
*string in : 	-2.60 +6.54i	-2.61 +6.54i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.61000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
cxpow(z1,z2) = 3.19482464929624288165008044682690e-9  	1.75733841543586942077963185514047e-8
*string in : 	-2.60 +6.54i	-2.60 +6.540000018i
 z1 =  		-2.60000000000000000000000000000000e+0  	6.54000000000000000000000000000000e+0
 z2 =  		-2.60000000000000000000000000000000e+0  	6.54000001799999999999999999999999e+0
cxpow(z1,z2) = 2.90789157388834649914884381871452e-9  	1.79797537781107626383852112351530e-8
*string in : 	-4.55 +8.71i	-4.550000012 +8.710000000008i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.55000001199999999999999999999999e+0  	8.71000000000799999999999999999997e+0
cxpow(z1,z2) = -2.19474651973058877661558026142752e-13  	-4.78944449482309885706333435244669e-13
*string in : 	-4.55 +8.71i	-4.54 +8.69i
 z1 =  		-4.55000000000000000000000000000000e+0  	8.70999999999999999999999999999999e+0
 z2 =  		-4.54000000000000000000000000000000e+0  	8.68999999999999999999999999999999e+0
cxpow(z1,z2) = -2.46735904942826562803073134260998e-13  	-5.04493143244564924079883287334069e-13
*string in : 	23.5  -6.5i	12.0 +8.5i
 z1 =  		2.35000000000000000000000000000000e+1  	-6.50000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.49999999999999999999999999999998e+0
cxpow(z1,z2) = 1.49091391973551097213886609636762e+17  	-4.11400510855522573213483657059324e+17
*string in : 	23    -6i	12 +8i
 z1 =  		2.30000000000000000000000000000000e+1  	-6.00000000000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.00000000000000000000000000000000e+0
cxpow(z1,z2) = -2.39796866740662120164111094214096e+17  	-7.25953195837996690865720390033941e+16
*string in : 	23.49999999999999999   -6.4999999999999999999i 	11.99999999999999999  +8.4999999999999999999i
 z1 =  		2.34999999999999999899999999999999e+1  	-6.49999999999999999989999999999999e+0
 z2 =  		1.19999999999999999900000000000000e+1  	8.49999999999999999989999999999997e+0
cxpow(z1,z2) = 1.49091391973551090944869263719796e+17  	-4.11400510855522558810632069678654e+17
*string in : 	22.9999999999999999999 -5.9999999999999999999i	11.999999999999999999 +7.99999999999999999999i
 z1 =  		2.29999999999999999998999999999999e+1  	-5.99999999999999999989999999999998e+0
 z2 =  		1.19999999999999999990000000000000e+1  	7.99999999999999999998999999999998e+0
cxpow(z1,z2) = -2.39796866740662119367316642908086e+17  	-7.25953195837996689025618681711353e+16
*string in : 	23.500001 -6.500001i				12.0 +8.50000000001i
 z1 =  		2.35000010000000000000000000000000e+1  	-6.50000100000000000000000000000000e+0
 z2 =  		1.20000000000000000000000000000000e+1  	8.50000000000999999999999999999998e+0
cxpow(z1,z2) = 1.49091553800521297373459715559581e+17  	-4.11400847181511040516724474602465e+17
*string in : 	23.00000000000100 -6.00000000000000002i 	12.00000000000003 +8.0000000000000003i
 z1 =  		2.30000000000010000000000000000000e+1  	-6.00000000000000002000000000000000e+0
 z2 =  		1.20000000000000300000000000000000e+1  	8.00000000000000029999999999999999e+0
cxpow(z1,z2) = -2.39796866740749296131374813306413e+17  	-7.25953195839429123255451957913468e+16
*string in : 	+4.55 -6i 	-0.50 +0.5i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999999999999999999999e-1  	5.00000000000000000000000000000000e-1
cxpow(z1,z2) = 5.78887903578586710430377245698216e-2  	5.74930258784074886080604153411546e-1
*string in : 	+4.55 -6i	-0.49999999999999999 -0.49999999999999978i
 z1 =  		4.55000000000000000000000000000000e+0  	-6.00000000000000000000000000000000e+0
 z2 =  		-4.99999999999999989999999999999990e-1  	-4.99999999999999779999999999999990e-1
cxpow(z1,z2) = 1.96114080869367594727591801286630e-1  	-1.19824217887870128503039092398247e-1
*string in : 	+7.60 -8i	+0.5000000000000023  -0.500000000000000000000123i
 z1 =  		7.60000000000000000000000000000000e+0  	-8.00000000000000000000000000000000e+0
 z2 =  		5.00000000000002299999999999999991e-1  	-5.00000000000000000000122999999986e-1
cxpow(z1,z2) = -7.80059186647805172247866510226878e-2  	-2.21305820409484019056075734422887e+0
*string in : 	+77.60 -788i	+21.5000000000000023 +12.500000000000000000000123i
 z1 =  		7.75999999999999999999999999999998e+1  	-7.87999999999999999999999999999998e+2
 z2 =  		2.15000000000000023000000000000000e+1  	1.25000000000000000000001230000000e+1
cxpow(z1,z2) = 1.42140988823697979772501033650456e+69  	2.05972253573410119543124721101146e+70
*string in : 	0.00 +0.0i	0.0 +765.21i
 z1 =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 z2 =  		0.00000000000000000000000000000000e+0  	7.65209999999999999999999999999997e+2
*** Bad exponent

*/
