/*
   Copyright (C)  2000    Daniel A. Atkinson  <DanAtk@aol.com>
   Copyright (C)  2004    Ivano Primi  <ivprimi@libero.it>    

   This file is part of the HPA Library.

   The HPA Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The HPA Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the HPA Library; if not, write to the Free
   Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
   02110-1301 USA.
*/

/*
    Test:  cxdrot   cxrrot  cxsqr  cxconj  cxneg
	   cxfloor cxceil   cxround  cxtrunc cxfrac

    Uses:  All the tested functions + atocx  xpr_asprint

    Input file:  data/cxgen.dat
*/
#include<stdio.h>
#include<stdlib.h>
#include"cxpre.h"

#define DECD 32

int
get_line (char *buffer, int size, FILE * fp)
{
  if ((fgets (buffer, size, fp)))
    {
      int i;

      for (i = 0; buffer[i] != '\0'; i++);
      if (i > 0 && buffer[i - 1] == '\n')
	buffer[--i] = '\0';
      return i;
    }
  else
    return -1;
}

typedef struct cxpr (*pfun) (struct cxpr);
extern int xErrNo;

int
main (int na, char **av)
{
  struct cxpr z, w;
  FILE *fp;
  char num[1024], *re, *im;
  const char *nlist[] = {
    "cxdrot", "cxrrot", "cxsqr", "cxconj", "cxneg",
    "cxfloor", "cxceil", "cxround", "cxtrunc", "cxfrac",
    NULL
  };
  pfun flist[] = {
    cxdrot, cxrrot, cxsqr, cxconj, cxneg,
    cxfloor, cxceil, cxround, cxtrunc, cxfrac
  };
  int i;

  if (na != 2)
    {
      printf ("para: input_file\n");
      exit (-1);
    }
  fp = fopen (*++av, "r");
  for (i = 0; nlist[i] != NULL; i++)
    {
      printf ("Test of %s\n", nlist[i]);
      while (get_line (num, 1024, fp) != -1)
	{
	  printf ("*string in : \t%s\n", num);
	  /* convert ascii string to extended precision complex */
	  z = atocx (num);
	  re = xpr_asprint (z.re, 1, 0, DECD);
	  im = xpr_asprint (z.im, 1, 0, DECD);
	  printf (" z =  \t\t%s  \t%s\n", re, im);
	  free ((void *) re);
	  free ((void *) im);
	  printf (" %s(z) = \t", nlist[i]);
	  xErrNo = 0;
	  w = (*flist[i]) (z);
	  switch (xErrNo)
	    {
	    case XENONE:
	      re = xpr_asprint (w.re, 1, 0, DECD);
	      im = xpr_asprint (w.im, 1, 0, DECD);
	      printf ("%s  \t%s\n", re, im);
	      free ((void *) re);
	      free ((void *) im);
	      break;
	    case XEDIV:
	      printf ("*** Division by zero\n");
	      break;
	    case XEDOM:
	      printf ("*** Out of domain\n");
	      break;
	    case XEBADEXP:
	      printf ("*** Bad exponent\n");
	      break;
	    case XFPOFLOW:
	      printf ("*** Floating point overflow\n");
	      break;
	    default:
	      printf ("*** Unknown error\n");
	    }
	}
      printf ("\n");
      rewind (fp);
    }
  fclose (fp);
  return 0;
}

/*  Test output

Test of cxdrot
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	2.31059999999999999999999999999571e-1571
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxdrot(z) = 	-9.77319999999999999999999999999977e-20  	8.77919999999999999999999999999979e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxdrot(z) = 	-2.32979999999999999999999999999567e-1571  	0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxdrot(z) = 	-8.22189999999999999999999999999983e-18  	-4.55919999999999999999999999999990e-19
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxdrot(z) = 	3.89319999999999999999999999999252e-1624  	0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxdrot(z) = 	7.33419999999999999999999999999984e-18  	-1.22909999999999999999999999999998e-17
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxdrot(z) = 	4.09869999999999999999999999999216e-1617  	0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxdrot(z) = 	8.22149999999999999999999999999982e-19  	3.99169999999999999999999999999991e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	1.00000000000000000009860000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxdrot(z) = 	-2.42999999999999999999999999999995e-18  	1.00000000000000000045670000000000e+0
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxdrot(z) = 	-3.45059999999999999999999999999470e-1298  	1.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxdrot(z) = 	-4.66769999999999999999999999999284e-1297  	9.99999999999999998735999999999976e-1
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	9.99999999999999999753999999999976e-1
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxdrot(z) = 	5.07829999999999999999999999999312e-1145  	9.99999999999999999337999999999976e-1
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxdrot(z) = 	6.70019999999999999999999999998969e-1301  	1.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxdrot(z) = 	5.33209999999999999999999999999153e-1344  	1.00000000000000074230000000000000e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	-9.99999999999999999988109999999973e-1
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxdrot(z) = 	-2.33439999999999999999999999999996e-15  	-9.99999999999999999766319999999974e-1
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxdrot(z) = 	-4.56009999999999999999999999999462e-997  	-1.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxdrot(z) = 	-7.43319999999999999999999999998655e-1532  	-1.00000000000000000000654000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	-1.00000000000000000000743000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxdrot(z) = 	6.99829999999999999999999999999985e-18  	-1.00000000000000000085023000000000e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxdrot(z) = 	5.00097999999999999999999999999395e-1023  	-1.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxdrot(z) = 	6.73209999999999999999999999999987e-16  	-9.99999999999999999764299999999974e-1
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxdrot(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxdrot(z) = 	-1.00000000000000000000000000000000e+0  	5.60979999999999999999999999998542e-2198
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxdrot(z) = 	-1.00000000000000000054210000000000e+0  	4.99869999999999999999999999999989e-18
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxdrot(z) = 	-1.00000000000000000000320100000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxdrot(z) = 	-1.00000000000000000809810000000000e+0  	-6.55409999999999999999999999999984e-20
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxdrot(z) = 	-1.00000000000000000000000000000000e+0  	-7.09979999999999999999999999998153e-2200
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxdrot(z) = 	-9.99999999999999999986399999999974e-1  	-6.84319999999999999999999999999986e-17
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxdrot(z) = 	-9.99999999999999890987319999999974e-1  	0.00000000000000000000000000000000e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxdrot(z) = 	-9.99999999999999999787699999999974e-1  	4.33289999999999999999999999999992e-16
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxdrot(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxdrot(z) = 	1.00000000000000000000000000000000e+0  	2.77619999999999999999999999999307e-2111
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxdrot(z) = 	9.99999999999999999655499999999974e-1  	3.33209999999999999999999999999993e-17
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxdrot(z) = 	9.99999999999999999996999972012965e-1  	0.00000000000000000000000000000000e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxdrot(z) = 	9.99999999999999999987230999999972e-1  	-7.55409999999999999999999999999985e-17
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxdrot(z) = 	1.00000000000000000000000000000000e+0  	-2.11559999999999999999999999999449e-2203
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxdrot(z) = 	1.00000000000000000089321000000000e+0  	-5.99229999999999999999999999999988e-16
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxdrot(z) = 	1.00000000000000000000716100000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxdrot(z) = 	1.00000000000000000008765000000000e+0  	8.66309999999999999999999999999981e-19
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	1.58759999999999999999999999999648e+1876
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	-6.43909999999999999999999999998503e+1965
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxdrot(z) = 	-3.42099999999999999999999999999185e+2015  	0.00000000000000000000000000000000e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxdrot(z) = 	4.55219999999999999999999999998917e+2011  	0.00000000000000000000000000000000e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxdrot(z) = 	-9.87649999999999999999999999997812e+1872  	2.19069999999999999999999999999520e+1854
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxdrot(z) = 	-7.57209999999999999999999999998357e+1834  	-3.29659999999999999999999999999241e+1946
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxdrot(z) = 	5.44109999999999999999999999998585e+2198  	-6.09869999999999999999999999998333e+2310
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxdrot(z) = 	3.86989999999999999999999999999111e+1943  	7.88949999999999999999999999998271e+1854
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	1.00098765431000912340000000000000e+1
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxdrot(z) = 	-6.11002230065009876000000000000000e+0  	9.89800000000065420089999999999997e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxdrot(z) = 	-1.79998888123400000076000000000000e+1  	0.00000000000000000000000000000000e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxdrot(z) = 	-2.08777722019200000000000000000000e+1  	-1.29822227000000000000000000000000e+1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	-1.79877771265000000000000000000000e+1
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxdrot(z) = 	3.29800000653999999999999999999999e+1  	-1.90987221098100000000000000000000e+1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxdrot(z) = 	4.39876001823449999999999999999999e+1  	0.00000000000000000000000000000000e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxdrot(z) = 	4.32189031228999999999999999999999e+1  	1.59822011923610000000000000000000e+1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	1.00000987654310009123400000000000e+3
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxdrot(z) = 	-6.11002230065009875999999999999998e+2  	9.89800000000065420089999999999996e+2
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxdrot(z) = 	-1.79998888123400000076000000000000e+2  	0.00000000000000000000000000000000e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxdrot(z) = 	-2.08777722019199999999999999999999e+2  	-1.29822227000000000000000000000000e+3
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxdrot(z) = 	-0.00000000000000000000000000000000e+0  	-1.79877771265098099999999999999999e+3
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxdrot(z) = 	3.29800700653999999999999999999999e+2  	-1.90987221098100000000000000000000e+2
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxdrot(z) = 	4.32189031228999999999999999999998e+3  	1.59822011923609999999999999999999e+3

Test of cxrrot
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	-2.31059999999999999999999999999571e-1571
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxrrot(z) = 	9.77319999999999999999999999999977e-20  	-8.77919999999999999999999999999979e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxrrot(z) = 	2.32979999999999999999999999999567e-1571  	-0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxrrot(z) = 	8.22189999999999999999999999999983e-18  	4.55919999999999999999999999999990e-19
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxrrot(z) = 	-3.89319999999999999999999999999252e-1624  	-0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxrrot(z) = 	-7.33419999999999999999999999999984e-18  	1.22909999999999999999999999999998e-17
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxrrot(z) = 	-4.09869999999999999999999999999216e-1617  	-0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxrrot(z) = 	-8.22149999999999999999999999999982e-19  	-3.99169999999999999999999999999991e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000009860000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxrrot(z) = 	2.42999999999999999999999999999995e-18  	-1.00000000000000000045670000000000e+0
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxrrot(z) = 	3.45059999999999999999999999999470e-1298  	-1.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxrrot(z) = 	4.66769999999999999999999999999284e-1297  	-9.99999999999999998735999999999976e-1
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	-9.99999999999999999753999999999976e-1
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxrrot(z) = 	-5.07829999999999999999999999999312e-1145  	-9.99999999999999999337999999999976e-1
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxrrot(z) = 	-6.70019999999999999999999999998969e-1301  	-1.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxrrot(z) = 	-5.33209999999999999999999999999153e-1344  	-1.00000000000000074230000000000000e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	9.99999999999999999988109999999973e-1
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxrrot(z) = 	2.33439999999999999999999999999996e-15  	9.99999999999999999766319999999974e-1
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxrrot(z) = 	4.56009999999999999999999999999462e-997  	1.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxrrot(z) = 	7.43319999999999999999999999998655e-1532  	1.00000000000000000000654000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000743000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxrrot(z) = 	-6.99829999999999999999999999999985e-18  	1.00000000000000000085023000000000e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxrrot(z) = 	-5.00097999999999999999999999999395e-1023  	1.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxrrot(z) = 	-6.73209999999999999999999999999987e-16  	9.99999999999999999764299999999974e-1
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxrrot(z) = 	1.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxrrot(z) = 	1.00000000000000000000000000000000e+0  	-5.60979999999999999999999999998542e-2198
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxrrot(z) = 	1.00000000000000000054210000000000e+0  	-4.99869999999999999999999999999989e-18
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxrrot(z) = 	1.00000000000000000000320100000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxrrot(z) = 	1.00000000000000000809810000000000e+0  	6.55409999999999999999999999999984e-20
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxrrot(z) = 	1.00000000000000000000000000000000e+0  	7.09979999999999999999999999998153e-2200
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxrrot(z) = 	9.99999999999999999986399999999974e-1  	6.84319999999999999999999999999986e-17
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxrrot(z) = 	9.99999999999999890987319999999974e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxrrot(z) = 	9.99999999999999999787699999999974e-1  	-4.33289999999999999999999999999992e-16
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxrrot(z) = 	-1.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxrrot(z) = 	-1.00000000000000000000000000000000e+0  	-2.77619999999999999999999999999307e-2111
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxrrot(z) = 	-9.99999999999999999655499999999974e-1  	-3.33209999999999999999999999999993e-17
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxrrot(z) = 	-9.99999999999999999996999972012965e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxrrot(z) = 	-9.99999999999999999987230999999972e-1  	7.55409999999999999999999999999985e-17
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxrrot(z) = 	-1.00000000000000000000000000000000e+0  	2.11559999999999999999999999999449e-2203
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxrrot(z) = 	-1.00000000000000000089321000000000e+0  	5.99229999999999999999999999999988e-16
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxrrot(z) = 	-1.00000000000000000000716100000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxrrot(z) = 	-1.00000000000000000008765000000000e+0  	-8.66309999999999999999999999999981e-19
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	-1.58759999999999999999999999999648e+1876
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	6.43909999999999999999999999998503e+1965
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxrrot(z) = 	3.42099999999999999999999999999185e+2015  	-0.00000000000000000000000000000000e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxrrot(z) = 	-4.55219999999999999999999999998917e+2011  	-0.00000000000000000000000000000000e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxrrot(z) = 	9.87649999999999999999999999997812e+1872  	-2.19069999999999999999999999999520e+1854
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxrrot(z) = 	7.57209999999999999999999999998357e+1834  	3.29659999999999999999999999999241e+1946
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxrrot(z) = 	-5.44109999999999999999999999998585e+2198  	6.09869999999999999999999999998333e+2310
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxrrot(z) = 	-3.86989999999999999999999999999111e+1943  	-7.88949999999999999999999999998271e+1854
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	-1.00098765431000912340000000000000e+1
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxrrot(z) = 	6.11002230065009876000000000000000e+0  	-9.89800000000065420089999999999997e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxrrot(z) = 	1.79998888123400000076000000000000e+1  	-0.00000000000000000000000000000000e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxrrot(z) = 	2.08777722019200000000000000000000e+1  	1.29822227000000000000000000000000e+1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	1.79877771265000000000000000000000e+1
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxrrot(z) = 	-3.29800000653999999999999999999999e+1  	1.90987221098100000000000000000000e+1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxrrot(z) = 	-4.39876001823449999999999999999999e+1  	-0.00000000000000000000000000000000e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxrrot(z) = 	-4.32189031228999999999999999999999e+1  	-1.59822011923610000000000000000000e+1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	-1.00000987654310009123400000000000e+3
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxrrot(z) = 	6.11002230065009875999999999999998e+2  	-9.89800000000065420089999999999996e+2
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxrrot(z) = 	1.79998888123400000076000000000000e+2  	-0.00000000000000000000000000000000e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxrrot(z) = 	2.08777722019199999999999999999999e+2  	1.29822227000000000000000000000000e+3
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxrrot(z) = 	0.00000000000000000000000000000000e+0  	1.79877771265098099999999999999999e+3
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxrrot(z) = 	-3.29800700653999999999999999999999e+2  	1.90987221098100000000000000000000e+2
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxrrot(z) = 	-4.32189031228999999999999999999998e+3  	-1.59822011923609999999999999999999e+3

Test of cxsqr
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	5.33887235999999999999999999998016e-3142  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxsqr(z) = 	-1.84410855999999999999999999999991e-39  	1.71601754879999999999999999999992e-38
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxsqr(z) = 	-5.42796803999999999999999999997983e-3142  	0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxsqr(z) = 	-6.73917765635999999999999999999971e-35  	-7.49705729599999999999999999999967e-36
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxsqr(z) = 	-1.51570062399999999999999999999417e-3247  	0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxsqr(z) = 	9.72781913599999999999999999999961e-35  	1.80289304399999999999999999999993e-34
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxsqr(z) = 	-1.67993416899999999999999999999357e-3233  	0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxsqr(z) = 	-5.16593933599999999999999999999978e-37  	-6.56355230999999999999999999999970e-37
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	1.00000000000000000019720000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxsqr(z) = 	1.00000000000000000091340000000000e+0  	4.86000000000000000221956199999989e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxsqr(z) = 	1.00000000000000000000000000000000e+0  	6.90119999999999999999999999998941e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxsqr(z) = 	9.99999999999999997471999999999952e-1  	9.33539999999999998820005439998545e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	9.99999999999999999507999999999952e-1  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxsqr(z) = 	9.99999999999999998675999999999952e-1  	-1.01565999999999999932763307999860e-1144
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxsqr(z) = 	1.00000000000000000000000000000000e+0  	-1.34003999999999999999999999999794e-1300
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxsqr(z) = 	1.00000000000000148460000000000055e+0  	-1.06642000000000079160356599999830e-1343
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	9.99999999999999999976219999999946e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxsqr(z) = 	9.99999999999999999532639999994497e-1  	-4.66879999999999999890899481599979e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxsqr(z) = 	1.00000000000000000000000000000000e+0  	-9.12019999999999999999999999998925e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxsqr(z) = 	1.00000000000000000001308000000000e+0  	-1.48664000000000000000972262559731e-1531
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	1.00000000000000000001486000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxsqr(z) = 	1.00000000000000000170046000000000e+0  	1.39966000000000000119003292179997e-17
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxsqr(z) = 	1.00000000000000000000000000000000e+0  	1.00019599999999999999999999999879e-1022
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxsqr(z) = 	9.99999999999999999528599999999495e-1  	1.34641999999999999968264880599994e-15
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxsqr(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxsqr(z) = 	-1.00000000000000000000000000000000e+0  	1.12195999999999999999999999999708e-2197
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxsqr(z) = 	-1.00000000000000000108420000000000e+0  	9.99740000000000000541959053999978e-18
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxsqr(z) = 	-1.00000000000000000000640200000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxsqr(z) = 	-1.00000000000000001619620000000000e+0  	-1.31082000000000001061515144199997e-19
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxsqr(z) = 	-1.00000000000000000000000000000000e+0  	-1.41995999999999999999999999999630e-2199
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxsqr(z) = 	-9.99999999999999999972799999999943e-1  	-1.36863999999999999998138649599994e-16
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxsqr(z) = 	-9.99999999999999781974639999999959e-1  	0.00000000000000000000000000000000e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxsqr(z) = 	-9.99999999999999999575399999999761e-1  	8.66579999999999999816025065999961e-16
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxsqr(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxsqr(z) = 	-1.00000000000000000000000000000000e+0  	-5.55239999999999999999999999998613e-2111
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxsqr(z) = 	-9.99999999999999999310999999999947e-1  	-6.66419999999999999770418309999969e-17
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxsqr(z) = 	-9.99999999999999999993999944025929e-1  	0.00000000000000000000000000000000e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxsqr(z) = 	-9.99999999999999999974461999999938e-1  	1.51081999999999999998070833941993e-16
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxsqr(z) = 	-1.00000000000000000000000000000000e+0  	4.23119999999999999999999999998897e-2203
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxsqr(z) = 	-1.00000000000000000178641999999964e+0  	1.19846000000000000107047645659998e-15
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxsqr(z) = 	-1.00000000000000000001432200000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxsqr(z) = 	-1.00000000000000000017530000000000e+0  	-1.73262000000000000015186414299996e-18
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	2.52047375999999999999999999998881e+3752  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	4.14620088099999999999999999998072e+3931  	-0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxsqr(z) = 	-1.17032409999999999999999999999442e+4031  	0.00000000000000000000000000000000e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxsqr(z) = 	-2.07225248399999999999999999999015e+4023  	-0.00000000000000000000000000000000e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxsqr(z) = 	-9.75452522499999999999999999995679e+3745  	4.32728970999999999999999999998093e+3727
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxsqr(z) = 	1.08675715599999999999999999999500e+3893  	-4.99243697199999999999999999997768e+3781
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxsqr(z) = 	3.71941416899999999999999999997968e+4621  	6.63672731399999999999999999996459e+4509
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxsqr(z) = 	-1.49761260099999999999999999999312e+3887  	-6.10631520999999999999999999997256e+3798
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	1.00197628408105432639763066379123e+2  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxsqr(z) = 	6.06380314855714247189890760302248e+1  	1.20954001463677349417136210730387e+2
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxsqr(z) = 	-3.23995997256602696009873909947566e+2  	0.00000000000000000000000000000000e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxsqr(z) = 	-2.67343265882868195251686399999999e+2  	-5.42079776410389615167999999999997e+2
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	3.23560125952636597002249999999998e+2  	-0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxsqr(z) = 	-7.22919218086038666578923899999996e+2  	1.25975171286118045196314799999999e+3
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxsqr(z) = 	-1.93490896980183787724969902499999e+3  	-0.00000000000000000000000000000000e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxsqr(z) = 	-1.61244283219351000237965567899999e+3  	-1.38146641004669385304833379999999e+3
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	1.00001975318374628607595976306637e+6  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxsqr(z) = 	6.06380314855714247189890760302248e+5  	1.20954001463677349417136210730387e+6
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxsqr(z) = 	-3.23995997256602696009873909947566e+4  	0.00000000000000000000000000000000e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxsqr(z) = 	1.64179292511242655147483135999999e+6  	-5.42079776410389615167999999999995e+5
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxsqr(z) = 	3.23560125952989517189472026236097e+6  	-0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxsqr(z) = 	-7.22923835290947822578923899999995e+4  	1.25975438668227582536314799999999e+5
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxsqr(z) = 	-1.61244283219351000237965567899999e+7  	-1.38146641004669385304833379999999e+7

Test of cxconj
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	2.31059999999999999999999999999571e-1571  	-0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxconj(z) = 	8.77919999999999999999999999999979e-20  	-9.77319999999999999999999999999977e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	-2.32979999999999999999999999999567e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxconj(z) = 	-4.55919999999999999999999999999990e-19  	-8.22189999999999999999999999999983e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	3.89319999999999999999999999999252e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxconj(z) = 	-1.22909999999999999999999999999998e-17  	7.33419999999999999999999999999984e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	4.09869999999999999999999999999216e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxconj(z) = 	3.99169999999999999999999999999991e-19  	8.22149999999999999999999999999982e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	1.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	1.00000000000000000009860000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxconj(z) = 	1.00000000000000000045670000000000e+0  	-2.42999999999999999999999999999995e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxconj(z) = 	1.00000000000000000000000000000000e+0  	-3.45059999999999999999999999999470e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxconj(z) = 	9.99999999999999998735999999999976e-1  	-4.66769999999999999999999999999284e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	9.99999999999999999753999999999976e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxconj(z) = 	9.99999999999999999337999999999976e-1  	5.07829999999999999999999999999312e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxconj(z) = 	1.00000000000000000000000000000000e+0  	6.70019999999999999999999999998969e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxconj(z) = 	1.00000000000000074230000000000000e+0  	5.33209999999999999999999999999153e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	-1.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	-9.99999999999999999988109999999973e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxconj(z) = 	-9.99999999999999999766319999999974e-1  	-2.33439999999999999999999999999996e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxconj(z) = 	-1.00000000000000000000000000000000e+0  	-4.56009999999999999999999999999462e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxconj(z) = 	-1.00000000000000000000654000000000e+0  	-7.43319999999999999999999999998655e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	-1.00000000000000000000743000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxconj(z) = 	-1.00000000000000000085023000000000e+0  	6.99829999999999999999999999999985e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxconj(z) = 	-1.00000000000000000000000000000000e+0  	5.00097999999999999999999999999395e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxconj(z) = 	-9.99999999999999999764299999999974e-1  	6.73209999999999999999999999999987e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxconj(z) = 	5.60979999999999999999999999998542e-2198  	-1.00000000000000000000000000000000e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxconj(z) = 	4.99869999999999999999999999999989e-18  	-1.00000000000000000054210000000000e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000320100000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxconj(z) = 	-6.55409999999999999999999999999984e-20  	-1.00000000000000000809810000000000e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxconj(z) = 	-7.09979999999999999999999999998153e-2200  	-1.00000000000000000000000000000000e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxconj(z) = 	-6.84319999999999999999999999999986e-17  	-9.99999999999999999986399999999974e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	-9.99999999999999890987319999999974e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxconj(z) = 	4.33289999999999999999999999999992e-16  	-9.99999999999999999787699999999974e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxconj(z) = 	2.77619999999999999999999999999307e-2111  	1.00000000000000000000000000000000e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxconj(z) = 	3.33209999999999999999999999999993e-17  	9.99999999999999999655499999999974e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	9.99999999999999999996999972012965e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxconj(z) = 	-7.55409999999999999999999999999985e-17  	9.99999999999999999987230999999972e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxconj(z) = 	-2.11559999999999999999999999999449e-2203  	1.00000000000000000000000000000000e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxconj(z) = 	-5.99229999999999999999999999999988e-16  	1.00000000000000000089321000000000e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000716100000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxconj(z) = 	8.66309999999999999999999999999981e-19  	1.00000000000000000008765000000000e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	1.58759999999999999999999999999648e+1876  	-0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	-6.43909999999999999999999999998503e+1965  	-0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	-3.42099999999999999999999999999185e+2015
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	4.55219999999999999999999999998917e+2011
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxconj(z) = 	2.19069999999999999999999999999520e+1854  	-9.87649999999999999999999999997812e+1872
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxconj(z) = 	-3.29659999999999999999999999999241e+1946  	-7.57209999999999999999999999998357e+1834
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxconj(z) = 	-6.09869999999999999999999999998333e+2310  	5.44109999999999999999999999998585e+2198
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxconj(z) = 	7.88949999999999999999999999998271e+1854  	3.86989999999999999999999999999111e+1943
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	1.00098765431000912340000000000000e+1  	-0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxconj(z) = 	9.89800000000065420089999999999997e+0  	-6.11002230065009876000000000000000e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	-1.79998888123400000076000000000000e+1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxconj(z) = 	-1.29822227000000000000000000000000e+1  	-2.08777722019200000000000000000000e+1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	-1.79877771265000000000000000000000e+1  	-0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxconj(z) = 	-1.90987221098100000000000000000000e+1  	3.29800000653999999999999999999999e+1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	4.39876001823449999999999999999999e+1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxconj(z) = 	1.59822011923610000000000000000000e+1  	4.32189031228999999999999999999999e+1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	1.00000987654310009123400000000000e+3  	-0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxconj(z) = 	9.89800000000065420089999999999996e+2  	-6.11002230065009875999999999999998e+2
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxconj(z) = 	0.00000000000000000000000000000000e+0  	-1.79998888123400000076000000000000e+2
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxconj(z) = 	-1.29822227000000000000000000000000e+3  	-2.08777722019199999999999999999999e+2
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxconj(z) = 	-1.79877771265098099999999999999999e+3  	-0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxconj(z) = 	-1.90987221098100000000000000000000e+2  	3.29800700653999999999999999999999e+2
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxconj(z) = 	1.59822011923609999999999999999999e+3  	4.32189031228999999999999999999998e+3

Test of cxneg
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	-2.31059999999999999999999999999571e-1571  	-0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxneg(z) = 	-8.77919999999999999999999999999979e-20  	-9.77319999999999999999999999999977e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	-2.32979999999999999999999999999567e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxneg(z) = 	4.55919999999999999999999999999990e-19  	-8.22189999999999999999999999999983e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	3.89319999999999999999999999999252e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxneg(z) = 	1.22909999999999999999999999999998e-17  	7.33419999999999999999999999999984e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	4.09869999999999999999999999999216e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxneg(z) = 	-3.99169999999999999999999999999991e-19  	8.22149999999999999999999999999982e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	-1.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	-1.00000000000000000009860000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxneg(z) = 	-1.00000000000000000045670000000000e+0  	-2.42999999999999999999999999999995e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxneg(z) = 	-1.00000000000000000000000000000000e+0  	-3.45059999999999999999999999999470e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxneg(z) = 	-9.99999999999999998735999999999976e-1  	-4.66769999999999999999999999999284e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	-9.99999999999999999753999999999976e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxneg(z) = 	-9.99999999999999999337999999999976e-1  	5.07829999999999999999999999999312e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxneg(z) = 	-1.00000000000000000000000000000000e+0  	6.70019999999999999999999999998969e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxneg(z) = 	-1.00000000000000074230000000000000e+0  	5.33209999999999999999999999999153e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	1.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	9.99999999999999999988109999999973e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxneg(z) = 	9.99999999999999999766319999999974e-1  	-2.33439999999999999999999999999996e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxneg(z) = 	1.00000000000000000000000000000000e+0  	-4.56009999999999999999999999999462e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxneg(z) = 	1.00000000000000000000654000000000e+0  	-7.43319999999999999999999999998655e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	1.00000000000000000000743000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxneg(z) = 	1.00000000000000000085023000000000e+0  	6.99829999999999999999999999999985e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxneg(z) = 	1.00000000000000000000000000000000e+0  	5.00097999999999999999999999999395e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxneg(z) = 	9.99999999999999999764299999999974e-1  	6.73209999999999999999999999999987e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxneg(z) = 	-5.60979999999999999999999999998542e-2198  	-1.00000000000000000000000000000000e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxneg(z) = 	-4.99869999999999999999999999999989e-18  	-1.00000000000000000054210000000000e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	-1.00000000000000000000320100000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxneg(z) = 	6.55409999999999999999999999999984e-20  	-1.00000000000000000809810000000000e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxneg(z) = 	7.09979999999999999999999999998153e-2200  	-1.00000000000000000000000000000000e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxneg(z) = 	6.84319999999999999999999999999986e-17  	-9.99999999999999999986399999999974e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	-9.99999999999999890987319999999974e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxneg(z) = 	-4.33289999999999999999999999999992e-16  	-9.99999999999999999787699999999974e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxneg(z) = 	-2.77619999999999999999999999999307e-2111  	1.00000000000000000000000000000000e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxneg(z) = 	-3.33209999999999999999999999999993e-17  	9.99999999999999999655499999999974e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	9.99999999999999999996999972012965e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxneg(z) = 	7.55409999999999999999999999999985e-17  	9.99999999999999999987230999999972e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxneg(z) = 	2.11559999999999999999999999999449e-2203  	1.00000000000000000000000000000000e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxneg(z) = 	5.99229999999999999999999999999988e-16  	1.00000000000000000089321000000000e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	1.00000000000000000000716100000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxneg(z) = 	-8.66309999999999999999999999999981e-19  	1.00000000000000000008765000000000e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	-1.58759999999999999999999999999648e+1876  	-0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	6.43909999999999999999999999998503e+1965  	-0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	-3.42099999999999999999999999999185e+2015
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	4.55219999999999999999999999998917e+2011
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxneg(z) = 	-2.19069999999999999999999999999520e+1854  	-9.87649999999999999999999999997812e+1872
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxneg(z) = 	3.29659999999999999999999999999241e+1946  	-7.57209999999999999999999999998357e+1834
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxneg(z) = 	6.09869999999999999999999999998333e+2310  	5.44109999999999999999999999998585e+2198
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxneg(z) = 	-7.88949999999999999999999999998271e+1854  	3.86989999999999999999999999999111e+1943
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	-1.00098765431000912340000000000000e+1  	-0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxneg(z) = 	-9.89800000000065420089999999999997e+0  	-6.11002230065009876000000000000000e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	-1.79998888123400000076000000000000e+1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxneg(z) = 	1.29822227000000000000000000000000e+1  	-2.08777722019200000000000000000000e+1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	1.79877771265000000000000000000000e+1  	-0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxneg(z) = 	1.90987221098100000000000000000000e+1  	3.29800000653999999999999999999999e+1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	4.39876001823449999999999999999999e+1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxneg(z) = 	-1.59822011923610000000000000000000e+1  	4.32189031228999999999999999999999e+1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	-1.00000987654310009123400000000000e+3  	-0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxneg(z) = 	-9.89800000000065420089999999999996e+2  	-6.11002230065009875999999999999998e+2
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxneg(z) = 	-0.00000000000000000000000000000000e+0  	-1.79998888123400000076000000000000e+2
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxneg(z) = 	1.29822227000000000000000000000000e+3  	-2.08777722019199999999999999999999e+2
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxneg(z) = 	1.79877771265098099999999999999999e+3  	-0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxneg(z) = 	1.90987221098100000000000000000000e+2  	3.29800700653999999999999999999999e+2
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxneg(z) = 	-1.59822011923609999999999999999999e+3  	4.32189031228999999999999999999998e+3

Test of cxfloor
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxfloor(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxfloor(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxfloor(z) = 	1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxfloor(z) = 	1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxfloor(z) = 	-2.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	-2.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxfloor(z) = 	-2.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxfloor(z) = 	-1.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-2.00000000000000000000000000000000e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxfloor(z) = 	2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxfloor(z) = 	-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxfloor(z) = 	-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxfloor(z) = 	7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	1.00000000000000000000000000000000e+1  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxfloor(z) = 	9.00000000000000000000000000000000e+0  	6.00000000000000000000000000000000e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	1.70000000000000000000000000000000e+1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxfloor(z) = 	-1.30000000000000000000000000000000e+1  	2.00000000000000000000000000000000e+1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	-1.80000000000000000000000000000000e+1  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxfloor(z) = 	-2.00000000000000000000000000000000e+1  	-3.30000000000000000000000000000000e+1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	-4.40000000000000000000000000000000e+1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxfloor(z) = 	1.50000000000000000000000000000000e+1  	-4.40000000000000000000000000000000e+1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	9.99999999999999999999999999999997e+2  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxfloor(z) = 	9.88999999999999999999999999999997e+2  	6.10999999999999999999999999999998e+2
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxfloor(z) = 	0.00000000000000000000000000000000e+0  	1.79000000000000000000000000000000e+2
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxfloor(z) = 	-1.29900000000000000000000000000000e+3  	2.08000000000000000000000000000000e+2
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxfloor(z) = 	-1.79899999999999999999999999999999e+3  	0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxfloor(z) = 	-1.91000000000000000000000000000000e+2  	-3.29999999999999999999999999999999e+2
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxfloor(z) = 	1.59800000000000000000000000000000e+3  	-4.32199999999999999999999999999999e+3

Test of cxceil
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	2.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxceil(z) = 	2.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxceil(z) = 	2.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxceil(z) = 	-1.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxceil(z) = 	-1.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxceil(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxceil(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	2.00000000000000000000000000000000e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	2.00000000000000000000000000000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	2.00000000000000000000000000000000e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxceil(z) = 	1.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxceil(z) = 	2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxceil(z) = 	-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxceil(z) = 	-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxceil(z) = 	7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	1.10000000000000000000000000000000e+1  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxceil(z) = 	1.00000000000000000000000000000000e+1  	7.00000000000000000000000000000000e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	1.80000000000000000000000000000000e+1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxceil(z) = 	-1.20000000000000000000000000000000e+1  	2.10000000000000000000000000000000e+1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	-1.70000000000000000000000000000000e+1  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxceil(z) = 	-1.90000000000000000000000000000000e+1  	-3.20000000000000000000000000000000e+1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxceil(z) = 	1.60000000000000000000000000000000e+1  	-4.30000000000000000000000000000000e+1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	1.00100000000000000000000000000000e+3  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxceil(z) = 	9.89999999999999999999999999999997e+2  	6.11999999999999999999999999999998e+2
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxceil(z) = 	0.00000000000000000000000000000000e+0  	1.80000000000000000000000000000000e+2
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxceil(z) = 	-1.29800000000000000000000000000000e+3  	2.09000000000000000000000000000000e+2
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxceil(z) = 	-1.79799999999999999999999999999999e+3  	0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxceil(z) = 	-1.90000000000000000000000000000000e+2  	-3.28999999999999999999999999999999e+2
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxceil(z) = 	1.59900000000000000000000000000000e+3  	-4.32099999999999999999999999999999e+3

Test of cxround
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxround(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxround(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxround(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxround(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxround(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxround(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxround(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxround(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxround(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxround(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxround(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxround(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxround(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxround(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxround(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxround(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxround(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxround(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxround(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxround(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxround(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxround(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxround(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxround(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxround(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxround(z) = 	1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxround(z) = 	-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxround(z) = 	0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxround(z) = 	2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxround(z) = 	-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxround(z) = 	-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxround(z) = 	7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxround(z) = 	1.00000000000000000000000000000000e+1  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxround(z) = 	1.00000000000000000000000000000000e+1  	6.00000000000000000000000000000000e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.80000000000000000000000000000000e+1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxround(z) = 	-1.30000000000000000000000000000000e+1  	2.10000000000000000000000000000000e+1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxround(z) = 	-1.80000000000000000000000000000000e+1  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxround(z) = 	-1.90000000000000000000000000000000e+1  	-3.30000000000000000000000000000000e+1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxround(z) = 	0.00000000000000000000000000000000e+0  	-4.40000000000000000000000000000000e+1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxround(z) = 	1.60000000000000000000000000000000e+1  	-4.30000000000000000000000000000000e+1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxround(z) = 	9.99999999999999999999999999999997e+2  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxround(z) = 	9.89999999999999999999999999999997e+2  	6.10999999999999999999999999999998e+2
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxround(z) = 	0.00000000000000000000000000000000e+0  	1.80000000000000000000000000000000e+2
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxround(z) = 	-1.29800000000000000000000000000000e+3  	2.09000000000000000000000000000000e+2
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxround(z) = 	-1.79899999999999999999999999999999e+3  	0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxround(z) = 	-1.91000000000000000000000000000000e+2  	-3.29999999999999999999999999999999e+2
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxround(z) = 	1.59800000000000000000000000000000e+3  	-4.32199999999999999999999999999999e+3

Test of cxtrunc
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxtrunc(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxtrunc(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxtrunc(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxtrunc(z) = 	1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxtrunc(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxtrunc(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxtrunc(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxtrunc(z) = 	-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxtrunc(z) = 	2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxtrunc(z) = 	-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxtrunc(z) = 	-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxtrunc(z) = 	7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	1.00000000000000000000000000000000e+1  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxtrunc(z) = 	9.00000000000000000000000000000000e+0  	6.00000000000000000000000000000000e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	1.70000000000000000000000000000000e+1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxtrunc(z) = 	-1.20000000000000000000000000000000e+1  	2.00000000000000000000000000000000e+1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	-1.70000000000000000000000000000000e+1  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxtrunc(z) = 	-1.90000000000000000000000000000000e+1  	-3.20000000000000000000000000000000e+1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	-4.30000000000000000000000000000000e+1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxtrunc(z) = 	1.50000000000000000000000000000000e+1  	-4.30000000000000000000000000000000e+1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	9.99999999999999999999999999999997e+2  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxtrunc(z) = 	9.88999999999999999999999999999997e+2  	6.10999999999999999999999999999998e+2
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxtrunc(z) = 	0.00000000000000000000000000000000e+0  	1.79000000000000000000000000000000e+2
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxtrunc(z) = 	-1.29800000000000000000000000000000e+3  	2.08000000000000000000000000000000e+2
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxtrunc(z) = 	-1.79799999999999999999999999999999e+3  	0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxtrunc(z) = 	-1.90000000000000000000000000000000e+2  	-3.28999999999999999999999999999999e+2
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxtrunc(z) = 	1.59800000000000000000000000000000e+3  	-4.32099999999999999999999999999999e+3

Test of cxfrac
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxfrac(z) = 	8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxfrac(z) = 	-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxfrac(z) = 	-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxfrac(z) = 	3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	9.85999999999999991962888386857995e-20  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxfrac(z) = 	4.56699999999999657579309096887778e-19  	2.42999999999999999999999999999995e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxfrac(z) = 	9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxfrac(z) = 	9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxfrac(z) = 	7.42299999999999999873237712882656e-16  	-5.33209999999999999999999999999153e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	-0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxfrac(z) = 	-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxfrac(z) = 	-0.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxfrac(z) = 	-6.53999999999965859221116525021664e-21  	7.43319999999999999999999999998655e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	-7.42999999999965924907034553009879e-21  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxfrac(z) = 	-8.50229999999999904742448791042056e-19  	-6.99829999999999999999999999999985e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxfrac(z) = 	-0.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxfrac(z) = 	-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxfrac(z) = 	5.60979999999999999999999999998542e-2198  	0.00000000000000000000000000000000e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxfrac(z) = 	4.99869999999999999999999999999989e-18  	5.42099999999999988940081119268881e-19
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	3.20099999999961804207653135147784e-21
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxfrac(z) = 	-6.55409999999999999999999999999984e-20  	8.09809999999999968826843756764752e-18
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxfrac(z) = 	-7.09979999999999999999999999998153e-2200  	0.00000000000000000000000000000000e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxfrac(z) = 	-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxfrac(z) = 	4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxfrac(z) = 	2.77619999999999999999999999999307e-2111  	-0.00000000000000000000000000000000e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxfrac(z) = 	3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxfrac(z) = 	-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxfrac(z) = 	-2.11559999999999999999999999999449e-2203  	-0.00000000000000000000000000000000e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxfrac(z) = 	-5.99229999999999999999999999999988e-16  	-8.93209999999999733050755303434376e-19
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	-7.16099999999963394852823947703790e-21
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxfrac(z) = 	8.66309999999999999999999999999981e-19  	-8.76499999999996838315021790765852e-20
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	-0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxfrac(z) = 	-0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxfrac(z) = 	-0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	9.87654310009123399999999999999934e-3  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxfrac(z) = 	8.98000000000654200899999999999980e-1  	1.10022300650098759999999999999997e-1
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	9.99888812340000007599999999999974e-1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxfrac(z) = 	-9.82222699999999999999999999999992e-1  	8.77772201919999999999999999999984e-1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	-9.87777126499999999999999999999985e-1  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxfrac(z) = 	-9.87221098099999999999999999999946e-2  	-9.80000065399999999999999999999980e-1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	-9.87600182344999999999999999999978e-1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxfrac(z) = 	9.82201192360999999999999999999986e-1  	-2.18903122899999999999999999999990e-1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	9.87654310009123399999999999998393e-3  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxfrac(z) = 	8.00000000065420089999999999999913e-1  	2.23006500987599999999999999989881e-3
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxfrac(z) = 	0.00000000000000000000000000000000e+0  	9.98888123400000075999999999999965e-1
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxfrac(z) = 	-2.22269999999999999999999999999666e-1  	7.77722019199999999999999999999943e-1
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxfrac(z) = 	-7.77712650980999999999999999999819e-1  	0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxfrac(z) = 	-9.87221098099999999999999999999983e-1  	-8.00700653999999999999999999999933e-1
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxfrac(z) = 	2.20119236099999999999999999999774e-1  	-8.90312289999999999999999999999490e-1

*/
