/* demux.h -- I/O demultiplexer, timer and secure signal handler
   Copyright (C) 2004 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifndef DEMUX_H
#define DEMUX_H 1

/* Timeout or periodic alarm identifier. */
struct timer_pq_node;
typedef struct timer_pq_node *timer_id;

/* Begin demultiplexing. Doesn't return until dmx_stop() is called. */
extern void dmx_main_loop ();

/* Exit dmx_main_loop(). Registered fd's and timers remain. */
extern void dmx_stop ();

/* Clean up all registered fd's and timers. You must free all data
   associated to fd's and timers, e.g. by calling dmx_remove_timer(),
   etc, before calling this, and free()ing the returned pointer. */
extern void dmx_cleanup ();

/* Call 'callback' with 'fd' and 'data' as parameters when some data is
   available for reading in 'fd'. */
extern void dmx_add_input_fd (int fd, void (*callback) (int, void *),
                              void *data);

/* Call 'callback' with 'fd' and 'data' as parameters when data can be
   sent to 'fd'. */
extern void dmx_add_output_fd (int fd, void (*callback) (int, void *),
                               void *data);

/* Check if data is available in 'fd'. Returns boolean. */
extern int dmx_chk_input_fd (int fd);

/* Check if data can be sent to 'fd'. Returns boolean. */
extern int dmx_chk_output_fd (int fd);

/* Call 'callback' with 'data' as parameter every 'seconds' seconds,
   taking last midnight plus 'shift' seconds as reference.
   If shift is -1, no reference is taken, and 'callback' is not
   called until 'seconds' seconds have happened. */
extern timer_id dmx_add_periodic_alarm (unsigned long int seconds,
					long int shift,
					void (*callback) (void *),
					void *data);

/* Call 'callback' with 'data' as parameter in 'seconds' seconds from now. */
extern timer_id dmx_add_timer (unsigned long int seconds,
			       void (*callback) (void *), void *data);

/* Check if timer needs to be dispatched and do it. Use when you are
   spending too much time in a callback. */
extern void dmx_dispatch_timer (timer_id id);

/* Call 'callback' when signal 'signum' arrives. This function must be used
   instead of signal() if the handler calls demux functions. You may use it
   for other handlers as well, for increased security and stability thru
   synchronous processing of signals. Works like BSD/GNU signal() even in
   POSIX environments, ie. the handler keeps assigned to the signal after
   that signal arrives. */
extern void dmx_signal (int signum, void (*callback) (int));

/* Stop reporting data availability in 'fd'. Returns 'data' as passed
   on creation. */
extern void *dmx_remove_input_fd (int fd);

/* Stop reporting availability to send data to 'fd'. Return 'data' as
   passed on creation. */
extern void *dmx_remove_output_fd (int fd);

/* Remove a timer or periodic alarm. Returns 'data' as passed on creation. */
extern void *dmx_remove_timer (timer_id id);

/* Stop reporting signal 'signum'. Restores default action for the signal. */
extern void dmx_remove_signal (int signum);

#endif
