/* parse.c -- parses lines written by the user
   Copyright (C) 2004 Julio A. Becerra

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define _POSIX_SOURCE
#define _ISOC99_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "common.h"
#include "parse.h"
#include "user_iface.h"
#include "protocol.h"
#include "contact_list.h"
#include "uconfig.h"
#include "transport.h"
#include "misc.h"

#define ADD_C 0
#define BEEP_C 1
#define CLOSE_C 2
#define DELETE_C 3
#define HELP_C 4
#define INFO_C 5
#define NICK_C 6
#define QUIT_C 7
#define SET_C 8
#define COMMANDS_NUM 9
#define C_LIST 1000

#define BEEP_V 0
#define LOGGING_V 1
#define PORT_V 2
#define TIMESTAMP_V 3
#define VARS_NUM 4

#define MAX_L 12
#define MAX_W 10

#define ERR_UNK (ERR_LOCAL -1)
#define ERR_AMB (ERR_LOCAL -2)


char commands [COMMANDS_NUM][MAX_L] = {"add", "beep", "close", "delete", "help",
                                       "info", "nick", "quit", "set"};
char vars [VARS_NUM][MAX_L] = {"beep", "logging", "port", "timestamps"};

static int get_words (char **t, const char *str, int max);
static void free_words (char **t, int max);
static int command_id (char *str, char (*cl)[MAX_L], int num);
static void help (int c_id, char *opt);
static void print_contact_info (contact_t *c);
static void toggle_setting (BOOL *set, char *p);
static void set_command (char *var, char *p);
static void set_help (int var_id);

void 
pa_parse_line (contact_t *win_contact, const char *line)
{
	char *words [MAX_W];
	int co;
	int co_h;
	int w;
	contact_t *ct;
		
	if (line [0] != '/') {
		if (win_contact) {
			pr_send_text (win_contact, line);
		}
		return;
	}
	
	for (w = 0; w < MAX_W; w++)
		words [w] = NULL;
	
	//ui_output_err (line);
	
	w = get_words (words, line, MAX_W);
	co = command_id (words [0]+1, commands, COMMANDS_NUM);
	
	switch (co) {
		case ADD_C:
			if (w > 2) {
				ct = malloc (sizeof (contact_t));
				if (ct == NULL) {
					dmx_stop ();
					return;
				}
				strncpy (ct->nick, words [1], MAX_NICK-1);
				ct->nick [MAX_NICK-1]='\0';
				strncpy (ct->hname, words [2], MAX_HNAME-1);
				ct->hname [MAX_HNAME-1]='\0';
				if (w == 4)
					ct->port = (port_t) strtoul (words [3],
					                             NULL, 10);
				else
					ct->port = DEFAULT_PORT;
				ct->next = NULL;
				ct->state.ip = 0;
				ct->state.socket=-1;
				ct->state.connected=FALSE;
				ct->state.hello_pending=TRUE;
				ct->state.in_buffer=NULL;
				ct->state.in_buf_size=0;
				ct->state.in_buf_rcvd=0;
				ct->state.out_buffer=NULL;
				ct->state.out_buf_size=0;
				ct->state.hello_timeout=NULL;
				ct->state.ui_info=NULL;
				cl_add_contact (ct);
			}
			else
				ui_output_err ("Not enough parameters.");
			break;
		case BEEP_C:
			if (w > 1) {
				if (strcmp (words[1], "*") == 0) {
					pr_send_beep (BROADCAST);
				}
				else {
					contact_t *c;
					c = cl_find_by_nick (words[1]);
					if (c == NULL) {
						ui_output_err ("Contact %s not "
						               "found.",
							       words[1]);
					}
					else {
						pr_send_beep (c);
					}
				}
			}
			else {
				contact_t *c = ui_current_contact ();
				if (c == NULL) {
					ui_output_err ("Contact not "
					               "connected.");
				}
				else {
					pr_send_beep (c);
				}
			}
			break;
		case CLOSE_C:
			if (w > 1) {
				contact_t *c = cl_find_by_nick (words[1]);
				if (c == NULL) {
					ui_output_err ("Contact %s not found.",
					               words[1]);
				}
				else {
					ui_close_win (c);
				}
			}
			else
				ui_close_cur_win ();
			break;
		case DELETE_C:
			if (w > 1)
				cl_remove_contact (words [1]);
			else
				ui_output_err ("Not enough parameters.");
			break;
		case HELP_C:
			if (w > 1) {
				co_h = command_id (words [1], commands, 
						   COMMANDS_NUM);
				if (co_h == ERR_UNK)
					ui_output_err ("No help for %s.", 
							words[1]);
				else if (co_h == ERR_AMB)
					ui_output_err ("Ambiguous command: %s", 
							words[1]);
				else
					help (co_h, words[2]);
			}
			else
				help (C_LIST, NULL);
			break;
		case INFO_C:
			if (w > 1) {
				contact_t *c = cl_find_by_nick (words[1]);
				if (c == NULL) {
					ui_output_err ("Contact %s not found.",
					               words[1]);
				}
				else {
					print_contact_info (c);
				}
			}
			else
				ui_output_err ("Not enough parameters.");
			break;
		case NICK_C:
			if (w > 1) {
				strncpy (cfg.nick, words[1], MAX_NICK-1);
				cfg.nick [MAX_NICK-1] = '\0';
				cf_save ();
				ui_redraw_nick ();
				pr_send_hello (BROADCAST);
			}
			else
				ui_output_err ("Not enough parameters.");
			break;
		case SET_C:
			set_command (words[1], words[2]);
			break;		
		case QUIT_C:
			dmx_stop ();
			break;
		case ERR_UNK:
			ui_output_err ("Unknown command: %s", words [0]+1);
			break;
		case ERR_AMB:
			ui_output_err ("Ambiguous command: %s", words [0]+1);
			break;
		case ERR:
			dmx_stop();
	}

	free_words (words, MAX_W);
}

static int 
get_words (char **t, const char *str, int max)
{
	int i = 0, j = 0, w = 0;
	
	while (w < max) {
		if (isspace (str [j]) || str[j] == '\0') {
			if (i != j) {
				t[w] = (char*) malloc ((j-i+1) * sizeof (char));
				strncpy (t[w], str+i, j-i);
				t[w][j-i]='\0';
				w++;
			}
			while (isspace (str[j]))
				j++;
			if (str[j] == '\0')
				break;
			i = j;			
		}
		else
			j++;
	}

	return w;
}

static void 
free_words (char **t, int max)
{
	int i = 0;

	while (i<max && t[i] != NULL) {
		free (t[i]);
		i++;
	}
}

static int 
command_id (char *str, char (*cl)[MAX_L], int num)
{
	int i = 0;
	int ret = ERR_UNK;
	char *c;

	c = malloc ((strlen (str) + 1) * sizeof (char));
	if (c == NULL)
		return ERR;

	strcpy (c, str);

	while (c[i] != '\0') {
		c[i] = tolower ((int) c[i]);
		i++;
	}
	
	i = 0;
	while (i < num) {
		if (strncmp (c, cl[i], strlen (c)) == 0) {
			if (ret == ERR_UNK)
				ret = i;
			else {
				ret = ERR_AMB;
				break;
			}
		}	

		i++;
	}

	free (c);
	return ret;	
}

static void
help (int c_id, char *opt)
{
	int v_id;
	
	switch (c_id) {
		case ADD_C:
			ui_output_info ("");
			ui_output_info ("ADD <nick> <host> [<port>]");
			ui_output_info ("");
			ui_output_info ("Creates a new contact. Host may be "
			                "given as an IP address or as a "
			                "hostname.");
			break;
		case BEEP_C:
			ui_output_info ("");
			ui_output_info ("BEEP [<nick>]");
			ui_output_info ("");
			ui_output_info ("Beeps current or specified contact "
			                "(use \"*\" to beep everybody).");
			break;
		case CLOSE_C:
			ui_output_info ("");
			ui_output_info ("CLOSE [<nick>]");
			ui_output_info ("");
			ui_output_info ("Closes current or specified window.");
			break;
		case DELETE_C:
			ui_output_info ("");
			ui_output_info ("DELETE <nick>");
			ui_output_info ("");
			ui_output_info ("Deletes a contact.");
			break;
		case HELP_C:
			ui_output_info ("");
			ui_output_info ("HELP [<command>]");
			ui_output_info ("");
			ui_output_info ("Shows help on commands.");
			break;
		case INFO_C:
			ui_output_info ("");
			ui_output_info ("INFO <nick>");
			ui_output_info ("");
			ui_output_info ("Shows information about a contact.");
			break;
		case NICK_C:
			ui_output_info ("");
			ui_output_info ("NICK <new nick>");
			ui_output_info ("");
			ui_output_info ("Changes your nick.");
			break;
		case SET_C:
			if (opt == NULL) {
				ui_output_info ("");
				ui_output_info ("SET <setting> [<value>]");
				ui_output_info ("");
				ui_output_info ("Changes IPchat settings.");
				ui_output_info ("You can use /HELP SET " 
						"[<setting>] to get more help "
					        "about a particular setting.");
				ui_output_info ("");
				ui_output_info ("\\b\\3Available settings");
				ui_output_info ("\\b\\3------------------");
				set_command (NULL, NULL);
			}
			else {
  				v_id = command_id (opt, vars, VARS_NUM);
				if (v_id == ERR_UNK)
					ui_output_err ("Unknown setting: %s.", 
							opt);
				else if (v_id == ERR_AMB)
					ui_output_err ("Ambiguous setting: %s",
							opt);
				else
					set_help (v_id);
			}
			break;
		case QUIT_C:
			ui_output_info ("");
			ui_output_info ("QUIT");
			ui_output_info ("");
			ui_output_info ("Ends your ipchat session.");
			break;
		case C_LIST:
			ui_output_info ("");
			ui_output_info ("\\b\\3Available commands");
			ui_output_info ("\\b\\3------------------");
			ui_output_info ("\\bADD");
			ui_output_info ("\\bBEEP");
			ui_output_info ("\\bCLOSE");
			ui_output_info ("\\bDELETE");
			ui_output_info ("\\bHELP");
			ui_output_info ("\\bINFO");
			ui_output_info ("\\bNICK");
			ui_output_info ("\\bSET");
			ui_output_info ("\\bQUIT");
			ui_output_info ("");
			ui_output_info ("For information on a command use "
			                "/help <command>.");
			ui_output_info ("Command name abbreviations are "
			                "allowed (eg: /p).");
			ui_output_info ("Text formatting: \\b\\\\b for bold, "
			                "\\u\\\\u for underline\\u, \\\\1 to "
			                "\\\\7 for colors, \\\\0 to reset.");
			ui_output_info ("");
	}
}

static void
set_help (int var_id)
{
	switch (var_id) {
		case PORT_V:
			ui_output_info ("");
			ui_output_info ("SET PORT [<port>]");
			ui_output_info ("");
			ui_output_info ("Shows or changes your listening "
					"port.");
			break;
		case LOGGING_V:
			ui_output_info ("");
			ui_output_info ("SET LOGGING [ON|OFF]");
			ui_output_info ("");
			ui_output_info ("Activates/deactivates logging.");
			break;
		case TIMESTAMP_V:
 			ui_output_info ("");
                        ui_output_info ("SET TIMESTAMPS [ON|OFF]");
			ui_output_info ("");
			ui_output_info ("Activates/deactivates timestamps.");					       
			break;
		case BEEP_V:
       	     		ui_output_info ("");	     
		      	ui_output_info ("SET BEEP [ON|OFF]");
			ui_output_info ("");
			ui_output_info ("Activates/deactivates beep.");
			break;
	}
}

static void
print_contact_info (contact_t *c)
{
	ip_t old_ip = c->state.ip;

	/* user may use /info to force IP resolution */
	tr_resolv (c);
	if (old_ip == 0) {
		/* try to connect */
		tr_connect (c);
	}

	ui_output_info ("");
	ui_output_info ("\\b\\3Contact Info");
	ui_output_info ("\\b\\3------------");
	ui_output_info ("\\b\\2Nickname:      \\0%s", c->nick);
	ui_output_info ("\\b\\2Host:          \\0%s", c->hname);
	if (c->state.ip == 0) {
		ui_output_info ("\\b\\2IP address:    \\0Not found!");
	}
	else {
		ui_output_info ("\\b\\2IP address:    \\0%s",
		                ipv4_to_string (c->state.ip));
	}
	ui_output_info ("\\b\\2Listen port:   \\0%u", c->port);
	ui_output_info ("\\b\\2State:         \\0%s",
	                (!c->state.connected) ? "Disconnected" :
	                c->state.hello_pending ? "Waiting HELLO..."
	                                       : "Connected");
	ui_output_info ("");
}

static void
set_command (char *var, char *p)
{
	port_t port;
	char *ptr;
	BOOL u = FALSE;
	int i=0;
	
	if (p == NULL) {
		if (var == NULL)
			var = "\0";
		else {
			while (var[i] != '\0') {
				var[i] = tolower ((int) var[i]);
     			      	i++;
			}
		}
		
		if (strncmp (vars[0], var, strlen(var)) == 0) {
	
			u = TRUE;
			if (cfg.beep)
				ui_output_info ("\\bbeep = ON");
			else
				ui_output_info ("\\bbeep = OFF");
		}
		if (strncmp (vars[1], var, strlen(var)) == 0) {

			u = TRUE;
			if (cfg.logging)
				ui_output_info ("\\blogging = ON");
			else
				ui_output_info ("\\blogging = OFF");
		}
	
		if (strncmp (vars[2], var, strlen(var)) == 0) {
			u = TRUE;
			ui_output_info ("\\bport = %u", cfg.listen_port);
		}
		if (strncmp (vars[3], var, strlen(var)) == 0) {
			u = TRUE;
			if (cfg.timestamps)
				ui_output_info ("\\btimestamps = ON");
			else
				ui_output_info ("\\btimestamps = OFF");
		}
	
		if (!u)
			ui_output_err ("Unknown setting: %s", var);
		
	}

	else {
		switch (command_id (var, vars, VARS_NUM)) {
			case BEEP_V:
				toggle_setting (&cfg.beep, p);
				break;
			case LOGGING_V:
				toggle_setting (&cfg.logging, p);
				break;
			case PORT_V:
				port = (port_t) strtoul (p, &ptr, 10);
				if (*ptr != '\0')
					ui_output_err ("Invalid port number.");
				else if (cfg.listen_port == port) {
					ui_output_info ("That's the current "
					                "listen port.");
				}
				else {
					cfg.listen_port = port;
					pr_send_hello (BROADCAST);
					ui_output_info ("Listen port changed "
					                "to %u. You need to "
					                "restart ipchat.",
					                cfg.listen_port);
				}
				break;
			case TIMESTAMP_V:
				toggle_setting (&cfg.timestamps, p);
				break;
			case ERR_AMB:
				ui_output_err ("Ambiguous setting: %s", var);
				break;
			case ERR:
				dmx_stop();
		} 
		set_command (var, NULL);
		cf_save();
	}
}

static void
toggle_setting (BOOL *set, char *p)
{
	int i = 0;

	while (p[i] != '\0') {
		p[i] = tolower ((int) p[i]);
		i++;
	}
		
	if (strcmp (p, "on") == 0)
		*set = TRUE;
	else if (strcmp (p, "off") == 0)
		*set = FALSE;
	else
		ui_output_err ("Value must be ON or OFF");
}
