/* contact_list.c -- manages the list of contacts
   Copyright (C) 2004 Julio A. Becerra

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#define _POSIX_SOURCE
#define _ISOC99_SOURCE
#define _XOPEN_SOURCE		/* TODO we should not need this... */
#define _XOPEN_SOURCE_EXTENDED	/* ... bug in GNU libc6? */

#include <ctype.h>  
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include "user_iface.h"
#include "contact_list.h"
#include "misc.h"
#include "common.h"
#include "transport.h"

#define IPCHAT_DIR ".ipchat"
#define CONTACTS_FILE ".ipchat/contacts"
#define CONTACTS_FILE_C ".ipchat/contacts_old"

/* Version of the file format. Minor number changes when the format
   changes, but older versions with the same major number can be converted.
   Major number changes when an incompatible change is done. */
#define MAJOR_VERSION 1
#define MINOR_VERSION 0
#define CONT_VERSION "1.0"
#define CONT_VERSION_SIZE 3

#define VERS_BUF_SIZE 8
#define PORT_BUF_SIZE 8
#define CONT_BUF_SIZE 1024
#define CONTACT_RET 3

static int cl_load_1_0 (int fd);
static int cl_save_1_0 (int fd);
static void free_list ();

contact_t *contacts = NULL;
	

int 
cl_load ()
{
	char *file;
	int fd;
	ssize_t ret;
	char vers[VERS_BUF_SIZE];
	int r;
		
	file = home_path (CONTACTS_FILE);	
	if (file == NULL)
		return ERR;
		
	fd = open (file, 0, 0);
	free (file);
	if (fd == -1) {
		if (errno == ENOENT)
			return OK;
		if (errno == EACCES)
			ui_output_err ("Cannot access contacts file");
		return ERR;
	}

	ret = read_tag (fd, NULL, "version", vers, VERS_BUF_SIZE);
	if (ret == ERR) {
		close (fd);
		return ERR;
	}
	
	else if (ret != 0)
		ui_output_err ("Invalid contacts file");
		    
	else {
		if (strcmp (CONT_VERSION, vers) == 0) {
			lseek (fd, 0, SEEK_SET);
			r = cl_load_1_0 (fd);
			if (r == ERR) {
				close (fd);
				return ERR;
			}
			else if (r == ERR_C)
				ui_output_err ("Invalid contacts file");
		} 
		else 
 			ui_output_err ("Unknown contacts file version");	
	}
	
	close (fd);
	ui_redraw_contacts ();
	return OK;
}

static int
cl_load_1_0 (int fd)
{
	contact_t **pc;
	int r;
	char *b;
	char port [PORT_BUF_SIZE];
	
	pc = &contacts;
	
	b = malloc (CONT_BUF_SIZE * sizeof (char));
	if (b == NULL)
		return ERR;
	
	while (1) {
		r = read_tag (fd, NULL, "contact", b, CONT_BUF_SIZE);
		CHECK (r != ERR);
		
		if (r == ERR_TAG)
			break;
		
		else if (r == CONTACT_RET) {	
			*pc = ct_new_contact ();
			CHECK (*pc != NULL);
				
			r = read_tag (0, b, "hname", (*pc)->hname, MAX_HNAME);
				
			if (r == 0)
				r=read_tag (0, b, "nick", (*pc)->nick,MAX_NICK);
			
			if (r == 0)
				r = read_tag (0, b, "port", port,PORT_BUF_SIZE);
						
			if (r == 0) {
				(*pc)->port = (port_t) strtoul (port, NULL, 10);
				pc = &((*pc)->next);
			}
			else {
				free (*pc);
				*pc = NULL;
				ui_output_err ("Invalid contact"); 
			}
		}
		else
			ui_output_err ("Invalid contact");
	}

	free (b);
	return OK;
	
error:
	free (b);
	free_list ();
	return ERR;
}	



static void
free_list ()
{
	contact_t *pc;

	while (contacts != NULL) {
		pc = contacts;
		contacts = contacts->next;
		free (pc);
	}

	contacts = NULL;
}

int 
cl_save ()
{
	char *file;
	char *file2;
	int fd;
	char vers[VERS_BUF_SIZE];
	int r;
					
        file = home_path (CONTACTS_FILE);
        if (file == NULL)
		return ERR;

	fd = open (file, O_CREAT | O_RDWR,  S_IRUSR | S_IWUSR);
	if (fd == -1) {
		if (errno == EACCES)
			ui_output_err ("Cannot access contacts file.");
		free (file);
		return ERR;
	}

	r = read_tag (fd, NULL, "version", vers, VERS_BUF_SIZE); 
	if (r == ERR) {
		free (file);
		close (fd);
		return ERR;
	}

		
	if (r == 0 && strncmp (vers, CONT_VERSION, CONT_VERSION_SIZE) != 0) {
		close (fd);
		file2 = home_path (CONTACTS_FILE_C);
		if (file == NULL) {
			free (file);
			return ERR;
		}

		r = rename (file, file2);
		free (file2);
		if (r == -1) {
			free (file);
			return ERR;
		}
	
		fd = open (file, O_CREAT | O_RDWR, S_IRUSR | S_IWUSR);
		if (fd == -1) {
			if (errno == EACCES)
				ui_output_err ("Cannot access contacts file");
			free (file);

			return ERR;
		}
	}
	else
		lseek (fd, 0, SEEK_SET); 
         
	free (file);
		
	if (ftruncate (fd, 0) < 0) {
		close (fd);
		return ERR;
	}

	r = write_tag (fd, "version", CONT_VERSION, 0);
	if (r == ERR) {
		close (fd);
		return ERR;
	}

	r = cl_save_1_0 (fd);
	if (r != OK) {
		close (fd);
		return ERR;
	}

	close (fd);
	return OK;
}


contact_t*
cl_find_by_nick (const char *nick)
{
	contact_t *pc;

	pc = contacts;

	while (pc != NULL) {
		if (strncmp (nick, pc->nick, MAX_NICK) == 0)
			return pc;
		pc=pc->next;
	}
		
	return NULL;
}

static int
cl_save_1_0 (int fd)
{
	contact_t *pc;
	char b[PORT_BUF_SIZE];
	
	pc = contacts;

	while (pc != NULL) {
		if (pc->nick[0] != '\0') {
			CHECK (write_tag (fd, "contact", NULL, 0) != ERR);
			CHECK (write_tag (fd, "hname", pc->hname, 1) != ERR);
			CHECK (write_tag (fd, "nick", pc->nick, 1) != ERR);
			snprintf (b, PORT_BUF_SIZE, "%u", pc->port);
			CHECK (write_tag (fd, "port", b, 1) != ERR);
			CHECK (write_tag (fd, NULL, NULL, 0) != ERR);
		}
		pc=pc->next;
	}

	return OK;

error:
	return ERR;
}
	
	
contact_t*
cl_find_by_ip (ip_t ip)
{
	contact_t *pc;
	
	pc = contacts;
	
	while (pc != NULL) {
		if (pc->state.ip == ip)
			return pc;
		pc=pc->next;
	}
	
	return NULL;
}

void 
cl_add_contact (contact_t *contact)
{
	contact_t **pc;

	pc = &contacts;

	while (*pc != NULL) {
		if (( (strcmp ((*pc)->nick, contact->nick) == 0) &&
		      contact->nick[0] != '\0') ||
		    ( (strcmp ((*pc)->hname, contact->hname) == 0) &&
		    ((*pc)->port == contact->port) )) {
			ui_output_err ("Contact already in list");
			return;
		}
			
		pc = &((*pc)->next);
	}

	*pc=contact;

	if (contact->nick[0]) {
		cl_save ();
		ui_output_info ("Contact created: %s", contact->nick);
		ui_redraw_contacts ();
	}

	if (contact->state.ip == 0) {
		/* try to contact it */
		tr_resolv (contact);
		tr_connect (contact);
	}
}

void 
cl_remove_contact (const char *nick)
{
	contact_t **pc;
	contact_t *p;
	
	pc = &contacts;
	
	while (*pc != NULL) {
		if (strncmp (nick, (*pc)->nick, MAX_NICK) == 0) {
			p = *pc;
			if (!CT_IS_CONNECTED (p)) {
				if (p->state.ui_info) {
					/* the contact has a window */
					ui_close_win (p);
				}
				(*pc) = (*pc)->next;
				if (p->state.socket >= 0)
					tr_disconnect (p); 
				free (p);	
				cl_save ();
				ui_output_info ("Contact deleted: %s", nick);
				ui_redraw_contacts ();
			}
			else {
				ui_output_err ("Cannot delete connected "
				               "contact.");
			}
			return;
		}
		
		pc = &((*pc)->next);
	}
}	


