/* ui_format.c -- text formatting for the ncurses user interface
   Copyright (C) 2004 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#define _POSIX_SOURCE

#include <stdio.h>		/* stdlib.h (on some systems) */
#include <stdlib.h>		/* malloc, free */
#include <string.h>		/* memcpy, memset */
#include <time.h>		/* time, strftime, localtime */
#if TM_IN_SYS_TIME
#include <sys/time.h>		/* struct tm */
#endif
#include "ui_common.h"
#include "uconfig.h"

#define DEBUG_PRINT_SEQS FALSE

#define ALIGN_PAD 1             /* additional spaces between nick and text */

static BOOL on_seq = FALSE;     /* a sequence started in format_waddnstr() */

/* Prototypes */
static void remove_first_line ();
static int print_cyclic (int first, int last);
static inline void reset_format_waddnstr ();
static const char *get_timestamp ();

/* Build a line and add it to the text buffer of contact window 'cur'.
   The line will be 'pref'+'nick'+'suff'+<align spaces>+'text'.
   If 'cur' is 'vcur', the line will be shown immediately. */
void
build_line (const char *pref, const char *nick, const char *suff,
            const wchar_t *text, int lpref, int lnick, int lsuff, int ltext)
{
	wchar_t *buf, *p;
	int   len, align;

	calc_align ();
	align = cur->align - lnick - NICK_PREF_SUFF_LEN;
	if (cfg.timestamps)
		align -= TSTAMP_LEN + TSTAMP_PREF_SUFF_LEN;

	len = 1 + lpref + lnick + lsuff + align + ltext;
	if (cfg.timestamps)
		len += TSTAMP_LEN + strlen (TSTAMP_PREF) + strlen (TSTAMP_SUFF);

	buf = malloc (len * sizeof(wchar_t));
	if (!buf) {
		dmx_stop ();
		return;
	}

	*buf = '\n';
	p = buf + 1;
	if (cfg.timestamps) {
		p += mbstowcs (p, TSTAMP_PREF, strlen (TSTAMP_PREF));
		p += mbstowcs (p, get_timestamp (), TSTAMP_LEN);
		p += mbstowcs (p, TSTAMP_SUFF, strlen (TSTAMP_SUFF));
	}
	memset (p, L' ', align * sizeof(wchar_t));
	p += align;
	p += mbstowcs (p, pref, lpref);
	p += mbstowcs (p, nick, lnick);
	p += mbstowcs (p, suff, lsuff);

	memcpy (p, text, ltext * sizeof(wchar_t));

	add_text (buf, len);
	free (buf);
}

/* Add text to the text buffer of 'cur'. Print it if 'cur' is 'vcur'. */
void
add_text (const wchar_t *txt, int len)
{
	int new_head, old_head, new_tail, old_tail, space, fstdiv, y, x;

	/* add to the text buffer */
	old_tail = cur->tbuf_tail;
	old_head = cur->tbuf_head;
	while (1) {
		if (cur->tbuf_head >= cur->tbuf_tail)
			space = TBUF_SIZE - (cur->tbuf_head - cur->tbuf_tail);
		else
			space = cur->tbuf_tail - cur->tbuf_head;
		space--;
		if (space >= len)
			break;
		if (cur->tbuf_tail == cur->tbuf_head) {
			/* this should never happen */
			ui_output_err ("Line greater than buffer.");
			return;
		}
		remove_first_line ();
	}
	new_head = (cur->tbuf_head + len) % TBUF_SIZE;
	fstdiv = TBUF_SIZE - cur->tbuf_head;
	memcpy (&cur->tbuf[cur->tbuf_head], txt,
	        MIN (len, fstdiv) * sizeof(wchar_t));
	if (len > fstdiv)
		memcpy (cur->tbuf, txt + fstdiv,
		        (len - fstdiv) * sizeof(wchar_t));
	cur->tbuf_head = new_head;

	if (cur == vcur) {
		if (show_text_idx == old_head) {
			/* pull showing index down */
			show_text_idx = new_head;
			/* add to the window */
			getsyx (y, x);
			print_cur_text (old_head, new_head);
			wstandend (win_text);
			wnoutrefresh (win_text);
			setsyx (y, x);
			doupdate ();
		}
		else {
			/* the user is watching old text, don't pull showing
			   index down, so we don't disturb, but let them
			   know something is happening */
			beep ();
			/* if tail was moved, showing index could have to
			   be moved */
			new_tail = cur->tbuf_tail;
			if ((new_tail > old_tail &&
			     show_text_idx >= old_tail &&
			     show_text_idx < new_tail) ||
			    (new_tail < old_tail &&
			     (show_text_idx >= old_tail ||
			      show_text_idx < new_tail))) {
				/* move it down, so next PgUp/PgDn
				   doesn't go krazy */
				show_text_idx = new_tail;
			}
		}
	}
	else {
		/* 'cur' is not the active window, make a "(*)"
		   appear in the contact list */
		if (!cur->alarm) {
			cur->alarm = TRUE;
			ui_redraw_contacts ();
		}
	}
}

/* Output all text in text buffer of 'cur' to the text window, from 'from'
   until 'to' (not included). 'from' is normally the beginning of a new line,
   or the beginning of the first line to show on refresh, window change,
   terminal size change, or PgUp/PgDn. This function avoids word splitting
   at end of window, and aligns multi-line messages. */
int
print_cur_text (int from, int to)
{
	int     i;            /* current index in text buffer */
	int     col;          /* current column */
	int     fst;          /* index of first still-not-printed character */
	int     last_spc;     /* last space character found */
	int     last_spc_col; /* column where last space character was */
	int     j;
	attr_t  attrs;
	short   pair;
	wchar_t*txt;

	/* forget started sequences */
	reset_format_waddnstr ();

	col = 0;
	i = from;
	fst = i;
	last_spc = i;
	last_spc_col = 0;
	txt = cur->tbuf;

	/* if the window is too small, process all the text with only one
	   call to format_waddnstr(), this provides better visual results,
	   and avoids lots of problems */
	if (i_text.ncols < cur->align + 10)
		i = to;

	/* emulate the work of format_waddnstr() to discover where words
	   would be splitted, and avoid it */
	while (i != to) {
		if (txt[i] == SEQCH) {
			i = (i + 1) % TBUF_SIZE;
			if (i != to) {
				if (txt[i] == SEQCH || txt[i] == '/') {
					col++;
				}
				else if (txt[i] == '\n') {
					continue;
				}
				i = (i + 1) % TBUF_SIZE;
			}
		}
		else {
			if (txt[i] == '\n') {
				col = 0;
				/* without this, the last word in the
				   previous line could be carried down */
				last_spc = fst;
			}
			else {
				col++;
				if (txt[i] == ' ' && col > cur->align) {
					/* we always remember where the
					   last space is, so we can tell
					   where to split a line if a word
					   is going to get splitted */
					last_spc = i;
					/* remember the column where it is,
					   so we know the initial number of
					   columns of the second part of the
					   line we split */
					last_spc_col = col;
				}
			}
			i = (i + 1) % TBUF_SIZE;
		}
		if (col >= i_text.ncols) {
			/* so, we are at the end of the window, a line
			   split is required */
			if (last_spc != fst) {
				/* there is some space char in the line;
				   print until it, then continue as if we
				   had processed from the character next to
				   that space as an individual line */
				if (print_cyclic (fst, last_spc) == ERR)
					return ERR;
				fst = (last_spc + 1) % TBUF_SIZE;
				col -= last_spc_col;
			}
			else {
				/* warning warning!! no space char found;
				   this means that we have a very long
				   word; we don't print the overflowing
				   character now (that provokes problems
				   and does not look nice) */
				col = 0;
				if (i == 0)
					i = TBUF_SIZE;
				i--;
				if (print_cyclic (fst, i) == ERR)
					return ERR;
				fst = i;
			}
			/* split the line by inserting an '\n', and insert
			   the alignment spaces (without formatting, so
			   we avoid problems with underlining) */
			waddch (win_text, '\n');
			wattr_get (win_text, &attrs, &pair, NULL);
			wstandend (win_text);
			last_spc = fst;
			j = 0;
			while (j < cur->align && col < i_text.ncols) {
				waddch (win_text, ' ');
				col++;
				j++;
			}
			wattr_set (win_text, attrs, pair, NULL);
		}
	}
	/* output remaining text (we know there will be no splitting
	   problems because we didn't reach the end of the text window) */
	if (print_cyclic (fst, to) == ERR)
		return ERR;

	return OK;
}

/* Process format sequences in string. */
void
proc_format (WINDOW *win, const wchar_t *str, int n)
{
	wchar_t c;

	reset_format_waddnstr ();

	while (n > 0) {
		if (*str == SEQCH) {
			if (n > 1) {
				c = str[1];
				if ((c >= '0' && c <= '7') || c == 'b'
				    || c == 'u') {
					format_waddnstr (win, str, 2, FALSE);
				}
				str++;
				n--;
			}
			else {
				format_waddnstr (win, str, 1, FALSE);
			}
		}
		str++;
		n--;
	}
}

/* Wrapper over waddnstr() to follow attribute sequences.
   If 'ps' is TRUE, print sequences (used in edit window).
   Call reset_format_waddnstr() before a series of calls. */
int
format_waddnstr (WINDOW *win, const wchar_t *str, int n, BOOL ps)
{
	BOOL    fmt;    /* to know if some formatting is active accels loop */
	attr_t  attrs;
	short   pair, bkpair;
	int     i;
	wchar_t c;
	wchar_t cc[2];

	cc[1] = L'\0';

#if (DEBUG_PRINT_SEQS == TRUE)
	ps = TRUE;
#endif

	/* init 'fmt' */
	wattr_get (win, &attrs, &pair, NULL);
	fmt = (pair != 0 || attrs != WA_NORMAL);

	while (n > 0) {
		if (on_seq) {
			on_seq = FALSE;
			c = *str;
			if (c >= '0' && c <= '7') {
				/* change color */
				wattr_get (win, &attrs, &pair, NULL);
				pair = c - '0';
				if (pair == 0) {
					/* '0' resets all attributes,
					   this is useful and at least with
					   ncurses this is needed for 'fmt'
					   to work, because otherwise
					   'attrs' remembers the old color
					   (it seems that setting pair to 0
					   only unsets the color flag, but
					   the old color remains) */
					attrs = A_NORMAL;
				}
				wattr_set (win, attrs, pair, NULL);
				if (ps) {
					cc[0] = c;
					waddwstr (win, cc);
				}
				fmt = (pair != 0 || attrs != WA_NORMAL);
			}
			else if (c == SEQCH || c == '/') {
				/* escaped SEQCH or '/' */
				cc[0] = c;
				waddwstr (win, cc);
			}
			else if (c == 'b' || c == 'u') {
				/* bold or underline */
				wattr_get (win, &attrs, &pair, NULL);
				attrs ^= (c == 'b' ? WA_BOLD : WA_UNDERLINE);
				wattr_set (win, attrs, pair, NULL);
				if (ps) {
					cc[0] = c;
					waddwstr (win, cc);
				}
				fmt = (pair != 0 || attrs != WA_NORMAL);
			}
			else if (c == '\n') {
				/* process it with on_seq unset */
				continue;
			}
			else if (ps) {
				/* error in sequence */
				wattr_get (win, &attrs, &pair, NULL);
				wattr_set (win, 0, seq_err_pair, NULL);
				cc[0] = c;
				waddwstr (win, cc);
				wattr_set (win, attrs, pair, NULL);
			}
			str++;
			n--;
			continue;
		}
		i = 0;
		if (fmt) {
			/* search sequence or new line */
			while (i < n && str[i] != '\n' && str[i] != SEQCH)
				i++;
		}
		else {
			/* accelerated search when no format is active
			   (that's most of the time) */
			while (i < n && str[i] != SEQCH)
				i++;
		}

		if (waddnwstr (win, str, i) == ERR)
			return ERR;

		str += i;
		n -= i;

		if (n > 0) {
			if (*str == '\n') {
				wstandend (win);
				fmt = FALSE;
			}
			else { /* SEQCH */
				if (ps) {
					waddch (win, (chtype)SEQCH
					  | COLOR_PAIR (COLOR_WHITE) | WA_BOLD);
				}
				on_seq = TRUE;
				str++;
				n--;
			}
		}
	}

	return OK;
}

/* Recalculate cur->align. That is the column where text is aligned, and
   depends on the length of local and remote nicks. */
void
calc_align ()
{
	int my_len = strlen (cfg.nick);
	int his_len = cur->contact ? strlen (cur->contact->nick) : 0;
	int max = my_len > his_len ? my_len : his_len;
	int time = cfg.timestamps ? TSTAMP_LEN + TSTAMP_PREF_SUFF_LEN : 0;

	cur->align = time + max + NICK_PREF_SUFF_LEN + ALIGN_PAD;
}

/* Remove first line in text buffer of current contact window. */
static void
remove_first_line ()
{
	int prev;

	while (cur->tbuf_tail != cur->tbuf_head) {
		prev = cur->tbuf_tail;
		cur->tbuf_tail = (prev + 1) % TBUF_SIZE;
		if (cur->tbuf[prev] == '\n')
			break;
	}
}

/* Print text buffer from character with index 'first' until the character
   before the index 'last'. */
static int
print_cyclic (int first, int last)
{
	int ret;

	if (first <= last) {
		ret = format_waddnstr (win_text, &cur->tbuf[first],
		                       last - first, FALSE);
	}
	else {
		ret = format_waddnstr (win_text, &cur->tbuf[first],
		                       TBUF_SIZE - first, FALSE);
		if (ret != ERR)
			ret = format_waddnstr (win_text, cur->tbuf, last,
			                       FALSE);
	}

	return ret;
}

/* Forget started sequence in format_waddnstr(), if any. */
static inline void
reset_format_waddnstr ()
{
	on_seq = FALSE;
}

/* Format and return current time. */
static const char *
get_timestamp ()
{
	static char tstamp[TSTAMP_LEN + 1];
	time_t t = time (NULL);

#if HAVE_STRFTIME
	strftime (tstamp, TSTAMP_LEN + 1, "%H:%M", localtime (&t));
#else
	struct tm *ptm = localtime (&t);
	snprintf (tstamp, TSTAMP_LEN + 1, "%02d:%02d", ptm->tm_hour,
	          ptm->tm_min);
#endif

	return tstamp;
}
