/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef KARD_SEQUENCE_H_
#define KARD_SEQUENCE_H_

#include "cardsequence.h"

#include <qwidget.h>
#include <qpoint.h>
#include <vector>
using std::vector;

class QBoxLayout;
class Kard;
class Card;

/**
 * This is a graphical representation of a sequence of Kards. This is most often used to show a player his hand.
 * @todo Fix memory leak.
 *
 * @author John Schneiderman
 */
class KardSequence: public QWidget
{
private:
    Q_OBJECT
    Q_PROPERTY(bool m_faceUp READ isFaceUp WRITE setFaceUp)
    Q_PROPERTY(bool m_vertical READ isVertical WRITE setVertical)
    Q_PROPERTY(bool m_sortPopup READ isSortPopup WRITE setSortPopup)

public:
    /**
     * This is the default constructor.
     * @param parent is the parent widget.
     * @param name is the name of our widget.
     * @param vertical is true if you want the cards to be displayed vertically, false elsewise.
     */
    KardSequence(QWidget *parent=0, const char * name=0, bool vertical=false);
    /**
     * The default destructor.
     */
    ~KardSequence();
    /**
     * Is the preferred size for our widget.
     * @return the preferred size.
     */
    QSize sizeHint();
    /**
     * This is the accessor for the cardSequence we're displaying.
     * @return the card sequence.
     */
    const CardSequence& cardSequence() const;
    /**
     * Tells us if the cards are to be face up or not.
     * @return true if the cards are face up, false elsewise.
     */
    bool isFaceUp() const;
    /**
     * Tells us if the cards are displayed vertically or horizontally.
     * @return true if the cards are displayed vertically, false if they're horizontal.
     */
    bool isVertical() const;
    /**
     * An accessor for how many cards are in our card sequence.
     * @return the number of cards in our sequence.
     */
    int size() const;
    /**
     * This is a modifier to set if our cards should be face up or face down.
     * @param faceUp true will set the cards to be face up, false will set them to be face down.
     */
    void setFaceUp(bool faceUp);
    /**
     * This is a modifier to set the sequence of cards displayed.
     * @param sequence is the sequence of cards we wish to display.
     */
    void setCardSequence(const CardSequence &sequence);
    /**
     * This will change our sequence to be either vertical or horizontal.
     * @param vertical is true will make our sequence to be displayed vertically, false will make our sequence to be displayed horizontally.
     */
    void setVertical(bool vertical);
    /**
     * This will add a card to the end of our card sequence.
     * @param card is the card to add to our sequence.
     */
    void addCard(const Card &card);
    /**
     * This will remove a card from our card sequence.
     * @param card is the card to remove from our sequence.
     * @throw KardsGTError if card is not in the sequence.
     */
    void removeCard(const Card &card);
    /**
     * This will clear everything in our widget.
     */
    void clear();
    /**
     * This selects a specific kard in our sequence.
     * @param card is the kard we want to select.
     */
    void setSelected(const Card &card);
    /**
     * This tells if the sort popup menu is enabled.
     * @return true if the sort popup menu is enabled, false elsewise.
     */
    bool isSortPopup() const;
    /**
     * This enables or disables the sort popup menu.
     * @param sortPopup is true if we wish to enable the sort popup menu, false elsewise.
     */
    void setSortPopup(bool sortPopup);
    /**
     * This sets the image to be used on the back of the kards in our widget.
     * @param backImage is the file name for the image to use on the back of our kards.
     */
    void setBackImage(const QString &backImage);
    /**
     * This sets the images to be used on the front of the kards in our widget.
     * @param imagePath is the path to the directory that contains our card images.
     */
    void setCardImagePath(const QString &imagePath);

signals:
    /**
     * This is a signal that notifies a slot that a card has been selected.
     * @param kard is the kard that was selected
     */
    void kardSelected(Kard &kard);
    /**
     * This is a signal that notifies a slot that a card has been moved in our widget.
     */
    void kardMoved();

public slots:
    /**
     * This will add a Kard to the end of our card sequence.
     * @param kard is the Kard to add to our card sequence.
     */
    void addKard(Kard &kard);
    /**
     * This will remove a Kard from our card sequence.
     * @param kard is the kard to remove from our card sequence.
     */
    void removeKard(Kard &kard);

protected:
    /**
     * This paints our KardSequence.
     * @param event is the generating event.
     */
    virtual void paintEvent(QPaintEvent *event);
    /**
     * These are the types of objects we can accept for dropping.
     * @param event is the generating event.
     */
    virtual void dragEnterEvent(QDragEnterEvent *event);
    /**
     * This handles when an object is dropped into us.
     * @param event is the generating event.
     */
    virtual void dropEvent(QDropEvent *event);
    /**
     * This displays the right mouse button click menu for sorting by rank or by suit.
     * @param event is the generating event.
     */
    virtual void mousePressEvent(QMouseEvent *event);

private:
    /**
     * This is the preferred size of the KardSequence widget.
     * @param PREFERRED_WIDTH is the preferred width of the widget, which should be wide enough to hold 5 standard cards.
     * @param PREFERRED_HEIGHT is the preferred height of the widget.
     */
    enum preferredSize { PREFERRED_WIDTH=325, PREFERRED_HEIGHT=95 };
    /**
     * This is the preferred spacing of the KardSequence widget.
     * @param MARGIN is the margin spacing in the widget.
     * @param HORIZONTAL is the horizontal spacing between the widgets.
     */
    enum spacing { MARGIN=2, HORIZONTAL=5 };

    /// @param m_faceUp is true if we want our sequence to be face up, false elsewise.
    bool m_faceUp;
    /// @param m_vertical is true if we want our sequence to be displayed vertically, false will make them horizontal.
    bool m_vertical;
    /// @param m_sortPopup is true if we want to have a popup menu for sorting, false elsewise.
    bool m_sortPopup;
    /// @param m_cardSequence is the card sequence we're displaying.
    CardSequence m_cardSequence;
    /// @param m_pLayout is the layout to display the Kards in.
    QBoxLayout *m_pLayout;
    /// @param m_kards are the kards used in the layout.
    vector<Kard *> m_kards;
    /// @param m_cardImagePath is the path to the directory that contains our kard front images.
    QString m_cardImagePath;
    /// @param m_cardBackImage is the image filename to use for the back of our kards.
    QString m_cardBackImage;

    /**
     * This draws our KardSequence.
     * @param painter is the device to paint our card sequence to.
     */
    void draw(QPainter &painter);
    /**
     * This moves a card to be in front of a specific position.
     * @param card is the card we want to move.
     * @param position is the location we want our card to be in front of.
     */
    void moveCard(const Card &card, QPoint position);

private slots:
    /**
     * This sorts our sequence by rank.
     */
    void sortByRank();
    /**
     * This sorts our sequence by suit.
     */
    void sortBySuit();
    /**
     * This sorts our sequence by random.
     */
    void sortByRandom();
};
#endif
