/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "cribbageinterface.h"
#include "cribbage.h"
#include "kardselection.h"
#include "cardsequence.h"
#include "cribbageplayer.h"
#include "kardmessagedisplay.h"
#include "cribbageboard.h"
#include "card.h"
#include "kard.h"
#include "kardsgterror.h"
#include "kardplayer.h"
#include "kardpile.h"
#include "kardsequence.h"
#include "kardplaysequence.h"
#include "userprofiledatabase.h"

#include <qlabel.h>
#include <qlcdnumber.h>
#include <qpixmap.h>
#include <qstring.h>
#include <qtextedit.h>
#include <qmessagebox.h>

CribbageInterface::CribbageInterface(UserProfileDatabase &profileDatabase, QWidget* parent, const char* name): CribbageInterfaceBase(parent, name), Cribbage(this, profileDatabase)
{
    m_pUserProfile = &profileDatabase;
    //Set the colours for our game
    setBackgroundColor(darkGreen);
    crib->setBackgroundColor(darkGreen);
    lcdTxt->setBackgroundColor(darkGreen);
    countDisplay->setBackgroundColor(darkGreen);
    textLabelCrib->setBackgroundColor(darkGreen);

    // Start the timer check for the non-human players
    m_computerTimerId=startTimer(COMPUTER_PLAYER_TIME);
    m_clearingDelayId=-1; // Don't want to waste time clearing when we've just started.
    // Connect the player's cards to the selected slots
    connect(player1Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player1CardPlayed(Kard &)));
    connect(player2Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player2CardPlayed(Kard &)));
    // Connect the player's cards to the moving slots
    connect(player1Cards, SIGNAL(kardMoved()), this, SLOT(player1CardMoved()));
    connect(player2Cards, SIGNAL(kardMoved()), this, SLOT(player2CardMoved()));
}

CribbageInterface::~CribbageInterface()
{}

void CribbageInterface::updateTable()
{
    // Set the card back images
    player1Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[1].name()));
    player2Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[1].name()));
    crib->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[1].name()));
    cardDeck->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[1].name()));
    playerDiscards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[1].name()));

    // Set the card front path
    player1Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[1].name()));
    player2Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[1].name()));
    crib->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[1].name()));
    cardDeck->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[1].name()));
    playerDiscards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[1].name()));

    //Update the scores.
    cribbageBoard->pegScore(red, m_players[0].score());
    cribbageBoard->pegScore(blue, m_players[1].score());
    countDisplay->display(m_roundPoints);

    // Update player hands
    player1Cards->setCardSequence(m_players[0].hand());
    player2Cards->setCardSequence(m_players[1].hand());
    if (m_playSequence.isEmpty()) // The play sequence must have just been cleared, that means we need a delay in the clearing to allow the human player the chance to see what was played.
    {
        if (m_clearingDelayId == 0) // Make sure we only set off the timer once.
            m_clearingDelayId=startTimer(CLEARING_DELAY_TIME);
    }
    else
        playerDiscards->setCardSequence(m_playSequence, m_playOrder);

    // Update table info
    crib->setCardSequence(m_crib);
    cardDeck->setCardPile(m_deck);
    kardPlayer1->setPlayerImage(m_players[0].name().lower() + ".png");
    kardPlayer2->setPlayerImage(m_pUserProfile->userImage(m_players[1].name()));
    kardPlayer1->setDealer(m_players[0].isDealer());
    kardPlayer2->setDealer(m_players[1].isDealer());
    kardPlayer1->setLevel(m_players[0].level());
    kardPlayer2->setLevel(m_players[1].level());
    kardPlayer1->setTurn(m_players[0].isTurn());
    kardPlayer2->setTurn(m_players[1].isTurn());
    cribbageBoard->setName(red, m_players[0].name());
    cribbageBoard->setName(blue, m_players[1].name());

    update();
    updateGeometry();
}

void CribbageInterface::player1CardPlayed(Kard &kard)
{
    // Make sure it's their turn.
    if (! m_players[0].isTurn())
    {
        promptMessage("Warning", m_players[0].name() + ", it is not your turn.");
        kard.setSelected(false);
        return;
    }

    // Try to play the card.
    if (kard.isSelected())
        if (! m_players[0].hand().isEmpty() && (m_rules.isLegalPlay(m_playSequence, kard.card(), m_players[0])))
        {
            cardPlayed(kard.card());
            updateTable();
        }
}

void CribbageInterface::player2CardPlayed(Kard &kard)
{
    // Make sure it's their turn.
    if (! m_players[1].isTurn())
    {
        promptMessage("Warning", m_players[1].name() + ", it is not your turn.");
        kard.setSelected(false);
        return;
    }

    // Try to play the card.
    if (kard.isSelected())
        if (! m_players[1].hand().isEmpty() && (m_rules.isLegalPlay(m_playSequence, kard.card(), m_players[0])))
        {
            cardPlayed(kard.card());
            updateTable();
        }
}

void CribbageInterface::player1CardMoved()
{
    m_players[0].hand().clear();
    for (int i = 0, size = player1Cards->cardSequence().size(); i < size; ++i)
        m_players[0].hand().addCard(player1Cards->cardSequence()[i]);
}

void CribbageInterface::player2CardMoved()
{
    m_players[1].hand().clear();
    for (int i = 0, size = player2Cards->cardSequence().size(); i < size; ++i)
        m_players[1].hand().addCard(player2Cards->cardSequence()[i]);
}

CardSequence CribbageInterface::requestCards(const QString &prompt, const Player &player, int numberOfCards)
{
    KardSelection *selectionInterface=new KardSelection(prompt, numberOfCards, player.hand(), m_pUserProfile->pathToCardImages(m_players[1].name()), m_pUserProfile->pathToCardBackImage(m_players[1].name()), this, "cribSelection");

    if (selectionInterface->exec())
        return selectionInterface->selection();
    else
        return requestCards(prompt, player, numberOfCards);
}

void CribbageInterface::displayMessage(const Player &player, const QString &message)
{
    if (m_players[0].name() == player.name())
        kardPlayer1->setCaption(message);
    else if (m_players[1].name() == player.name())
        kardPlayer2->setCaption(message);
    updateTable();
}

void CribbageInterface::promptMessage(const QString &caption, const QString &message)
{
    QMessageBox::information(this, caption, message);
    updateTable();
}

pair<int, int> CribbageInterface::previousScores() const
{
    pair<int, int> scores;

    scores.first=cribbageBoard->scores(red).second;
    scores.second=cribbageBoard->scores(blue).second;
    return scores;
}

void CribbageInterface::setPreviousScores(int redScore, int blueScore)
{
    cribbageBoard->setScores(red, pair<int, int>(m_players[0].score(), redScore));
    cribbageBoard->setScores(blue, pair<int, int>(m_players[1].score(), blueScore));
    updateTable();
}

void CribbageInterface::showStarterCard(bool show)
{
    cardDeck->setTopCardUp(show);
    update();
    updateGeometry();
}

void CribbageInterface::computerPlay()
{
    CribbagePlayer *compPlayer=static_cast<CribbagePlayer *>(&m_players[0]);

    player1Cards->setSelected(compPlayer->playCard(m_playSequence));
}

void CribbageInterface::timerEvent(QTimerEvent *event)
{
    if (m_players.isEmpty()) //Prevent any player timers from going off when we have no players
    {
        QWidget::timerEvent(event);
        return;
    }
    if (m_players[0].isTurn() && (event->timerId() == m_computerTimerId))
        computerPlay();
    else if (event->timerId() == m_clearingDelayId)
    {
        killTimer(m_clearingDelayId); // Stop our clearing timer delay
        m_clearingDelayId = 0; // Reset our id
        updateTable();
    }
    else
        QWidget::timerEvent(event);
}

void CribbageInterface::displayRoundSummary(const QString &caption, const vector<CardSequence> &displayHands, const vector<QString> &messages)
{
    KardMessageDisplay summary(caption, displayHands, messages, m_pUserProfile->pathToCardImages(m_players[1].name()), m_pUserProfile->pathToCardBackImage(m_players[1].name()), this, "RoundSummary");

    summary.exec();
    updateTable();
}

void CribbageInterface::resetGame()
{
    cardDeck->clear();
    player1Cards->clear();
    player2Cards->clear();
    playerDiscards->clear();
    crib->clear();
    updateTable();
}

void CribbageInterface::updateStakes(int roundWinnerIndex)
{
    resetGame();
    cribbageBoard->reset();
    cribbageBoard->pegStakes(red, m_players[0].gamesWon());
    cribbageBoard->pegStakes(blue, m_players[1].gamesWon());
    // Give the player some skill points for winning the round
    if (roundWinnerIndex == 1)
        m_pProfileDatabase->setUserGamesWon(m_players[1].name(), "cribbage", true);
    else if (roundWinnerIndex == 0)
        m_pProfileDatabase->setUserGamesWon(m_players[1].name(), "cribbage", false);
    updateTable();
}

void CribbageInterface::gameWon()
{
    if (m_players[0].gamesWon() == m_rules.winningGameScore())
    {
        promptMessage("Game Over", QString(m_players[0].name() + " won.\n" + "Final Scores: " + m_players[0].name() + " %1, " + m_players[1].name() + " %2").arg(m_players[0].gamesWon()).arg(m_players[1].gamesWon()));
        m_pProfileDatabase->setUserGamesWon(m_players[1].name(), "cribbage", false);
    }
    else if (m_players[1].gamesWon() == m_rules.winningGameScore())
    {
        promptMessage("Game Over", QString(m_players[1].name() + " won.\n" + "Final Scores: " + m_players[0].name() + " %1, " + m_players[1].name() + " %2").arg(m_players[0].gamesWon()).arg(m_players[1].gamesWon()));
        m_pProfileDatabase->setUserGamesWon(m_players[1].name(), "cribbage", true);
    }
    else
        throw KardsGTError("CribbageInterface", "gameWon", "No one won?");
}
