/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "kardsgtinterface.h"
#include "gamefactory.h"
#include "playerfactory.h"
#include "userlogin.h"
#include "kardsgterror.h"
#include "playerselection.h"
#include "gamebase.h"
#include "configurekardsgt.h"

#include <qlabel.h>
#include <qimage.h>
#include <qwidgetstack.h>
#include <qaction.h>
#include <vector>
using std::pair;

KardsGTInterface::KardsGTInterface(QString path, QWidget* parent, const char* name, QString fileName):KardsGTInterfaceBase(parent, name), m_userProfileDB(path)
{
    title->setPixmap(QPixmap(QImage::fromMimeSource("kardsgt.png")));
    setIcon(QPixmap::fromMimeSource("kardsgticon.png"));
    if (fileName.isEmpty()) //Not loading a game
    {
        UserLogin *userLogin=new UserLogin(this, "KardsGT User Login");
        userLogin->loadUserProfiles(path);
        if (userLogin->exec())
            player->setName(userLogin->getUserName());
        else
        {
            delete userLogin;
            exit(0);
        }
        delete userLogin;
    }
    else
    {
        layout->raiseWidget(0);
        loadGame(fileName);
    }
}

KardsGTInterface::~KardsGTInterface()
{}

void KardsGTInterface::setupGame()
{
    // Find out which game we've selected
    if (gamePlayed.isEmpty())
        return;
    else
    {
        if (! gameFactory->createGame(m_userProfileDB, gamePlayed, *layout))
            throw KardsGTError("KardsGTInterface", "setupGame", "Failed to create game!");
        if (! loading)
        {
            PlayerSelection *playerSelection=new PlayerSelection(this, "KardsGT Player Selection");
            playerSelection->setGameName(gamePlayed);
            playerSelection->setPlayersNeeded(gameFactory->game().minimumPlayers(), gameFactory->game().maximumPlayers());
            if (playerSelection->exec())
            {
                opponents = playerSelection->selectedPlayers();
                createOpponents();
            }
            else
            {
                delete playerSelection;
                closeOpenGame();
                return;
            }
            delete playerSelection;
        }
        else
        {
            createOpponents(); // loadGame should have filled our opponents information.
        }
    }
    player->setLevel(playersLevel(gamePlayed));
    startGame();
    this->setBaseSize(layout->widget(1)->size());
    layout->raiseWidget(1);
}

Player::skill_Level KardsGTInterface::playersLevel(const QString &game) const
{
    Player::skill_Level level= Player::Skill_Error;
    pair<int, int> stat;
    int score=-1;

    stat=m_userProfileDB.userStatistics(player->name(), game);
    // Calculate winnings score
    if (stat.second != 0)
        score = static_cast<int>((stat.first / static_cast<double>(stat.second)) * 100);
    else
        score = 0;
    // Determine level of ability.
    if ((score >= 95) && (score <= 100))
        level = Player::Master;
    else if ((score >= 84) && (score <= 94))
        level = Player::Pwner;
    else if ((score >= 73) && (score <= 83))
        level = Player::Professional;
    else if ((score >= 57) && (score <= 72))
        level = Player::Apprentice;
    else if ((score >= 41) && (score <= 56))
        level = Player::Amateur;
    else if ((score >= 20) && (score <= 40))
        level = Player::Novice;
    else if ((score >= 0) && (score <= 20))
        level = Player::Newb;
    else
        level = Player::Skill_Error;
    return level;
}

void KardsGTInterface::createOpponents()
{
    for (int i=0; i < static_cast<int>(opponents.size()); ++i)
        if (! playerFactory->createPlayer(opponents[i], gamePlayed))
            throw KardsGTError("KardsGTInterface", "createOpponents", "Failed to create player!");
}

void KardsGTInterface::settingsConfigureKardsGT()
{
    ConfigureKardsGT conf(this);

    conf.setProfileDatabase(m_userProfileDB, player->name());
    if (conf.exec())
        m_userProfileDB.setUserSettings(player->name(), conf.settings());
}
