/***************************************************************************
*   Copyright (C) 2006 by John Schneiderman                               *
*   JohnMS@member.fsf.org                                                 *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
***************************************************************************/
#include "jackcribbageai.h"
#include "cardproperties.h"
#include "../../games/cribbage/cribbagerules.h"
#include "kardsgterror.h"

#include <vector>
using std::vector;

JackCribbageAI::JackCribbageAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand): basicStrategies(rules)
{
    m_playSequence = playSequence;
    m_hand = hand;
    m_pRules = &rules;
}

JackCribbageAI::~JackCribbageAI()
{}

Card JackCribbageAI::selectCard() const
{
    int score = 0;

    // See if we can make 15
    for (int index = 0, size = m_playSequence.size(); index < size; ++index)
        score += m_pRules->cardValue(m_playSequence[index]);
    for (int index = 0, size = m_hand.size(); index < size; ++index)
        if ((score + m_pRules->cardValue(m_hand[index])) == 15)
            return m_hand[index];

    // Play a random card
    return basicStrategies.randomCards(1, m_hand, m_playSequence).front();
}

CardSequence JackCribbageAI::myCrib() const
{
    vector<int> scores;
    vector<CardSequence> scoreCardsRemoved;
    vector<CardSequence> testHands;
    CardProperties testProp(m_hand);
    int highScoreIndex = 0;

    // Calcualte the different scores when we remove each two-card combinations.
    testHands = testProp.combinations();
    for (int index = 0, endCheck = testHands.size(); index < endCheck; ++index)
    {
        // See if the combinations is four cards
        if (testHands[index].size() != 4)
            continue;

        // Keep track of the cards we're testing
        scoreCardsRemoved.push_back(CardSequence());
        for (int handIndex = 0, size = m_hand.size(); handIndex < size; ++handIndex)
            if (! testHands[index].hasCard(m_hand[handIndex]))
                scoreCardsRemoved.back().addCard(m_hand[handIndex]);

        // Calculate and store the score for that combination
        scores.push_back(calculateScore(testHands[index]));
    }

    // Find the lowest score
    for (int index = 1, size = scores.size(); index < size; ++ index)
        if (scores[index] > scores[highScoreIndex])
            highScoreIndex = index;

    return scoreCardsRemoved[highScoreIndex];
}

CardSequence JackCribbageAI::opponentsCrib() const
{
    CardSequence testHand = m_hand;
    CardProperties handProp(m_hand);
    CardSequence removeCards;

    // Don't give sevens, eights
    removeCards += handProp.ranks(Card::SEVEN);
    removeCards += handProp.ranks(Card::EIGHT);
    testHand -= removeCards;

    // See if we already have a selection made
    if ((! testHand.isEmpty()) && (testHand.size() < 3))
    {
        if (testHand.size() == 2)
            return testHand;
        else
        {
            Card card;

            card = basicStrategies.randomCardsWithNoLegalChecks(1, testHand).front();
            while (card != testHand.front())
                card = basicStrategies.randomCardsWithNoLegalChecks(1, testHand).front();
            testHand.addCard(card);
            return testHand;
        }
    }
    if (testHand.isEmpty())
        return basicStrategies.randomCardsWithNoLegalChecks(2, m_hand);
    else
        return basicStrategies.randomCardsWithNoLegalChecks(2, testHand);
}

int JackCribbageAI::calculateScore(const CardSequence &hand) const
{
    CardProperties handProp(hand);
    const CribbageRules *rules = static_cast<const CribbageRules *>(m_pRules);
    vector<CardSequence> pairs, fifteens;
    CribbageRules::ScoreRuns runScore;
    int finalScore = 0;

    // Calculate 15's
    fifteens = handProp.summations(15, *m_pRules);
    for (int i=0; i < static_cast<int>(fifteens.size()); ++i)
        finalScore += 2;

    // Score our pairs
    pairs = rules->createPairTypes(handProp.pairs());
    for (int i=0; i < static_cast<int>(pairs.size()); ++i)
        if (rules->pairType(pairs[i]) != CribbageRules::PAIR_ERR)
            finalScore += rules->pairType(pairs[i]);

    // Score our runs
    runScore = rules->runType(rules->createRunTypes(handProp.runs()));
    if (runScore != CribbageRules::RUN_ERR)
        if (runScore == CribbageRules::RUN)
            finalScore += rules->createRunTypes(handProp.runs()).back().size();
        else
            finalScore += runScore;

    return finalScore;
}
