/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "kardplayer.h"
#include "kardsgterror.h"

#include <qimage.h>
#include <qevent.h>
#include <qpainter.h>
#include <qpoint.h>
#include <qfont.h>
#include <qbrush.h>

const QString KardPlayer::CAPTION_MARKER = "~";

KardPlayer::KardPlayer(QWidget *parent, const char *name): QWidget(parent, name, WPaintClever | WRepaintNoErase), m_player(), m_playerImage(), m_caption(), m_captionTimerIds()
{
    m_captionExtended=false;
}

QSize KardPlayer::sizeHint() const
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

const QString& KardPlayer::caption() const
{
    return m_caption;
}

void KardPlayer::setCaption(const QString &caption)
{
    if (m_caption.isEmpty())
        m_caption=caption;
    else
    {
        m_caption=caption + CAPTION_MARKER + m_caption;
        m_captionExtended = true;
    }
    m_captionTimerIds.push(startTimer(CAPTION_DISPLAY_TIME));
    update();
    updateGeometry();
}

void KardPlayer::setCaption(const QString &caption, int length)
{
    if (m_caption.isEmpty())
        m_caption=caption;
    else
    {
        m_caption=caption + CAPTION_MARKER + m_caption;
        m_captionExtended = true;
    }
    m_captionTimerIds.push(startTimer(length));
    update();
    updateGeometry();
}

void KardPlayer::clear()
{
    if (m_caption.isEmpty())
        return;
    if (m_captionExtended)
    {
        int lastCaptionStarts = m_caption.findRev(CAPTION_MARKER);

        m_caption.remove(lastCaptionStarts, m_caption.length() - lastCaptionStarts);
        // Does our caption still have multiple messages?
        if (m_caption.contains(CAPTION_MARKER) == 0)
            m_captionExtended = false;
    }
    else
        m_caption="";
    killTimer(m_captionTimerIds.top());
    m_captionTimerIds.pop();
    update();
    updateGeometry();
}

void KardPlayer::setDealer(bool dealer)
{
    m_player.setDealer(dealer);
    update();
    updateGeometry();
}

void KardPlayer::setLevel(Player::skill_Level level)
{
    m_player.setLevel(level);
    update();
    updateGeometry();
}

void KardPlayer::setTurn(bool turn)
{
    m_player.setTurn(turn);
    update();
    updateGeometry();
}

QPixmap KardPlayer::playerImage() const
{
    return m_playerImage;
}

void KardPlayer::setPlayerImage(const QPixmap &playerImage)
{
    if (playerImage.isNull())
        throw KardsGTError("KardPlayer", "setPlayerImage", "playerImage is null!");
    else
        m_playerImage=playerImage;
    update();
    updateGeometry();
}

void KardPlayer::setPlayerImage(const QString &imageFilename)
{
    m_playerImage=QImage::fromMimeSource(imageFilename);
    if (m_playerImage.isNull())
    {
        // Try loading it as a path
        m_playerImage=QImage(imageFilename);
        if (m_playerImage.isNull())
            throw KardsGTError(QString("KardPlayer"), QString("setPlayerImage"), QString("%1 was not found.").arg(imageFilename));
    }
    update();
    updateGeometry();
}

void KardPlayer::paintEvent(QPaintEvent *event)
{
    static QPixmap pixmap;
    QRect rect=event->rect();

    QSize newSize=rect.size().expandedTo(pixmap.size());
    pixmap.resize(newSize);
    pixmap.fill(this, rect.topLeft());

    QPainter painter(&pixmap, this);
    painter.translate(-rect.x(), -rect.y());
    painter.setWindow(0, 0, 75, 150);
    draw(painter);

    // Draw creation
    bitBlt(this, rect.x(), rect.y(), &pixmap, 0, 0, rect.width(), rect.height());
}

void KardPlayer::draw(QPainter &painter)
{
    const int FONT_SIZE = 10; // The font size we want to use.
    const int CHARS_PER_LINE = 7; // How many characters should appear on a single line.
    const int SCALE_FONT_WIDTH = 17; // The scale factor needed to make 1 char = 1 L.U.. Emperically determined.
    const int SCALE_FONT_HEIGHT = 10; // The scale factor needed to make 1 char = 1 L.U.. Emperically determined.
    const int MARGIN = 5; // The caption margin space

    painter.drawPixmap(painter.window(), m_playerImage);
    painter.setFont(QFont("courier", FONT_SIZE));
    if (! m_caption.isEmpty())
    {
        int textX=painter.window().width() / 4; // X-Position for the text
        int textY=painter.window().height() / 4; // Y-Position for the text
        int windowWidth=painter.window().width();
        int width; // Width of our caption bubble
        int height; // height of our caption bubble
        int lines = 1; // How many lines we break up the caption into
        bool textWillFit=true;
        QString caption = formatCaption(CHARS_PER_LINE);

        // Find the width of our caption bubble
        if ((static_cast<int>(m_caption.length()) * SCALE_FONT_WIDTH + textX) < windowWidth)
            width = static_cast<int>(m_caption.length() * SCALE_FONT_WIDTH) + 1;
        else
        {
            width = windowWidth - textX + 1;
            textWillFit = false;
        }
        // Find the height of our caption bubble
        if (textWillFit)
            height = SCALE_FONT_HEIGHT + MARGIN;
        else
        {
            lines = caption.length() / CHARS_PER_LINE;

            if ((caption.length() % CHARS_PER_LINE) != 0)
                ++lines;
            height = (SCALE_FONT_HEIGHT * lines) + MARGIN;
        }

        // Create the caption bubble
        painter.setBackgroundMode(Qt::TransparentMode);
        painter.setBrush(Qt::white);
        painter.drawRect(textX - 1, textY - SCALE_FONT_HEIGHT, width, height);

        // Draw the caption
        painter.setPen(Qt::black);
        if (textWillFit)
            painter.drawText(textX, textY, m_caption);
        else
        {
            int length = caption.length();

            for (int charsPos=0, line = 0; charsPos < length; charsPos+=CHARS_PER_LINE, ++line)
                painter.drawText(textX, textY + (line * SCALE_FONT_HEIGHT), caption.mid(charsPos, CHARS_PER_LINE));
        }
    }
    painter.setPen(Qt::red)
    ;
    painter.setBackgroundColor(Qt::white);
    painter.setBackgroundMode(Qt::OpaqueMode);
    // Show the player if it's their turn, they're the dealer, else show them the skill level.
    if (m_player.isTurn())
        painter.drawText(0, 10, "T"); // 0, 10 should be the upper-left corner with enough room for the letter.
    else if (m_player.isDealer())
        painter.drawText(0, 10, "D");
    else
        painter.drawText(0, 10, QString("%1").arg(m_player.level()));
}

void KardPlayer::timerEvent(QTimerEvent *event)
{
    if (event->timerId() == m_captionTimerIds.top())
        clear();
    else
        QWidget::timerEvent(event);
}

QString KardPlayer::formatCaption(int charsPerLine) const
{
    QString formattedCaption = "";
    int length = m_caption.length();
    bool fitInLine = false;

    for (int linePos=0; linePos < length; linePos+=charsPerLine)
    {
        // Check to see if it fits on the line
        if ((m_caption[linePos + charsPerLine] == ' ') || ((linePos + charsPerLine + 1) > length) || (m_caption[linePos + charsPerLine] == '~'))
            fitInLine = true;
        else
            fitInLine = false;

        // Add as many characters as we can on the line
        if (fitInLine)
        {
            // Skip ending marker or space marking for the begging of the line.
            if ((m_caption[linePos] == ' ') || (m_caption[linePos] == '~'))
                ++linePos;
            formattedCaption+=m_caption.mid(linePos, charsPerLine);
        }
        else
        {
            // Break up the word.
            formattedCaption+=m_caption.mid(linePos, charsPerLine - 1);
            if ((m_caption[linePos + charsPerLine + 1] != ' ') || (m_caption[linePos + charsPerLine + 1] != '~'))
                formattedCaption+="-";
            // Adjust the linePos since we're not adding a character all the way to the end.
            --linePos;
        }
    }
    return formattedCaption;
}
