/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef HEARTSINTERFACE_H
#define HEARTSINTERFACE_H

#include "heartsinterfacebase.h"
#include "hearts.h"

class CardSequence;
class Player;
class Kard;

/**
 * This is the interface for a game of hearts
 *
 * @author John Schneiderman
 */
class HeartsInterface: public HeartsInterfaceBase, public Hearts
{
    Q_OBJECT

public:
    /**
     * This is the general constructor.
     * @param profileDatabase is the user profile database.
     * @param parent is our parent widget.
     * @param name is the name of us.
     */
    HeartsInterface(UserProfileDatabase &profileDatabase, QWidget *parent = 0, const char *name = 0);
    /**
     * General destructor.
     */
    ~HeartsInterface();
    /**
     * Updates all the widgets.
     */
    void updateTable();
    /**
     * Asks a player to pass his cards a specific direction.
     * @param player is the player whose cards are to be passed.
     * @param direction is the direction the cards will be passed.
     * @return the cards the player has selected.
     */
    CardSequence passCards(const Player &player, const QString &direction);
    /**
     * Displays a message to a player.
     * @param player is the player to see the message.
     * @param message is the message for the player to see.
     * @throw KardsGTError if the player is not found.
     */
    void displayMessage(const Player &player, const QString &message);
    /**
     * Prompts a message.
     * @param caption is the caption for the message displayed.
     * @param message is the message for the player to see.
     */
    void promptMessage(const QString &caption, const QString &message);
    /**
     * This displays the round summary.
     * @param caption is the caption for the display window.
     * @param displayHands are the hands to display.
     * @param messages are the messages to accompany each hand.
     */
    void displayRoundSummary(const QString &caption, const vector<CardSequence> &displayHands, const vector<QString> &messages);
    /**
     * Displays the ending game scores, and who won.
     * @throw KardsGTError if we cannot determine who won.
     */
    void gameWon();

protected slots:
    /**
     * Handles the kard selected by player 1.
     * @param kard is the card the player selected.
     */
    void player1CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 2.
     * @param kard is the card the player selected.
     */
    void player2CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 3.
     * @param kard is the card the player selected.
     */
    void player3CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 4.
     * @param kard is the card the player selected.
     */
    void player4CardPlayed(Kard &kard);
    /**
     * Has each player announce his own score.
     */
    virtual void displayCurrentScores();
    /**
     * Displays the tricks taken by each player.
     */
    virtual void displayTricksTaken();
    /**
     * Saves the player 1's new card arrangement.
     */
    void player1CardMoved();
    /**
     * Saves the player 2's new card arrangement.
     */
    void player2CardMoved();
    /**
     * Saves the player 3's new card arrangement.
     */
    void player3CardMoved();
    /**
     * Saves the player 4's new card arrangement.
     */
    void player4CardMoved();

private:
    /**
     * These are the number of ms to launch each timer.
     * @param COMPUTER_PLAYER_TIME is how often to check to see if a computer player can play.
     * @param CLEARING_DELAY_TIME is how long our delay should be before we clear the playing sequence. @note This should be less than the time for a computer player to play.
     */
    enum TimerLengths { COMPUTER_PLAYER_TIME=1500, CLEARING_DELAY_TIME=500 };

    /// @param m_computerTimerId is the id for the timer for the computer player.
    int m_computerTimerId;
    /// @param m_clearingDelayId is the id for the timer for the clearing delay.
    int m_clearingDelayId;
    /// @param m_pUserProfile is the user profile database.
    const UserProfileDatabase *m_pUserProfile;

    /**
     * Hangles the actual card played.
     * @param playerIndex is the index for the player who played a card.
     * @param kard is the card the player has selected.
     */
    void kardPlayed(PlayerIndexes playerIndex, Kard &kard);
    /**
     * This handles the timer events.
     * @param event is the generating event.
     */
    void timerEvent(QTimerEvent *event);
    /**
     * This handles the computer card playing.
     * @param playerIndex is the index of the player who should play a card.
     * @throw KardsGTError if the playerIndex is the third player or a non-player.
     */
    void computerPlay(PlayerIndexes playerIndex);
};
#endif
