/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef SPADESINTERFACE_H
#define SPADESINTERFACE_H

#include "spades.h"
#include "spadesinterfacebase.h"

#include <qwidget.h>

class Kard;

/**
 * This is the interface for a game of spades.
 *
 * @author John Schneiderman
 */
class SpadesInterface: public SpadesInterfaceBase, public Spades
{
    Q_OBJECT

public:
    /**
     * This is the general constructor.
     * @param profileDatabase is the user profile database.
     * @param parent is our parent widget.
     * @param name is the name of us.
     */
    SpadesInterface(UserProfileDatabase &profileDatabase, QWidget *parent = 0, const char *name = 0);
    /**
     * General destructor.
     */
    ~SpadesInterface();
    /**
     * This updates all of the interface widgets.
     */
    void updateTable();
    /**
     * Requests if the human player wants to bid double.
     * @return true if the player wants to bid double, false elsewise.
     */
    bool bidDouble();
    /**
     * Requests that the human player select cards to exchange with his partner.
     * @param player is the player to exchange cards from.
     * @param partner is the player to exchange cards to.
     * @return the cards the player selects to exchange with his partner.
     */
    CardSequence selectExchangeCards(const Player &player, const Player &partner);
    /**
     * Displays a specific players hand.
     * @param show is true if want to show the player's hand, false elsewise.
     * @param index is the index of the player whose hand we want to show or hide.
     * @throw KardsGTError if we cannot find the player's index.
     */
    void showCards(bool show, Spades::PlayerIndexes index);
    /**
     * Requests that a human player make a bid.
     * @return is the bid the human player made.
     */
    int bid();
    /**
     * This displays the bid of a player.
     * @param index is the index of the player.
     * @param bid is the bid the player made.
     * @throw KardsGTError if we cannot find the player's index.
     */
    void displayBid(Spades::PlayerIndexes index, int bid);
    /**
     * Displays a message to a player.
     * @param player is the player to see the message.
     * @param message is the message for the player to see.
     * @throw KardsGTError if the player is not found.
     */
    void displayMessage(const Player &player, const QString &message);
    /**
     * Prompts a message.
     * @param caption is the caption for the message displayed.
     * @param message is the message for the player to see.
     */
    void promptMessage(const QString &caption, const QString &message);
    /**
     * Displays the ending game scores, and who won.
     * @throw KardsGTError if we cannot determine who won.
     */
    void gameWon();

protected slots:
    /**
     * Handles the kard selected by player 1.
     * @param kard is the card the player selected.
     */
    void player1CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 2.
     * @param kard is the card the player selected.
     */
    void player2CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 3.
     * @param kard is the card the player selected.
     */
    void player3CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 4.
     * @param kard is the card the player selected.
     */
    void player4CardPlayed(Kard &kard);
    /**
     * This displays the scores for each partnership.
     */
    void displayScores();
    /**
     * This displays each trick a player has taken.
     */
    void displayTricks();
    /**
     * This displays the contract each partnership made.
     */
    void displayContracts();
    /**
     * Saves the player 1's new card arrangement.
     */
    void player1CardMoved();
    /**
     * Saves the player 1's new card arrangement.
     */
    void player2CardMoved();
    /**
     * Saves the player 1's new card arrangement.
     */
    void player3CardMoved();
    /**
     * Saves the player 1's new card arrangement.
     */
    void player4CardMoved();

private:
    /**
     * These are the number of ms to launch each timer.
     * @param COMPUTER_PLAYER_TIME is how often to check to see if a computer player can play.
     * @param CLEARING_DELAY_TIME is how long our delay should be before we clear the playing sequence.
     * @note This should be less than the time for a computer player to play.
     */
    enum TimerLengths { COMPUTER_PLAYER_TIME=1500, CLEARING_DELAY_TIME=500 };

    /// @param m_computerTimerId is the id for the timer for the computer player.
    int m_computerTimerId;
    /// @param m_clearingDelayId is the id for the timer for the clearing delay.
    int m_clearingDelayId;
    /// @param m_pUserProfile is the user profile database.
    const UserProfileDatabase *m_pUserProfile;

    /**
     * Hangles the actual card played.
     * @param playerIndex is the index for the player who played a card.
     * @param kard is the card the player has selected.
     */
    void kardPlayed(PlayerIndexes playerIndex, Kard &kard);
    /**
     * This handles the timer events.
     * @param event is the generating event.
     */
    void timerEvent(QTimerEvent *event);
    /**
     * This handles the computer card playing.
     * @param playerIndex is the index of the player who should play a card.
     * @throw KardsGTError if the playerIndex is the third player or a non-player.
     */
    void computerPlay(PlayerIndexes playerIndex);
};
#endif
