/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef PLAYERFACTORY_H
#define PLAYERFACTORY_H

#include <vector>
using std::vector;

class QString;
class ComputerPlayer;
class AIBase;

/**
 * This gives us access to all of our players.
 *
 * @author John Schneiderman
 */
class PlayerFactory
{
public:
    /**
     * Default constructor.
     */
    PlayerFactory();
    /**
     * Default destructor.
     * @note we don't need to free our m_ais because the freeing of m_players in @ref GameBase takes care of it.
     */
    ~PlayerFactory();
    /**
     * An accessor to the current players.
     * @return the current player listings.
     */
    const vector<ComputerPlayer *>& players() const;
    /**
     * An accessor to the current players.
     * @return the current player listings.
     */
    vector<ComputerPlayer *>& players();
    /**
     * Creates a computer player.
     * @param playersName is the name of the player we want to create.
     * @note The name of the player must be an AI player that already exists!
     * @param game is the name of the game the player will be playing.
     * @return true if the computer player was created, false if not.
     */
    bool createPlayer(const QString &playersName, const QString &game);
    /**
     * Clears our listing of players.
     */
    void clear();
    /**
     * Lists the names of all the players known to the factory.
     * @note the names are grammatically correct.
     * @return an array filled with the names of the players.
     */
    vector<QString> playerNames() const;

protected:
    /**
     * This sets up the AI for the computer player.
     * @param playersName is the name of the player we want to create based on our AI listings.
     * @return true if the AI component was successfully created, false elsewise.
     */
    bool setupAIBase(const QString &playersName);
    /**
     * This sets up the computer player.
     * @param game is the name of the game we want the computer player to play.
     * @return true if the computer player was successfully create, false elsewise.
     */
    bool setupGamePlayer(const QString &game);

private:
    /// @param m_players is the listing of computer players we have created.
    vector<ComputerPlayer *> m_players;
    /// @param m_ais is the listing of ais we've created.
    vector<AIBase *> m_ais;
};
#endif
