/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "abigailspadesai.h"
#include "rulebase.h"
#include "cardproperties.h"
#include "../../games/spades/spadesrules.h"

AbigailSpadesAI::AbigailSpadesAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand): basicStrategies(rules)
{
    m_playSequence = playSequence;
    m_pRules = &rules;
    m_hand = hand;
}

AbigailSpadesAI::~AbigailSpadesAI()
{}

CardSequence AbigailSpadesAI::selectCards(const Card &partnersCard) const
{
    CardSequence selection, test;
    bool selectedCard = false;
    CardProperties handProperties(m_hand);
    CardProperties playSequenceProperties(m_playSequence);

    // If leading, then lead with the highest card in our hand
    if (m_playSequence.isEmpty())
    {
        // If spades has been broken, then lead with the highest card
        if (static_cast<const SpadesRules *>(m_pRules)->isSpadesBroken())
        {
            selection = basicStrategies.highestCard(m_hand);
            // Don't lead with a J if it's too early for an A to be likely played
            if ((selection.front().rank() >= Card::JACK) && (m_hand.size() == (m_pRules->numberOfCardsToDeal() - 5)))
                selection = basicStrategies.lowestCard(m_hand);
            selectedCard = true;
        }
        else // Lead with the highest non-trump card
        {
            selection = basicStrategies.highestNonTrumpCard(m_hand, Card::SPADES);
            // Don't lead with a K if it's too early for an A to be likely played
            if ((selection.front().rank() == Card::KING) && (m_hand.size() == (m_pRules->numberOfCardsToDeal() - 5)))
                selection = basicStrategies.lowestNonTrumpCard(m_hand, Card::SPADES);
            selectedCard = true;
        }
    }

    // Make sure not to play over partner if parnter leads
    if ((! selectedCard))
    {
        Card highestCard = basicStrategies.highestCardOfSuit(m_playSequence, m_playSequence.front().suit());

        // Play the lowest legal card
        if ((! partnersCard.isEmpty()) && (highestCard == partnersCard))
        {
            selectedCard = true;
            selection = basicStrategies.lowestCardOfSuit(m_hand, highestCard.suit());
            if (selection.isEmpty())
                selection = basicStrategies.lowestCard(m_hand);
        }
    }

    // If lead suit is trump
    if ((! selectedCard) && (m_playSequence.front().suit() == Card::SPADES))
    {
        test = basicStrategies.highestCardOfSuit(m_hand, m_playSequence.front().suit());
        Card highestCard = basicStrategies.highestCardOfSuit(m_playSequence, m_playSequence.front().suit());

        if ((! test.isEmpty()) && (((highestCard.rank() != Card::ACE) && (test.front() > highestCard))
                 || (test.front().rank() == Card::ACE))) // Make sure we'll beat the highest card
        {
            selection = test;
            selectedCard = true;
        }
        else // Play our lowest card
        {
            test = basicStrategies.lowestCardOfSuit(m_hand, m_playSequence.front().suit());
            if (! test.isEmpty())
            {
                selection = test;
                selectedCard = true;
            }
        }
    }

    // If trump hasn't been played, then play the highest card of the lead suit
    if ((! selectedCard) && playSequenceProperties.suits(Card::SPADES).isEmpty())
    {
        test = basicStrategies.highestCardOfSuit(m_hand, m_playSequence.front().suit());
        Card highestCard = basicStrategies.highestCardOfSuit(m_playSequence, m_playSequence.front().suit());
        if (test.isEmpty()) // If don't have a card of the lead suit, then play a trump
        {
            test = basicStrategies.highestCardOfSuit(m_hand, Card::SPADES);
            if (! test.isEmpty())
            {
                selection = test;
                selectedCard = true;
            }
        }
        else if (((highestCard.rank() != Card::ACE) && (test.front() > highestCard))
                 || (test.front().rank() == Card::ACE)) // Make sure we'll beat the highest card
        {
            selection = test;
            selectedCard = true;
        }
        else // Play our lowest card
        {
            selection = basicStrategies.lowestCardOfSuit(m_hand, m_playSequence.front().suit());
            selectedCard = true;
        }
    }
    else if (! selectedCard)// If trump has been played, then play a low card of the lead suit
    {
        test = basicStrategies.lowestCardOfSuit(m_hand, m_playSequence.front().suit());
        // If don't have a card of the lead suit, then play a higher trump
        if (test.isEmpty())
        {
            test = basicStrategies.highestCardOfSuit(m_hand, Card::SPADES);
            if (! test.isEmpty()) // Make sure we have a trump
            {
                // See if we have a higher trump
                if ((test.front() > basicStrategies.highestCardOfSuit(m_playSequence, Card::SPADES)) || (test.front().rank() == Card::ACE))
                {
                    selection = test;
                    selectedCard = true;
                }
                else // Play the lowest non-trump card
                {
                    test = basicStrategies.lowestNonTrumpCard(m_hand, Card::SPADES);
                    if (! test.isEmpty()) // Make sure we have a trump
                    {
                        selection = test;
                        selectedCard = true;
                    }
                }
            }
        }
        else
        {
            selection = test;
            selectedCard = true;
        }
    }

    // Play the lowest card from a suit
    if (! selectedCard)
        selection = basicStrategies.lowestCard(m_hand);
    return selection;
}

int AbigailSpadesAI::bidNil(int partnersBid) const
{
    int bid = SpadesRules::NON_BID;

    // Count how many Aces, Kings, and high spades (>= Q) we have
    for (int index = 0; index < m_hand.size(); ++index)
        if ((m_hand[index].rank() >= Card::JACK) || (m_hand[index].rank() == Card::ACE))
            if (bid == SpadesRules::NON_BID) // Reset to 1
                bid = 1;
            else
                ++bid;

    // See if we should bid nil
    if ((bid == SpadesRules::NON_BID) || (bid == 3))
    {
        // Count how many low cards (< 7) we have
        for (int index = 0; index < m_hand.size(); ++index)
            if ((m_hand[index].rank() < Card::SEVEN) && (m_hand[index].rank() != Card::ACE))
                if (bid == SpadesRules::NON_BID)
                    bid = 1;
                else
                    ++bid;
        // If we only have less than half all middle cards bid nil
        if (bid < 7)
            bid = SpadesRules::NIL;
        else
            bid = (bid % 3) + 1;
    }
    // Adjust our bid according to our partner's bid
    if ((bid != SpadesRules::NIL) && ((partnersBid != SpadesRules::NIL) || (partnersBid != SpadesRules::DOUBLE_NIL)))
        if ((bid + partnersBid) > 9)
        {
            bid = 9 - partnersBid;
            if (bid == 0) // Don't bid nil just because our partner made a high bid
                bid = 1;
        }
    return bid;
}

int AbigailSpadesAI::bidDoubleNil(int score, int opponetScore) const
{
    // Bid double nil we we trail by 125
    if ((opponetScore - score) > 125)
        return SpadesRules::DOUBLE_NIL;
    // Bid doulbe nil if we're negative by 75
    if (score <= -75)
        return SpadesRules::DOUBLE_NIL;
    return SpadesRules::NON_BID;
}

CardSequence AbigailSpadesAI::exchangeCards() const
{
    CardSequence exchange, cards;

    if (bidNil(0) == SpadesRules::NIL) // If we initiated the nil bid, get rid of our high cards.
    {
        for (int index = 0; index < m_hand.size(); ++index)
            if ((m_hand[index].rank() >= Card::TEN) || (m_hand[index].rank() == Card::ACE))
            {
                exchange.addCard(m_hand[index]);
                if (exchange.size() == 3)
                    break;
            }
    }
    else // If we didn't give our low cards.
    {
        for (int index = 0; index < m_hand.size(); ++index)
            if ((m_hand[index].rank() < Card::SEVEN) && (m_hand[index].rank() != Card::ACE))
            {
                exchange.addCard(m_hand[index]);
                if (exchange.size() == 3)
                    break;
            }
    }
    while (exchange.size() < 3)
    {
        cards=basicStrategies.randomCardsWithNoLegalChecks(1, m_hand);
        if (! exchange.hasCard(cards.front()))
            exchange.addCard(cards.front());
    }
    return exchange;
}

CardSequence AbigailSpadesAI::selectCardsForNilBid() const
{
    bool selectedCard = false;
    CardProperties handProperties(m_hand);
    CardProperties playSequenceProperties(m_playSequence);
    CardSequence selection;

    // If leading lead with lowest card
    if (m_playSequence.isEmpty())
    {
        // If spades has been broken, then lead with the lowest card
        if (static_cast<const SpadesRules *>(m_pRules)->isSpadesBroken())
        {
            selection = basicStrategies.lowestCard(m_hand);
            selectedCard = true;
        }
        else // Lead with the lowest non-trump card
        {
            selection = basicStrategies.lowestNonTrumpCard(m_hand, Card::SPADES);
            selectedCard = true;
        }
    }

    // If trump hasn't been played, then play the lowest card of the lead suit
    if ((! selectedCard) && playSequenceProperties.suits(Card::SPADES).isEmpty())
    {
        selection = basicStrategies.lowestCardOfSuit(m_hand, m_playSequence.front().suit());
        if (! selection.isEmpty())
            selectedCard = true;
    }
    else if (! selectedCard)// If trump has been played, then play a high card of the lead suit
    {
        selection = basicStrategies.highestCardOfSuit(m_hand, m_playSequence.front().suit());
        if (! selection.isEmpty())
            selectedCard = true;
    }

    // Play the highest card from a suit
    if (! selectedCard)
        selection = basicStrategies.highestCard(m_hand);
    return selection;
}
