/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef BASICGAMESTRATEGIES_H
#define BASICGAMESTRATEGIES_H

#include "card.h"
#include "cardsequence.h"

class RuleBase;

/**
 * Basic Game AI Decision Making
 *
 * @author John Schneiderman
 */
class BasicGameStrategies
{
public:
    /**
     * Constructor.
     * @param rules is the rules to use in our decision making.
     */
    BasicGameStrategies(const RuleBase &rules);
    /**
     * Destructor.
     */
    ~BasicGameStrategies();
    /**
     * This finds the highest card in a card sequence based on the cards rank value.
     * @param sequence is the sequence of cards to examine.
     * @return the highest card found, if no high card is found an empty card is given.
     */
    Card highestCard(const CardSequence &sequence) const;
    /**
     * This finds the highest card that is not a trump based on the cards rank value.
     * @param sequence is the sequence of cards to examine
     * @param trumpSuit is the suit that is trump in the game.
     * @return the highest card found, if no high card is found an empty card is given.
     */
    Card highestNonTrumpCard(const CardSequence &sequence, Card::Suit trumpSuit) const;
    /**
     * This finds the highest cards of a specific suit based on the cards rank value.
     * @param sequence is the sequence of cards to examine.
     * @param suit is the suit of the highest card we want.
     * @return the highest card found, if no high card is found an empty card is given.
     */
    Card highestCardOfSuit(const CardSequence &sequence, Card::Suit suit) const;
    /**
     * This finds the highest card that is less than a specific card based on rank value.
     * @param sequence is the sequence of cards to examine.
     * @param highCard is the highest card we wish to be below.
     * @return the highest card found, if no card lower than the high card is found the high card is given.
     */
    Card nextHighestCard(const CardSequence &sequence, const Card &highCard) const;
    /**
     * This finds the highest card of a specific suit that is less than a specific card based on rank value.
     * @param sequence is the sequence of cards to examine.
     * @param highCard is the highest card we wish to be below.
     * @param suit is the suit of the high card we want.
     * @return the highest card found, if no card lower than the high card is found the high card is given.
     * @note We assume that the high card and the suit both have the same suit.
     */
    Card nextHighestCardOfSuit(const CardSequence &sequence, const Card &highCard, Card::Suit suit) const;
    /**
     * This finds the lowest card in a card sequence that is based on rank value.
     * @param sequence is the sequence of cards to examine.
     * @return the lowest card found, if no card is found an empty card is given.
     */
    Card lowestCard(const CardSequence &sequence) const;
    /**
     * This finds the lowest card not of a trump suit based on rank value.
     * @param sequence is the sequence of cards to examine.
     * @param trumpSuit is the suit we don't want the lowest card to be from.
     * @return the lowest card found, if no card is found an empty card is given.
     */
    Card lowestNonTrumpCard(const CardSequence &sequence, Card::Suit trumpSuit) const;
    /**
     * This finds the lowest card of a specific suit based on rank value.
     * @param sequence is the sequence of cards to examine.
     * @param suit is the suit we want the lowest card from.
     * @return the lowest card found of the specified suit, if no card is found an empty card is given.
     */
    Card lowestCardOfSuit(const CardSequence &sequence, Card::Suit suit) const;
    /**
     * This selects a specific number of cards from our hand based on what is a legal play.
     * @param numberOfCards is the number of cards we have to select.
     * @param hand is the sequence of cards to choose from.
     * @param playSequence is the sequence of cards currently in play
     * @return the cards we have chosen.
     * @throw KardsGTError if we cannot select as many cards specified in numberOfCards.
     */
    CardSequence selectCards(int numberOfCards, const CardSequence &hand, const CardSequence &playSequence) const;
    /**
     * Selects cards randomly that are legal to play for the game.
     * @param numberOfCards is the number of cards to select.
     * @param hand is the sequence of cards to choose from.
     * @note If there is no legal cards in the hand an infinite loop is created!
     * @param playSequence is the sequence of cards currently in play
     * @return the cards we have chosen.
     */
    CardSequence randomCards(int numberOfCards, const CardSequence &hand, const CardSequence &playSequence) const;
    /**
     * Selects cards to play for the game.
     * @param numberOfCards is the number of cards to select.
     * @param hand is the sequence of cards to choose from.
     * @return the cards we have chosen.
     */
    CardSequence selectCardsWithNoLegalChecks(int numberOfCards, const CardSequence &hand) const;
    /**
     * Selects cards randomly to play for the game without regards for the rules of the game.
     * @param numberOfCards is the number of cards to select.
     * @param hand is the sequence of cards to choose from.
     * @return the cards we have chosen.
     */
    CardSequence randomCardsWithNoLegalChecks(int numberOfCards, const CardSequence &hand) const;
    /**
     * Makes a random bid in a specified range.
     * @param minimum is the lowest possible bid.
     * @param maximum is the highest possible bid.
     * @return the bid generated.
     */
    int randomBid(int minimum, int maximum) const;
    /**
     * Determines the number of cards in a sequence of cards that is higher than a specific card.
     * @param hand is the sequence of cards to examine.
     * @param rank is the rank that each card must be greater than.
     * @return the number of cards in hand that is greater than rank.
     */
    int numberOfCardsHigher(const CardSequence &hand, const Card::Rank &rank) const;
    /**
     * Determines the number of card in a sequence of cards that has a specific suit.
     * @param hand is the sequence of cards to examine.
     * @param suit is the suit that each card must be to be counted.
     * @return the number of cards in hand that is of the specified suit.
     */
    int numberOfCardsOfSuit(const CardSequence &hand, const Card::Suit &suit) const;

private:
    /// @param m_pRules are the rules for the current game.
    const RuleBase *m_pRules;
};
#endif
