/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "card.h"
#include "kardsgterror.h"

#include <QTextStream>
#include <QString>
#include <QtDebug>

#include <string>
using std::string;

Card::Card(): m_rank(RANK_ERR), m_suit(SUIT_ERR)
{}

Card::Card(Rank rank, Suit suit): m_rank(rank), m_suit(suit)
{
    if ((rank > RANK_ERR) || (rank < ACE))
        throw KardsGTError(QString("Card"), QString("Card"), QString("The value of rank (%1) is out of scope!").arg(rank));
    if ((suit > SUIT_ERR) || (suit < SPADES))
        throw KardsGTError(QString("Card"), QString("Card"), QString("The value of suit (%1) is out of scope!").arg(suit));
}

Card::Suit Card::suit() const
{
    return m_suit;
}

Card::Rank Card::rank() const
{
    return m_rank;
}

Card::CardColour Card::colour() const
{
    CardColour colour;

    switch (m_suit)
    {
    case SPADES:
    case CLUBS:
        colour = BLACK;
        break;
    case HEARTS:
    case DIAMONDS:
        colour = RED;
        break;
    default:
        colour = COLOUR_ERR;
        break;
    }
    return colour;
}

void Card::setCard(const QString &card)
{
    // Since we represent all cards as RankSuit they have to be 2 chars in length.
    if (card.length() != 2)
    {
        m_rank=Card::RANK_ERR;
        m_suit=Card::SUIT_ERR;
    }
    else
    {
        char rank=card[0].toAscii(), suit=card[1].toAscii();

        // Determine the rank
        m_rank=generateRank(rank);
        // Determine the suit
        m_suit=generateSuit(suit);
    }
}

void Card::setCard(Rank rank, Suit suit)
{
    if ((rank > RANK_ERR) || (rank < ACE))
        throw KardsGTError("Card", "Card", "rank out of scope!");
    if ((suit > SUIT_ERR) || (suit < SPADES))
        throw KardsGTError("Card", "Card", "suit out of scope!");
    m_rank = rank;
    m_suit = suit;
}

void Card::setRank(Rank rank)
{
    setCard(rank, m_suit);
}

void Card::setSuit(Suit suit)
{
    setCard(m_rank, suit);
}

Card& Card::operator=(const Card &card)
{
    if (this != &card)
    {
        m_suit=card.suit();
        m_rank=card.rank();
    }
    return *this;
}

int Card::operator+(const Card &card)
{
    return (m_rank + 1) + (card.rank() + 1);
}

int Card::operator-(const Card &card)
{
    return (m_rank + 1) - (card.rank() + 1);
}

bool Card::operator==(const Card &card) const
{
    if ((m_rank == card.rank()) && (m_suit == card.suit()))
        return true;
    else
        return false;
}

bool Card::operator!=(const Card &card) const
{
    if (*this == card)
        return false;
    else
        return true;
}

bool Card::operator<(const Card &card) const
{
    if (m_rank < card.rank())
        return true;
    else
        return false;
}

bool Card::operator>(const Card &card) const
{
    if (m_rank > card.rank())
        return true;
    else
        return false;
}

bool Card::lessThanSuit(const Card &card) const
{
    if (m_suit < card.suit())
        return true;
    else
        return false;
}

ostream& operator<<(ostream &out, const Card &card)
{
    out << card.generateHumanReadable().toStdString();
    return out;
}

istream& operator>>(istream &in, Card &card)
{
    string cardInput;

    in >> cardInput;
    card.setCard(QString(cardInput.c_str()));
    return in;
}

QTextStream& operator<<(QTextStream &out, const Card &card)
{
    out << card.generateHumanReadable();
    return out;
}

QDebug& operator<<(QDebug &out, const Card &card)
{
    out << card.generateHumanReadable();
    return out;
}

QTextStream& operator>>(QTextStream &in, Card &card)
{
    QString cardInput;

    in >> cardInput;
    card.setCard(cardInput);
    return in;
}

inline QString Card::generateHumanReadable() const
{
    QString line;

    //Determine the rank
    switch (m_rank)
    {
    case ACE:
        line="A";
        break;
    case TWO:
        line="2";
        break;
    case THREE:
        line="3";
        break;
    case FOUR:
        line="4";
        break;
    case FIVE:
        line="5";
        break;
    case SIX:
        line="6";
        break;
    case SEVEN:
        line="7";
        break;
    case EIGHT:
        line="8";
        break;
    case NINE:
        line="9";
        break;
    case TEN:
        line="T";
        break;
    case JACK:
        line="J";
        break;
    case QUEEN:
        line="Q";
        break;
    case KING:
        line="K";
        break;
    case JOKER:
        line="O";
        break;
    default:
        line="-"; // Means an error in the determination of the rank.
        break;
    }

    //Determine the suit
    switch (m_suit)
    {
    case SPADES:
        line+="S";
        break;
    case HEARTS:
        line+="H";
        break;
    case DIAMONDS:
        line+="D";
        break;
    case CLUBS:
        line+="C";
        break;
    default:
        line+="-"; // Means an error in the determination of the suit.
        break;
    }
    return line;
}

inline Card::Rank Card::generateRank(char rank) const
{
    switch (toupper(rank))
    {
    case 'A':
        return ACE;
    case '2':
        return TWO;
    case '3':
        return THREE;
    case '4':
        return FOUR;
    case '5':
        return FIVE;
    case '6':
        return SIX;
    case '7':
        return SEVEN;
    case '8':
        return EIGHT;
    case '9':
        return NINE;
    case 'T':
        return TEN;
    case 'J':
        return JACK;
    case 'Q':
        return QUEEN;
    case 'K':
        return KING;
    case 'O':
        return JOKER;
    }
    return RANK_ERR;
}

inline Card::Suit Card::generateSuit(char suit) const
{
    switch (toupper(suit))
    {
    case 'S':
        return SPADES;
    case 'H':
        return HEARTS;
    case 'D':
        return DIAMONDS;
    case 'C':
        return CLUBS;
    }
    return SUIT_ERR;
}

QString Card::toString() const
{
    return generateHumanReadable();
}

bool Card::isEmpty() const
{
    // A Joker card has a rank, but no suit.
    if ((m_rank == RANK_ERR) || ((m_suit == SUIT_ERR) && (m_rank != JOKER)))
        return true;
    else
        return false;
}
