/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef GAMEINFORMATION_H
#define GAMEINFORMATION_H

#include <iostream>
using std::ostream;
#include <map>
using std::map;

#include <QString>

class QTextStream;

/**
 * This holds game specific state information for a player.
 *
 * @author John Schneiderman
 */
class GameInformation
{
public:
    /**
     * General constructor.
     */
    GameInformation();
    /**
     * General destructor.
     */
    ~GameInformation();
    /**
     * Adds a key-value pair into the game state information. If the key is already present, then the value is updated.
     * @param key is the key for the game state information, generally a description of the value.
     * @param value is the information to store for the game state information.
     * @throw KardsGTError if key has a "_" or a "~" in it.
     */
    void add(const QString &key, int value);
    /**
     * Adds a key-value pair into the game state information. If the key is already present, then the value is updated.
     * @param key is the key for the game state information, generally a description of the value.
     * @param value is the information to store for the game state information.
     * @throw KardsGTError if key has a "_" or a "~" in it.
     */
    void add(const QString &key, double value);
    /**
     * Adds a key-value pair into the game state information. If the key is already present, then the value is updated.
     * @param key is the key for the game state information, generally a description of the value.
     * @param value is the information to store for the game state information.
     * @throw KardsGTError if key or value has a "_" or a "~" in them.
     */
    void add(const QString &key, const QString &value);
    /**
     * Retrieves the value for a specific key.
     * @param key is the key for the game state value we want to know.
     * @return the value found for the key, if the key is not found an empty string is returned.
     */
    QString value(const QString &key) const;
    /**
     * Clears the contents of the Game Information.
     */
    void clear();
    /**
     * Stream insertion operator.
     * @param out is the stream to get the game information from.
     * @param gameInformation is the game information object to put the information into.
     * @return out.
     */
    friend ostream& operator<<(ostream &out, const GameInformation &gameInformation);
    /**
     * Stream insertion operator.
     * @param out is the stream to get the game information from.
     * @param gameInformation is the game information object to put the information into.
     * @return out.
     */
    friend QTextStream& operator<<(QTextStream &out, const GameInformation &gameInformation);
    /**
     * Stream extraction operator.
     * @param in is the stream to put the game information into.
     * @param gameInformation is the game information object to get the information from.
     * @return in.
     */
    friend QTextStream& operator>>(QTextStream &in, GameInformation &gameInformation);

private:
    /// @param m_information stores all the game information in a key-value pair.
    map<QString, QString> m_information;
};
#endif
