/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef PLAYER_H
#define PLAYER_H

#include "cardsequence.h"
#include "gameinformation.h"

#include <QString>
#include <iostream>
using std::ostream;

class Card;
class QTextStream;

/**
 * This class is meant to represent a local human being playing a card game.
 *
 * @author John Schneiderman
 */
class Player
{
public:
    /**
     * This is the range for the various skill levels.
     * @param Skill_Error is when there is a problem with the skill level.
     * @param Master is when the player wins more than 95% of the time.
     * @param Pwner is when the player wins from 84% to 94% of the time.
     * @param Professional is when the player wins from 73% to 83% of the time.
     * @param Apprentice is when the player wins from 57% to 72% of the time.
     * @param Amateur is when the player wins from 41% to 56% of the time.
     * @param Novice is when the player wins from 20% to 40% of the time.
     * @param Newb is when the player wins less than 20% of the time.
     */
    enum SkillLevel
    {
        Skill_Error=0,
        Master,
        Pwner,
        Professional,
        Apprentice,
        Amateur,
        Novice,
        Newb
    };
    /**
     * These are the lower percentate levels for each of the skill levels.
     * @param MasterLowScore is the master skill low score.
     * @param PwnerLowScore is the pwner skill low score.
     * @param ProfessionalLowScore is the professional skill low score.
     * @param ApprenticeLowScore is the apprentice skill low score.
     * @param AmateurLowScore is the amateur skill low score.
     * @param NoviceLowScore is the novice skill low score.
     * @param NewbLowScore is the newb skill low score.
     */
    enum SkillLevelPercentages
    {
        MasterLowScore = 95,
        PwnerLowScore = 84,
        ProfessionalLowScore = 73,
        ApprenticeLowScore = 57,
        AmateurLowScore = 41,
        NoviceLowScore = 20,
        NewbLowScore = 0
    };

    /**
     * This is a general constructor.
     */
    Player();
    /**
     * Destructor.
     */
    virtual ~Player();
    /**
     * This gives the name/nickname of the player.
     * @return The name/nickname of the player.
     */
    QString name() const;
    /**
     * This sets the name/nickname of the player.
     * @param name is the name/nickname for the player.
     */
    void setName(const QString &name);
    /**
     * This gives the current hand a player is holding.
     * @return the hand a player is holding.
     */
    const CardSequence& hand() const;
    /**
     * This gives the current hand a player is holding.
     * @return a reference to the hand a player is holding.
     */
    CardSequence& hand();
    /**
     * This returns the current score a player has earned during the game.
     * @return The current value of the players score.
     */
    int score() const;
    /**
     * This adds points to a players score.
     * @param point is the number of points to increase the score by.
     */
    void addPoint(int point);
    /**
     * This removes points from players score.
     * @param point is the number of points to decrease the score by.
     */
    void removePoint(int point);
    /**
     * This determines if a player is the dealer or not.
     * @return true if the player is the dealer, false elsewise.
     */
    bool isDealer() const;
    /**
     * This sets a player to be the dealer or not.
     * @param dealer true if you want the player to be a dealer, false if not.
     */
    void setDealer(bool dealer);
    /**
     * Increases the number of games won by one.
     */
    void incrementGamesWon();
    /**
     * Decreases the number of games won by one.
     */
    void decrementGamesWon();
    /**
     * This gives the number of games a player has won.
     * @return the number of games won.
     */
    int gamesWon() const;
    /**
     * This determines if it is a players turn.
     * @return true if it is the players turn, false elsewise.
     */
    bool isTurn() const;
    /**
     * This sets a player's turn.
     * @param turn should be set to true if it is time for a players turn, false if it is not.
     */
    void setTurn(bool turn);
    /**
     * This is the accessor to a player's skill level.
     * @return the current skill level the player has for the current game.
     */
    SkillLevel skillLevel() const;
    /**
     * This sets the skill level for the player.
     * @param level is the level to set for the player for the current game.
     */
    void setSkillLevel(SkillLevel level);
    /**
     * Equivalent operator
     * @param player is the what you wish to compare *this to.
     * @return true if their names and hands are the same, false elsewise.
     */
    bool operator==(const Player &player) const;
    /**
     * This resets a player's hand, turn setting, and dealer setting to their starting values.
     */
    void reset();
    /**
     * This resets a player's round score to their starting values.
     */
    void resetRoundScore();
    /**
     * This clears everthing a players has.
     */
    void clear();
    /**
     * An accessor to the game information that is known to the player.
     * @return the game information known to the player.
     */
    GameInformation gameInformation() const;
    /**
     * A modifier to the game information that is known to the player.
     * @return the game information known to the player.
     */
    GameInformation& gameInformation();
    /**
     * This is the stream insertion for a player.
     * @param out is the stream to insert the player data into.
     * @param player is the player to insert into the stream.
     * @note If a player's name contains spaces, the spaces are replaced with an underscore.
     * @return out.
     */
    friend QTextStream& operator<<(QTextStream &out, const Player &player);
    /**
     * This is the stream extraction for a player.
     * @param in is the stream to extract the player data from.
     * @param player is the player to fill in with the data extracted.
     * @return in.
     */
    friend QTextStream& operator>>(QTextStream &in, Player &player);
    /**
     * This is the stream insertion for a player.
     * @param out is the stream to insert the player data into.
     * @param player is the player to insert into the stream.
     * @note If a player's name contains spaces, the spaces are replaced with an underscore.
     * @return out.
     */
    friend ostream& operator<<(ostream &out, const Player &player);

protected:
    /// @param m_hand is the players hand.
    CardSequence m_hand;
    /// @param m_name is the players name/nickname.
    QString m_name;
    /// @param m_score is the players score during a round/phase.
    int m_score;
    /// @param m_gamesWon is the number of games a player has won.
    int m_gamesWon;
    /// @param m_dealer is true if he is a dealer, false elsewise.
    bool m_dealer;
    /// @param m_turn is true if it is his turn, false elsewise.
    bool m_turn;
    /// @param m_level is the level that the player plays the current game at.
    SkillLevel m_level;
    /// @param m_gameInformation is used to hold game specific information, generally used by the computer based player AI's.
    GameInformation m_gameInformation;
};
#endif
