/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef RULESBASE_H
#define RULESBASE_H

#include "player.h"

class Card;
class CardSequence;
class PlayerList;

/**
 * This is an abstract interface for the rules of a card game.
 * This class is used in partnership with the @ref GameBase class.
 * This class should contain the specific rules of the game as well any specific values used in the game.
 * If a method is not needed for a specific game, the method should throw a KardsGTError to prevent it's usage.
 *
 * @author John Schneiderman
 */
class RuleBase
{
public:
    /**
     * This is the default constructor.
     */
    RuleBase();
    /**
     * Default destructor.
     */
    virtual ~RuleBase();
    /**
     * When this method is implemented it will give the maximum number of players that can play the game.
     * @return the maximum players for the card game.
     */
    virtual int maximumNumberOfPlayers() const=0;
    /**
     * When this method is implemented it will give the minimum number of players that can play the game.
     * @return the minimum players needed for the card game.
     */
    virtual int minimumNumberOfPlayers() const=0;
    /**
     * When this method is implemented it will give the score a player must meet in order to win the game.
     * @param numberOfPlayers is the number of players in the game.
     * @return the winning score.
     */
    virtual int winningGameScore(int numberOfPlayers = -1) const=0;
    /**
     * When this method is implemented it will give the number of cards to deal to each player in a round.
     * @param numberOfPlayers is the number of players in the game.
     * @return the number of cards for each player.
     */
    virtual int numberOfCardsToDeal(int numberOfPlayers = -1) const=0;
    /**
     * When this method is implemented it will give the point value of a card for that game.
     * @param card is the card whose value we wish to determine.
     * @return the worth of the card.
     */
    virtual int cardValue(const Card &card) const=0;
    /**
     * Gives the rank order value for a card.
     * @param card is the card we wish to examine.
     * @return the rank order value of the card.
     */
    virtual int rankValue(const Card &card) const=0;
    /**
     * When this method is implemented it will determine if a card being played is going to be a legal play.
     * @param sequence is the sequence of cards played thus far in the round.
     * @param cardPlayed is the card wishing to be played.
     * @param player is the player who wants to play the card.
     * @return true if it is a legal play, false elsewise.
     */
    virtual bool isLegalPlay(const CardSequence &sequence, const Card &cardPlayed, const Player &player=Player()) const=0;
    /**
     * When this method is implemented it will determine if conditions have been met for the end of the game.
     * @param players are the players in the game.
     * @return true if the game over conditions have been met, false if they have not.
     */
    virtual bool isGameOver(const PlayerList &players) const=0;
    /**
     * When this method is implemented it will determine if conditions have been met for the end of a round.
     * @param players are the players in the game.
     * @return true if the round over conditions have been met, false if they have not.
     */
    virtual bool isRoundOver(const PlayerList &players) const=0;
    /**
     * When this method is implemented it will determine if conditions have been met for the end of a phase.
     * @note A phase is defined as period in a game that has different check conditions or reset meanings than that of a round.
     * @param players are the players in the game.
     * @param playSequence is the sequence of cards that have been played.
     * @return true if the phase over conditions have been met, false if they have not.
     */
    virtual bool isPhaseOver(const PlayerList &players, const CardSequence &playSequence) const=0;
};
#endif
