/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef USERPROFILE_H
#define USERPROFILE_H

#include <vector>
using std::vector;
using std::pair;

#include <QSettings>

class QString;

/**
 * This is the database of user profiles.
 *
 * When a database is loaded it contains the information on what profiles we are tracking,
 * and how well they're playing each game. The database is stored in the system specific location.
 *
 * @todo Create an enumerated list for the range of emotions for the players.
 *
 * @author John Schneiderman
 */
class UserProfileDatabase
{
public:
    /**
     * The default constructor.
     */
    UserProfileDatabase();
    /**
     * The default destructor.
     */
    ~UserProfileDatabase();
    /**
     * An accessor to a users game statistics.
     * @param game is the name of the game we want the statistics for.
     * @return A pair whose first value is the number of games won, and whose second value is the number of games played.
     */
    pair<int, int> gameStatistics(const QString &game) const;
    /**
     * Updates the game statistics for a user, if the game statistics file doesn't exist one is created.
     * @note If the game statistic file is not found, one will be created.
     * @param game is the name of the game we want to update the statistics for.
     * @param won is true if the user has won the game, false if he has lost.
     */
    void setGameStatistics(const QString &game, bool won);
    /**
     * This retrieves the image to use for the user.
     * @note The information is found in the user settings file.
     * @param mood is the mood of the image we're getting.
     * @return the stored player's image file-name, if not found player-male.png is returned.
     */
    QString userMoodImage(const QString &mood = "NeutralMood") const;
    /**
     * This retrieves the path to the directory that has the card images.
     * @note The information is pulled from the user settings file.
     * @return the stored path to the directory, if not found default is returned.
     */
    QString pathToCardFrontImages() const;
    /**
     * This retrieves the image to use for the back of the cards.
     * @note The information is pulled from the user settings file.
     * @return the stored file-name for the image to use on the back of the card, if not found "back00.png" is returned.
     */
    QString cardBackImageFilename() const;
    /**
     * This stores the image file-name to use for a specific mood.
     * @param mood is the mood depicted in the image.
     * @param filename is the file-name of the image.
     */
    void setUserMoodImage(const QString &mood, const QString &filename);
    /**
     * This stores the path to the directory for the card front images.
     * @param path is the path to the directory that contains all the front images for the cards.
     */
    void setPathToCardFrontImages(const QString &path);
    /**
     * This stores the file-name for the card back image.
     * @param image is the file-name.
     */
    void setCardBackImageFilename(const QString &image);
    /**
     * An accessor to the user's name.
     * @return the user's name.
     */
    QString userName() const;
    /**
     * This stores the user's name.
     * @param userName is the user's name to store.
     */
    void setUserName(const QString &userName);

private:
    /// @param m_settings are the settings for KardsGT.
    QSettings m_settings;
};
#endif
