/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef KARDPLAYER_H
#define KARDPLAYER_H

#include "player.h"

#include <QWidget>
#include <QPixmap>
#include <QString>
#include <QPaintEvent>
#include <QTimerEvent>
#include <stack>
using std::stack;

class QPainter;

/**
 * This is the graphical representation of a player.
 *
 * @author John Schneiderman
 */
class KardPlayer: public QWidget
{
    Q_OBJECT
    Q_PROPERTY(QPixmap m_playerImage READ playerImage WRITE setPlayerImage)
    Q_PROPERTY(QString m_caption READ caption WRITE setCaption)

public:
    /**
     * KardPlayer constructor
     * @param parent is the parent window.
     */
    KardPlayer(QWidget *parent = 0);
    /**
     * This gives the preferred size of the player's image.
     * @return the size of the player's image.
     */
    QSize sizeHint() const;
    /**
     * This is the accessor for the current player's image.
     */
    QPixmap playerImage() const;
    /**
     * This sets the image of the player.
     * @param playerImage is what to set the player's image to.
     * @throw KardsGTError if the playerImage is null.
     */
    void setPlayerImage(const QPixmap &playerImage);
    /**
     * This sets the image of the player.
     * @param imageFilename is the name of the file for the player.
     * @note The image file-name can be either a built in image or a file-name to an image.
     * @throw KardsGTError if the image file cannot be found.
     */
    void setPlayerImage(const QString &imageFilename);
    /**
     * This is the accessor to what the player "said".
     */
    const QString& caption() const;
    /**
     * This causes the player to "say" something.
     * @param caption is what the player says.
     * @note if we're still showing a previous caption, we'll combine the messages.
     */
    void setCaption(const QString &caption);
    /**
     * This causes the player to "say" something.
     * @param caption is what the player says.
     * @param length is how many seconds to display the message
     * @note if we're still showing a previous caption, we'll combine the messages.
     */
    void setCaption(const QString &caption, int length);
    /**
     * This clears what the player has said.
     */
    void clear();
    /**
     * Updates the icon notification for the player dealer status.
     * @param dealer is true if we are the dealer, false elsewise.
     */
    void setDealer(bool dealer);
    /**
     * Updates the icon notification for the player level status.
     * @param level is the level the player has attained.
     * @throw KardsGTError If level is an invalid value.
     */
    void setSkillLevel(Player::SkillLevel level);
    /**
     * Updates the icon notification for the player turn status.
     * @param turn is true if it is the player's turn, false elsewise.
     */
    void setTurn(bool turn);

protected:
    /**
     * This draws the players image.
     * @param event is the generating event.
     */
    virtual void paintEvent(QPaintEvent *event);
    /**
     * This will erase the caption.
     * @param event is the timer event to clear our caption.
     */
    virtual void timerEvent(QTimerEvent *event);

private:
    /**
     * These are the constants that are used in this class.
     * @param PREFERRED_WIDTH is the preferred width of a player's image.
     * @param PREFERRED_HEIGHT is the preferred height of a player's image.
     * @param CAPTION_DISPLAY_TIME is how many ms to display the caption.
     */
    enum PrivateConstants
    {
        PREFERRED_WIDTH=90,
        PREFERRED_HEIGHT=140,
        CAPTION_DISPLAY_TIME=5000
    };

    /// @param m_player is the player we represent.
    Player m_player;
    /// @param m_captionExtended is true if the caption has more than one message, false elsewise.
    bool m_captionExtended;
    /// @param m_playerImage is the players image.
    QPixmap m_playerImage;
    /// @param m_caption is the caption for the player.
    QString m_caption;
    /// @param m_captionTimerIds are the identifications for the caption timers.
    stack<int> m_captionTimerIds;
    /// @param CAPTION_MARKER is the marker that separates multiple captions.
    static const QString CAPTION_MARKER;

    /**
     * This draws our player and anything he says.
     * @param painter is what we want to paint on.
     */
    void draw(QPainter &painter);
    /**
     * This formats the caption to fit nicely in the caption bubble.
     * @param charsPerLine is the number of characters available per line.
     * @return the formatted caption.
     */
    QString formatCaption(int charsPerLine) const;
};
#endif
