/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "kardplaysequence.h"
#include "kardsgterror.h"
#include "kard.h"

#include <QLayout>
#include <QPainter>
#include <QPixmap>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QPaintEvent>

KardPlaySequence::KardPlaySequence(QWidget *parent): QWidget(parent), m_kardSequences()
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_numberOfPlayers = 0;
    m_faceUp = false;
    m_cardImagePath = "";
    m_cardBackImage = "";
    createLayout();
    setAcceptDrops(true);
}

KardPlaySequence::KardPlaySequence(int numberOfPlayers, QWidget *parent): QWidget(parent), m_kardSequences()
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_numberOfPlayers = numberOfPlayers;
    m_faceUp = false;
    m_cardImagePath = "";
    m_cardBackImage = "";
    createLayout();
    setAcceptDrops(true);
}

KardPlaySequence::~KardPlaySequence()
{
    clearLayout();
}

QSize KardPlaySequence::sizeHint()
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

int KardPlaySequence::numberOfPlayers() const
{
    return m_numberOfPlayers;
}

bool KardPlaySequence::isFaceUp() const
{
    return m_faceUp;
}

void KardPlaySequence::setNumberOfPlayers(int numberOfPlayers)
{
    clearLayout();
    m_numberOfPlayers = numberOfPlayers;
    createLayout();
    update();
    updateGeometry();
}

void KardPlaySequence::setFaceUp(bool faceUp)
{
    int size = m_kardSequences.size();

    m_faceUp = faceUp;
    for (int i = 0; i < size; ++i)
        m_kardSequences[i]->setFaceUp(m_faceUp);
    update();
    updateGeometry();
}

const CardSequence& KardPlaySequence::cardSequence(int playerIndex) const
{
    if ((playerIndex >= m_numberOfPlayers) || (playerIndex < 0))
        throw KardsGTError("KardPlaySequence", "cardSequence", "The playerIndex value is out of range.");

    return m_kardSequences[playerIndex]->cardSequence();
}

void KardPlaySequence::setCardSequence(const CardSequence &sequence, int starterPlayerIndex)
{
    if ((starterPlayerIndex >= m_numberOfPlayers) || (starterPlayerIndex < 0))
        throw KardsGTError(QString("KardPlaySequence"), QString("setCardSequence"), QString("The starterPlayerIndex value is out of range."));
    int size = sequence.size();
    vector<CardSequence> playSequences;

    // Create a play sequence for each player
    for (int i=0; i < m_numberOfPlayers; ++i)
        playSequences.push_back(CardSequence());
    // Add each card that the player played to their play sequence.
    for (int cardIndex=0, playerIndex = starterPlayerIndex; cardIndex < size; playerIndex = (playerIndex + 1) % m_numberOfPlayers, ++cardIndex)
        playSequences[playerIndex].addCard(sequence[cardIndex]);
    // Set their new play sequence.
    for (int index = 0; index < m_numberOfPlayers; ++index)
        m_kardSequences[index]->setCardSequence(playSequences[index]);
}

void KardPlaySequence::setCardSequence(const CardSequence &sequence, vector<int> playerCardIndexes)
{
    if (sequence.size() != static_cast<int>(playerCardIndexes.size()))
        throw KardsGTError(QString("KardPlaySequence"), QString("setCardSequence"), QString("The sequence and playerCardIndexes are not the same size!"));
    int size = sequence.size();
    vector<CardSequence> playSequences;

    // Create a play sequence for each player
    for (int i=0; i < m_numberOfPlayers; ++i)
        playSequences.push_back(CardSequence());
    // Add each card that the player played to their play sequence.
    for (int cardIndex=0; cardIndex < size; ++cardIndex)
        playSequences[playerCardIndexes[cardIndex]].addCard(sequence[cardIndex]);
    // Set their new play sequence.
    for (int index = 0; index < m_numberOfPlayers; ++index)
        m_kardSequences[index]->setCardSequence(playSequences[index]);
}

void KardPlaySequence::clear()
{
    int size = m_kardSequences.size();

    for (int i = 0; i < size; ++i)
        m_kardSequences[i]->clear();
    update();
    updateGeometry();
}

void KardPlaySequence::paintEvent(QPaintEvent *)
{
    QPainter painter(this);

    draw(painter);
}

void KardPlaySequence::createLayout()
{
    QHBoxLayout *player1Layout;
    QHBoxLayout *player2Layout;
    QHBoxLayout *player2And4Layout;
    QHBoxLayout *player2And3Layout;
    QHBoxLayout *player3Layout;

    m_pLayout=new QVBoxLayout(this);
    // Assemble the kard sequences
    for (int i=0; i < m_numberOfPlayers; ++i)
    {
        m_kardSequences.push_back(new KardSequence(this));
        m_kardSequences[i]->setMinimumSize(Kard::PREFERRED_WIDTH, Kard::PREFERRED_HEIGHT / 4);
        m_kardSequences[i]->setMaximumHeight(Kard::PREFERRED_HEIGHT);
        m_kardSequences[i]->setSortPopup(false);
        m_kardSequences[i]->setFaceUp(m_faceUp);
        m_kardSequences[i]->setCardBackImage(m_cardBackImage);
        m_kardSequences[i]->setCardFrontImagePath(m_cardImagePath);
    }

    // Create the final layout
    switch (m_numberOfPlayers)
    {
    case 0:
        break;
    case 1:
        // Create Layouts
        player1Layout=new QHBoxLayout();
        // Add Widgets to each layout
        player1Layout->addWidget(m_kardSequences[0]);
        // Add the layouts to the main layout
        m_pLayout->addLayout(player1Layout);
        break;
    case 2:
        // Create Layouts
        player1Layout=new QHBoxLayout();
        player2Layout=new QHBoxLayout();
        // Add Widgets to each layout
        player1Layout->addWidget(m_kardSequences[0]);
        player2Layout->addWidget(m_kardSequences[1]);
        // Add the layouts to the main layout
        m_pLayout->addLayout(player1Layout);
        m_pLayout->addLayout(player2Layout);
        break;
    case 3:
        // Create Layouts
        player1Layout=new QHBoxLayout();
        player2Layout=new QHBoxLayout();
        player3Layout=new QHBoxLayout();
        player2And3Layout=new QHBoxLayout();
        // Add Widgets to each layout
        player1Layout->addWidget(m_kardSequences[0]);
        player2And3Layout->addWidget(m_kardSequences[2]);
        player2And3Layout->addWidget(m_kardSequences[1]);
        // Add the layouts to the main layout
        m_pLayout->addLayout(player1Layout);
        m_pLayout->addLayout(player2And3Layout);
        break;
    case 4:
        // Create Layouts
        player1Layout=new QHBoxLayout();
        player2Layout=new QHBoxLayout();
        player3Layout=new QHBoxLayout();
        player2And4Layout=new QHBoxLayout();
        // Add Widgets to each layout
        player1Layout->addWidget(m_kardSequences[0]);
        player2And4Layout->addWidget(m_kardSequences[3]);
        player2And4Layout->addWidget(m_kardSequences[1]);
        player3Layout->addWidget(m_kardSequences[2]);
        // Add the layouts to the main layout
        m_pLayout->addLayout(player1Layout);
        m_pLayout->addLayout(player2And4Layout);
        m_pLayout->addLayout(player3Layout);
        break;
    default:
        throw KardsGTError("KardPlaySequence", "createLayout", "Layout is not specified for that many players!");
    }
}

void KardPlaySequence::clearLayout()
{
    int size = m_kardSequences.size();

    for (int i = 0; i < size; ++i)
        delete m_kardSequences[i];
    m_kardSequences.clear();
    delete m_pLayout;
}

void KardPlaySequence::draw(QPainter &painter)
{
    int size = m_kardSequences.size();

    painter.drawRect(painter.window());
    for (int i=0; i < size; ++i)
        m_kardSequences[i]->show();
}

void KardPlaySequence::setCardBackImage(const QString &backImage)
{
    m_cardBackImage = backImage;
    for (int i=0; i < m_numberOfPlayers; ++i)
        m_kardSequences[i]->setCardBackImage(m_cardBackImage);
}

void KardPlaySequence::setCardFrontImagePath(const QString &imagePath)
{
    m_cardImagePath = imagePath;
    for (int i=0; i < m_numberOfPlayers; ++i)
        m_kardSequences[i]->setCardFrontImagePath(m_cardImagePath);
}
