/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef PLAY_SEQUENCE_H
#define PLAY_SEQUENCE_H

#include "kardsequence.h"

#include <QWidget>
#include <QPaintEvent>
#include <vector>
using std::vector;

class QBoxLayout;

/**
 * This creates a nice play sequence of cards, laying them out based on the player
 * layout and the sequence of cards played.
 * 
 * @note Currently we only have the interface defined for up to four players.
 *
 * @todo Make the layout creation general for any number of players.
 *
 * @author John Schneiderman
 */
class KardPlaySequence: public QWidget
{
    Q_OBJECT
    Q_PROPERTY(int m_numberOfPlayers READ numberOfPlayers WRITE setNumberOfPlayers)
    Q_PROPERTY(bool m_faceUp READ isFaceUp WRITE setFaceUp)

public:
    /**
     * The creates an empty play sequence layout.
     * @param parent is the parent of our widget.
     */
    KardPlaySequence(QWidget *parent=0);
    /**
     * This creates a layout for a specific number of players.
     * @param numberOfPlayers is the number of players needed for this layout.
     * @param parent is the parent of our widget.
     * @throw KardsGTError if the number of players is greater than our currently defined maximum.
     */
    KardPlaySequence(int numberOfPlayers, QWidget *parent=0);
    /**
     * Destructor for our widget.
     */
    ~KardPlaySequence();
    /**
     * The preferred size of our widget.
     */
    QSize sizeHint();
    /**
     * An accessor to how many players our layout is setup for.
     * @return the number of players in our layout.
     */
    int numberOfPlayers() const;
    /**
     * An accessor if we're display all the kards in the layout face up.
     * @return true if all the kards are face up, false elsewise.
     */
    bool isFaceUp() const;
    /**
     * A modifier to set the number of players in our layout.
     * @param numberOfPlayers is the number of players for our layout.
     * @throw KardsGTError if the number of players is greater than our currently defined maximum.
     */
    void setNumberOfPlayers(int numberOfPlayers);
    /**
     * A modifier to set all of the cards in the play sequence face up.
     * @param faceUp true if all the cards should be face up, false elsewise.
     */
    void setFaceUp(bool faceUp);
    /**
     * Gets the card sequence for a specific player.
     * @param playerIndex is the index of the player we want.
     * @return the card sequence for the player.
     * @throw KardsGTError if the playerIndex is out of range.
     */
    const CardSequence& cardSequence(int playerIndex) const;
    /**
     * This sets all of the card sequences. It alternates kards starting with the index of the player for the first kard in the sequence.
     * @param sequence is the sequence of cards.
     * @param starterPlayerIndex is the player index who starts the sequence of cards.
     * @throw KardsGTError if the startPlayerIndex is out of range.
     * @note The first player index is 0.
     */
    void setCardSequence(const CardSequence &sequence, int starterPlayerIndex);
    /**
     * This sets all of the card sequences. It places each of the kards according to the list of player indexes.
     * @param sequence is the sequence of cards.
     * @param playerCardIndexes is a list of each kard for the corresponding player.
     * @throw KardsGTError if the sequence list and the playerCardIndexes are not the same size.
     */
    void setCardSequence(const CardSequence &sequence, vector<int> playerCardIndexes);
    /**
     * This clears all of the play sequences.
     */
    void clear();
    /**
     * This sets the image used for the back of the kards.
     * @param backImage is the file-name of the image for the back of the kard.
     */
    void setCardBackImage(const QString &backImage);
    /**
     * This sets the directory that has the kard front images.
     * @param imagePath is the path to the directory for our kard images.
     */
    void setCardFrontImagePath(const QString &imagePath);

protected:
    /**
     * This paints our widget.
     * @param event is the generating event.
     */
    virtual void paintEvent(QPaintEvent *event);

private:
    /**
     * This is the preferred size of our widget.
     * @param PREFERRED_WIDTH is the preferred width of our widget.
     * @param PREFERRED_HEIGHT is the preferred height of our widget.
     */
    enum PreferredSize
    {
        PREFERRED_WIDTH=400,
        PREFERRED_HEIGHT=200
    };

    /// @param m_kardSequences are the sequences of kards we're displaying.
    vector<KardSequence *> m_kardSequences;
    /// @param m_numberOfPlayers are the number of player's we're displaying.
    int m_numberOfPlayers;
    /// @param m_faceUp is true if we want the player's cards to be face up, false elsewise.
    bool m_faceUp;
    /// @param m_pLayout is the layout for our widget.
    QBoxLayout *m_pLayout;
    /// @param m_cardImagePath is the path to the directory that contains all of our images for the front of our kards in the kard sequences.
    QString m_cardImagePath;
    /// @param m_cam_cardBackImagerdBackImage is the file-name for the back of the kards in our kard sequences.
    QString m_cardBackImage;

    /**
     * This creates a layout appropriate for the number of players in our widget.
     */
    void createLayout();
    /**
     * This clears all of the widgets in our widget/
     */
    void clearLayout();
    /**
     * This draws the widget.
     * @param painter is the device to paint our widget on.
     */
    void draw(QPainter &painter);
};
#endif
