/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2007-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef EUCHREPLAYER_H
#define EUCHREPLAYER_H

#include "computerplayer.h"

class AIBase;
class EuchreRules;

/**
 * This is a computer player for the game of euchre.
 *
 * @author John Schneiderman
 */
class EuchrePlayer: public ComputerPlayer
{
public:
    /**
     * The general constructor.
     * @param aiBase is the AI for our computer player.
     */
    EuchrePlayer(AIBase &aiBase);
    /**
     * The default destructor.
     */
    ~EuchrePlayer();
    /**
     * Requests that the AI choose a card for game play.
     * @param playSequence is the current play sequence.
     * @return the card the AI has chosen.
     */
    virtual Card playCard(const CardSequence& playSequence) const;
    /**
     * Requests that the AI choose the turn-up card.
     * @param suit is the suit of the turn-up card.
     * @return true if the AI wants the suit, false elsewise.
     */
    bool makeTurnUp(Card::Suit suit) const;
    /**
     * Requests that the AI select a trump suit.
     * @param suit is the suit of the turn-up card.
     * @return the suit the AI has chosen, SUIT_ERR if it passes.
     */
    Card::Suit selectTrump(Card::Suit suit) const;
    /**
     * Requests that the AI choose if it wishes to play alone.
     * @return true if the AI does, false elsewise.
     */
    bool playAlone() const;
    /**
     * Requests that the AI choose if it wishes to defend alone.
     * @return true if the AI does, false elsewise.
     */
    bool defendAlone() const;
    /**
     * This sets the rules for the euchre player.
     * @param rules is the rules for the euchre player.
     * @note The rules must be the same object as used in the game.
     */
    void setRules(const EuchreRules &rules);
    /**
     * Requests that the AI choose if it wishes to have the turn-up card.
     * @param turnUp is the turn-up card.
     * @return true if the AI does, false elsewise.
     */
    bool dealerWantsTurnUp(const Card &turnUp) const;
    /**
     * Requests that the AI discard one of the card from it's hand.
     * @return the card selected to be discarded.
     */
    Card discard() const;

private:
    /// @param m_pRules are the rules for the game of euchre.
    const EuchreRules *m_pRules;
};
#endif
