/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef SPADES_H
#define SPADES_H

#include "gamebase.h"
#include "spadesrules.h"
#include "carddeck.h"

#include <vector>
using std::vector;
using std::pair;

class SpadesInterface;

/**
 * This controls and keeps track of the state of the game spades.
 *
 * @author John Schneiderman
 */
class Spades: public GameBase
{
public:
    /**
     * The constructor for the spades.
     * @param gameInterface is the interface for the game.
     * @param profileDatabase is the user profile database.
     */
    Spades(SpadesInterface *gameInterface, UserProfileDatabase &profileDatabase);
    /**
     * The destructor.
     */
    ~Spades();
    /**
     * This starts the game of spades.
     * @throw KardsGTError if the wrong number of players were created for the game.
     */
    virtual void startPlay();
    /**
     * This saves the game of spades.
     * @param filename is the name of the file to save the game to.
     * @return true if we successfully save the game, false elsewise.
     */
    virtual bool save(const QString &filename);
    /**
     * This loads the game of spades.
     * @param filename is the name of the file to load.
     * @return true if we load the game successfully, false elsewise.
     */
    virtual bool load(const QString &filename);
    /**
     * This gives the minimum number of players needed to play the game of spades.
     * @return the minimum number of players.
     */
    virtual int minimumPlayers() const;
    /**
     * This gives the maximum number of players allowed to play the game of spades.
     * @return the maximum number of players.
     */
    virtual int maximumPlayers() const;
    /**
     * Adds the opponent to play against the player, and sets their rule pointer.
     * @param opponent is the player to play against our local player.
     */
    virtual void addOpponent(ComputerPlayer &opponent);

protected:
    /**
     * These are the indexes for each player.
     * @param PLAYER_ONE_INDEX is the index for player one.
     * @param PLAYER_TWO_INDEX is the index for player two.
     * @param PLAYER_THREE_INDEX is the index for player three.
     * @param PLAYER_FOUR_INDEX is the index for player four.
     * @param NON_PLAYER is the index for an invalid player.
     */
    enum PlayerIndexes
    {
        PLAYER_ONE_INDEX=0,
        PLAYER_TWO_INDEX=1,
        PLAYER_THREE_INDEX=2,
        PLAYER_FOUR_INDEX=3,
        NON_PLAYER=-1
    };
    /**
     * These are the indexes for each partnership.
     * @param TEAM_ONE is the first partnership.
     * @param TEAM_TWO is the second partnership.
     * @param NON_TEAM is the index for an invalid team.
     */
    enum TeamIndexes
    {
        TEAM_ONE = 0,
        TEAM_TWO = 1,
        NON_TEAM = -1
    };

    /// @param m_rules is the rules to the game of spades.
    SpadesRules m_rules;
    /// @param m_deck is the deck of cards used by the players.
    CardDeck m_deck;
    /// @param m_playerCardOrderIndexes is the order of the cards that each player played.
    vector<PlayerIndexes> m_playerCardOrderIndexes;
    /// @param m_tricksWon
    vector<int> m_tricksWon;

    /**
     * Handles a card being played.
     * @param card is the card the player wants to play.
     * @throw KardsGTError if the card cannot be found in the current player's hand.
     */
    virtual bool cardPlayed(const Card &card);
    /**
     * Deals out the cards to each player.
     */
    virtual void deal();
    /**
     * Disables the current player, and enables the next player's turn.
     */
    virtual void setupNextPlayer();
    /**
     * Determines and handles the various stages in game play.
     * @return the last major GameStatusCode value after handling the changes.
     */
    virtual int handleGameStatus();
    /**
     * Calculates the contract for a partnership.
     * @param teamIndex is the index of each of the players in the partnership.
     * @return the partnership's total contract.
     */
    int calculateContract(pair<int, int> teamIndex) const;
    /**
     * Determines which partnership won.
     * @return the team index for the partnership that won the game.
     */
    TeamIndexes teamWhoWon() const;

private:
    /**
     * These are the game ending states for Spades.
     * @param NO_CHANGE indicates there has been no change in the current state.
     * @param PHASE_ENDED indicates that the phase has ended.
     * @param ROUND_ENDED indicates that the round has ended.
     * @param GAME_OVER indicates that the game has ended.
     */
    enum GameStatusCodes
    {
        NO_CHANGE = 0,
        PHASE_ENDED = 1,
        ROUND_ENDED = 2,
        GAME_OVER = 3
    };

    /// @param m_pInterface is the interface to our graphical interface.
    SpadesInterface *m_pInterface;
    /// @param m_playerBids is the individual bids from each player.
    vector<int> m_playerBids;
    /// @param playerPlayedHighestCard is the index to the player who played the highest card.
    PlayerIndexes m_playerPlayedHighestCard;
    /// @param m_partnerIndexes is the indexes of each partner.
    PlayerIndexes m_partnerIndexes[4];
    /// @param m_contractMade holds if each partnership made their contract.
    pair<bool, bool> m_contractMade;

    /**
     * Handles the contract bidding
     */
    void handleBid();
    /**
     * Handles the exchange of cards from a player and his partner.
     * @param index is the index of the player who wishes to exchange cards.
     * @param sequence is the sequence of cards the player at playersIndex has chosen to exchange.
     * @note The partnership is determined by the value in m_partnerIndexes.
     */
    void handleExchangePartnerCards(PlayerIndexes index, const CardSequence &sequence);
    /**
     * This adds the current player to the card playing order.
     */
    void setupCardOrder();
    /**
     * Handles the end of the current phase.
     * @note Doesn't reset the current play sequence.
     */
    void handelEndOfPhase();
    /**
     * Searches through the playing sequence and finds the highest card or highest trump that was played.
     * @return the index of the player who played the highest card or trump.
     */
    PlayerIndexes playerPlayedHighestCard();
    /**
     * Handles the end of the round.
     */
    void handleEndOfRound();
    /**
     * Handles the scoring for each partnership.
     */
    void handleScoring();
    /**
     * Calculates the total score for the partnership.
     * @param teamIndex is the indexes of each player in the partnership.
     * @return the score they earned that round.
     */
    int calculateScore(pair<int, int> teamIndex) const;
    /**
     * Calculates the tricks won by the team.
     * @param teamIndex is the indexes of each player on the team.
     * @return the sum of the tricks won if neither player bids nil or double nil. The tricks won for the partner player if the other player bid nil or double nil. SpadesRules::DOUBLE_NIL if both players bid double nil.
     */
    int calculateTricksWon(pair<int, int> teamIndex) const;
    /**
     * Determines if both players in a partnership bid nil.
     * @param teamIndex is the indexes of each player on the team.
     * @return true if both players bid either nil or double nil, false elsewise.
     */
    bool bothPlayersBidNil(pair<int, int> teamIndex) const;
    /**
     * Handles the creation of the round summary.
     */
    void handleRoundSummary();
    /**
     * This handles all the tasks for the end of the game.
     */
    void handleGameOver();
    /**
     * Resets the game back to it's beginning round state.
     */
    void reset();
};
#endif
