/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "kardsgtinterface.h"
#include "gamefactory.h"
#include "playerfactory.h"
#include "kardsgterror.h"
#include "playerselection.h"
#include "gamebase.h"
#include "configurekardsgt.h"

#include <QLabel>
#include <QImage>
#include <QStackedWidget>
#include <QAction>
#include <QPixmap>
#include <QAssistantClient>
#include <QTextCodec>
#include <QInputDialog>
#include <QFileDialog>
#include <QMessageBox>

#include <vector>
using std::pair;
#include <string>
using std::string;
#include <iostream>
using std::cerr;

const QString KardsGTInterface::m_fileFilter = tr("KardsGT game files (*.kgf)\n" "All files (*)");
/// @param VERSION is the current version of our programme. @note Defined in main.cpp
extern const string VERSION;
/// @param COPYRIGHT is the current copyright years. @note Defined in main.cpp
extern const string COPYRIGHT_YEARS;

KardsGTInterface::KardsGTInterface(QString fileName, QWidget* parent): QMainWindow(parent), m_userProfileDB()
{
    Ui_KardsGTInterface::setupUi(this);
    m_pAssistant=new QAssistantClient("", this);
    m_pPlayer=new Player();
    m_pPlayerFactory=new PlayerFactory();
    m_pGameFactory=new GameFactory();
    m_loading=false;
    if (fileName.isEmpty()) //Not loading a game
    {
        QString userName = m_userProfileDB.userName();
        while (userName == "UNKNOWN")
        {
            changeName();
            userName = m_userProfileDB.userName();
        };
        m_pPlayer->setName(userName);
    }
    else
    {
        Ui_KardsGTInterface::layout->setCurrentIndex(0);
        loadGame(fileName);
    }

    // signals and slots connections
    connect(gameExitAction, SIGNAL(activated()), this, SLOT(close()));
    connect(gameExitAction, SIGNAL(activated()), this, SLOT(closeOpenGame()));
    connect(helpAboutAction, SIGNAL(activated()), this, SLOT(helpAbout()));
    connect(gameCloseAction, SIGNAL(activated()), this, SLOT(closeOpenGame()));
    connect(gameNewAction, SIGNAL(activated()), this, SLOT(newGame()));
    connect(gameSaveAction, SIGNAL(activated()), this, SLOT(saveGame()));
    connect(gameLoadAction, SIGNAL(activated()), this, SLOT(loadGame()));
    connect(helpHandbookAction, SIGNAL(activated()), this, SLOT(helpHandbook()));
    connect(helpAbout_QTAction, SIGNAL(activated()), this, SLOT(helpAboutQT()));
    connect(settingsConfigureKardsGTAction, SIGNAL(activated()), this, SLOT(settingsConfigureKardsGT()));
    connect(actionChangeName, SIGNAL(activated()), this, SLOT(changeName()));
    connect(gameSelectCribbageAction, SIGNAL(activated()), this, SLOT(turnOnCribbage()));
    connect(gameSelectWarAction, SIGNAL(activated()), this, SLOT(turnOnWar()));
    connect(gameSelectHeartsAction, SIGNAL(activated()), this, SLOT(turnOnHearts()));
    connect(gameSelectSpadesAction, SIGNAL(activated()), this, SLOT(turnOnSpades()));
    connect(gameSelectOld_MaidAction, SIGNAL(activated()), this, SLOT(turnOnOldMaid()));
    connect(gameSelectEuchreAction, SIGNAL(activated()), this, SLOT(turnOnEuchre()));
    connect(gameSelectCrazyEightsAction, SIGNAL(activated()), this, SLOT(turnOnCrazyEights()));
}

KardsGTInterface::~KardsGTInterface()
{
    delete m_pPlayerFactory;
    delete m_pGameFactory;
    if (m_pPlayer != NULL)
        delete m_pPlayer;
    if (m_pAssistant != NULL)
        delete m_pAssistant;
}

void KardsGTInterface::setupGame()
{
    // Find out which game we've selected
    if (m_gamePlayed.isEmpty())
        return;
    else
    {
        if (! m_pGameFactory->createGame(m_userProfileDB, m_gamePlayed, *Ui_KardsGTInterface::layout))
            throw KardsGTError("KardsGTInterface", "setupGame", "Failed to create game!");
        if (! m_loading)
        {
            PlayerSelection *m_pPlayerSelection=new PlayerSelection(this);
            m_pPlayerSelection->setGameName(m_gamePlayed);
            m_pPlayerSelection->setPlayersNeeded(m_pGameFactory->game().minimumPlayers(), m_pGameFactory->game().maximumPlayers());
            if (m_pPlayerSelection->exec())
            {
                m_opponents = m_pPlayerSelection->selectedPlayers();
                createOpponents();
            }
            else
            {
                delete m_pPlayerSelection;
                closeOpenGame();
                return;
            }
            delete m_pPlayerSelection;
        }
        else
        {
            createOpponents(); // loadGame should have filled our m_opponents information.
        }
    }
    m_pPlayer->setSkillLevel(playersLevel(m_gamePlayed));
    this->setBaseSize(Ui_KardsGTInterface::layout->widget(1)->size());
    Ui_KardsGTInterface::layout->setCurrentIndex(1);
    startGame();
}

Player::SkillLevel KardsGTInterface::playersLevel(const QString &game) const
{
    Player::SkillLevel level= Player::Skill_Error;
    pair<int, int> stat;
    int score=-1;

    stat=m_userProfileDB.gameStatistics(game);
    // Calculate winnings score
    if (stat.second != 0)
        score = static_cast<int>((stat.first / static_cast<double>(stat.second)) * 100);
    else
        score = 0;
    // Determine level of ability.
    if (score >= Player::MasterLowScore)
        level = Player::Master;
    else if ((score >= Player::PwnerLowScore) && (score < Player::MasterLowScore))
        level = Player::Pwner;
    else if ((score >= Player::ProfessionalLowScore) && (score < Player::PwnerLowScore))
        level = Player::Professional;
    else if ((score >= Player::ApprenticeLowScore) && (score < Player::ProfessionalLowScore))
        level = Player::Apprentice;
    else if ((score >= Player::AmateurLowScore) && (score < Player::ApprenticeLowScore))
        level = Player::Amateur;
    else if ((score >= Player::NoviceLowScore) && (score < Player::AmateurLowScore))
        level = Player::Novice;
    else if ((score >= Player::NewbLowScore) && (score < Player::NoviceLowScore))
        level = Player::Newb;
    else
        level = Player::Skill_Error;
    return level;
}

void KardsGTInterface::createOpponents()
{
    for (int i=0; i < static_cast<int>(m_opponents.size()); ++i)
        if (! m_pPlayerFactory->createPlayer(m_opponents[i], m_gamePlayed))
            throw KardsGTError("KardsGTInterface", "createOpponents", "Failed to create player!");
}

void KardsGTInterface::settingsConfigureKardsGT()
{
    ConfigureKardsGT conf(m_userProfileDB, this);

    if (conf.exec())
    {
        m_userProfileDB.setPathToCardFrontImages(conf.cardFrontImagePath());
        m_userProfileDB.setCardBackImageFilename(conf.cardBackImageFilename());
        m_userProfileDB.setUserMoodImage("NeutralMood", conf.neutralMoodFilename());
        if (m_pGameFactory->hasGame())
            dynamic_cast<QWidget *>(& m_pGameFactory->game())->repaint();
    }
}

void KardsGTInterface::changeName()
{
    bool ok;
    QString userName, suggestedName;

    if (m_userProfileDB.userName() == "UNKNOWN")
        suggestedName = QDir::home().dirName();
    else
        suggestedName = m_userProfileDB.userName();
    userName = QInputDialog::getText(
                   this, tr("KardsGT - Player Name"),
                   tr("Please, enter your name:"),
                   QLineEdit::Normal,
                   suggestedName,
                   &ok);
    if (ok)
        m_userProfileDB.setUserName(userName);
}

void KardsGTInterface::helpHandbook()
{
    m_pAssistant->openAssistant();
    // Set the path based on if we're developing or compiling for release
    #ifdef QT_NO_DEBUG
        m_pAssistant->showPage("/usr/share/kardsgt/toc.html");
    #else
        m_pAssistant->showPage("src/doc/toc.html");
    #endif
}

void KardsGTInterface::startGame()
{
    int size=m_pPlayerFactory->players().size();

    for (int i=0; i < size; ++i)
        m_pGameFactory->game().addOpponent(*m_pPlayerFactory->players()[i]);
    m_pGameFactory->game().addPlayer(*m_pPlayer);
    if (! m_loading)
    {
        m_pGameFactory->game().startPlay();
        this->setWindowTitle(QString("KardsGT - %1").arg(m_gamePlayed.toUpper()));
    }
    setActions(true);
}

void KardsGTInterface::closeOpenGame()
{
    if (m_pGameFactory->hasGame())
    {
        if (m_pPlayer != NULL)
            m_pPlayer->clear();
        setActions(false);
        Ui_KardsGTInterface::layout->setCurrentIndex(0);
        if (m_pPlayerFactory->players().size() > 0)
            m_pPlayerFactory->clear();
        dynamic_cast<QWidget *>(&m_pGameFactory->game())->close();
        m_pGameFactory->clear();
        m_gamePlayed = "";
        setWindowTitle(QString("KardsGT"));
        if (! m_loading)
            m_opponents.clear(); // Handled by the loadGame
    }
}

void KardsGTInterface::helpAbout()
{
    QMessageBox::about(this, "About KardsGT",
                       QString("<h2>KardsGT version %1</h2>"
                               "<p>Copyright &copy; %2 John Schneiderman"
                               "<p>KardsGT comes with ABSOLUTELY NO WARRANTY. <br>This is free software, and you are welcome to redistribute it under certain conditions; see the COPYING file for details, or the Free Software Foundation's GPL.").arg(VERSION.c_str()).arg(COPYRIGHT_YEARS.c_str()));
}

void KardsGTInterface::helpAboutQT()
{
    QMessageBox::aboutQt(this, tr("KardsGT"));
}

void KardsGTInterface::newGame()
{
    if (m_gamePlayed == "cribbage")
    {
        closeOpenGame();
        turnOnCribbage();
    }
    else if (m_gamePlayed == "war")
    {
        closeOpenGame();
        turnOnWar();
    }
    else if (m_gamePlayed == "euchre")
    {
        closeOpenGame();
        turnOnEuchre();
    }
    else if (m_gamePlayed == "hearts")
    {
        closeOpenGame();
        turnOnHearts();
    }
    else if (m_gamePlayed == "old maid")
    {
        closeOpenGame();
        turnOnOldMaid();
    }
    else if (m_gamePlayed == "spades")
    {
        closeOpenGame();
        turnOnSpades();
    }
    else if (m_gamePlayed == "crazy eights")
    {
        closeOpenGame();
        turnOnCrazyEights();
    }
    else
        throw KardsGTError("KardsGTInterface", "newGame", "Failed to find what game we're trying to restart!");
}

void KardsGTInterface::saveGame()
{
    QString filename = QFileDialog::getSaveFileName(
                           this,
                           "Choose a name to save game under...",
                           ".",
                           m_fileFilter
                       );
    if (! filename.isEmpty())
        if (filename.endsWith(".kgf") || filename.contains('.'))
        {
            if (! m_pGameFactory->game().save(filename))
                QMessageBox::critical(this, "Save Error", QString("The game file %1 failed to save!").arg(filename));
        }
        else
        {
            if (! m_pGameFactory->game().save(filename + ".kgf"))
                QMessageBox::critical(this, "Save Error", QString("The game file %1 failed to save!").arg(filename + ".kgf"));
        }
}

void KardsGTInterface::loadGame(QString filename)
{
    int numberOfPlayers=0;
    int m_pPlayersFound=1; // We always have a human player

    try
    {
        if (! filename.isEmpty())
        {
            QFile file(filename);
            QTextStream in(&file);
            QString garbage, gameName;

            if (! file.open(QIODevice::ReadOnly))
            {
                QMessageBox::critical(this, "Load Error", QString("The game file %1 failed to load!").arg(filename));
                file.close();
                return;
            }
            in.setCodec(QTextCodec::codecForName("UTF-8"));
            in >> garbage >> gameName;
            gameName.replace("_", " ");
            selectGameAction(gameName);
            if (m_gamePlayed == "unknown")
                throw KardsGTError("KardsGTInterface", "loadGame", "Failed to identify type of game file!");
            m_loading=true;
            m_opponents.clear();
            in >> garbage; //Read players label
            in >> garbage >> numberOfPlayers;
            do // Find the AI players
            {
                in >> garbage;
                if (garbage == "Name:")
                {
                    in >> garbage; // Should be players name.
                    m_opponents.push_back(garbage.toLower());
                    m_pPlayersFound++;
                }
            }
            while (numberOfPlayers != m_pPlayersFound);
            while (true) // Find the human players's name
            {
                in >> garbage;
                if (garbage == "Name:")
                    break;
            };
            in >> garbage;
            m_pPlayer->setName(garbage); // Set the human players name.
            file.close();
            newGame();
            if (m_pGameFactory->hasGame() && (! m_pGameFactory->game().load(filename)))
            {
                QMessageBox::critical(this, "Load Error", QString("The game file %1 failed to load!").arg(filename));
                closeOpenGame();
            }
            m_loading=false;
            this->setWindowTitle(QString("KardsGT - %1").arg(m_gamePlayed.toUpper()));
        }
    }
    catch (KardsGTError error)
    {
        cerr << error << std::endl;
        QMessageBox::critical(this, "Load Error", error.errorMessage());
        closeOpenGame();
    }
}

void KardsGTInterface::setActions(bool actionOn)
{
    // En/disable file operations.
    gameNewAction->setEnabled(actionOn);
    gameSaveAction->setEnabled(actionOn);
    gameCloseAction->setEnabled(actionOn);
    // En/disable games available.
    gameSelectCrazyEightsAction->setEnabled(! actionOn);
    gameSelectCribbageAction->setEnabled(! actionOn);
    gameSelectEuchreAction->setEnabled(! actionOn);
    gameSelectHeartsAction->setEnabled(! actionOn);
    gameSelectOld_MaidAction->setEnabled(! actionOn);
    gameSelectSpadesAction->setEnabled(! actionOn);
    gameSelectWarAction->setEnabled(! actionOn);
    // Toggle on/off the game we're playing
    if ((! actionOn) && gameSelectCrazyEightsAction->isChecked())
        gameSelectCrazyEightsAction->setChecked(actionOn);
    if ((! actionOn) && gameSelectCribbageAction->isChecked())
        gameSelectCribbageAction->setChecked(actionOn);
    if ((! actionOn) && gameSelectEuchreAction->isChecked())
        gameSelectEuchreAction->setChecked(actionOn);
    if ((! actionOn) && gameSelectHeartsAction->isChecked())
        gameSelectHeartsAction->setChecked(actionOn);
    if ((! actionOn) && gameSelectOld_MaidAction->isChecked())
        gameSelectOld_MaidAction->setChecked(actionOn);
    if ((! actionOn) && gameSelectWarAction->isChecked())
        gameSelectWarAction->setChecked(actionOn);
    if ((! actionOn) && gameSelectSpadesAction->isChecked())
        gameSelectSpadesAction->setChecked(actionOn);
}

void KardsGTInterface::selectGameAction(const QString &game)
{
    if (game == "crazy eights")
    {
        gameSelectCrazyEightsAction->setChecked(true);
        m_gamePlayed="crazy eights";
    }
    else if (game == "cribbage")
    {
        gameSelectCribbageAction->setChecked(true);
        m_gamePlayed="cribbage";
    }
    else if (game == "war")
    {
        gameSelectWarAction->setChecked(true);
        m_gamePlayed="war";
    }
    else if (game == "hearts")
    {
        gameSelectHeartsAction->setChecked(true);
        m_gamePlayed="hearts";
    }
    else if (game == "old maid")
    {
        gameSelectOld_MaidAction->setChecked(true);
        m_gamePlayed="old maid";
    }
    else if (game == "spades")
    {
        gameSelectSpadesAction->setChecked(true);
        m_gamePlayed="spades";
    }
    else if (game == "euchre")
    {
        gameSelectEuchreAction->setChecked(true);
        m_gamePlayed="euchre";
    }
    else
        m_gamePlayed="unknown";
}

void KardsGTInterface::loadGame()
{
    loadGame(QFileDialog::getOpenFileName(
                 this,
                 "Choose a game to load...",
                 "",
                 m_fileFilter
             ));
}


void KardsGTInterface::turnOnCribbage()
{
    selectGameAction("cribbage");
    setupGame();
}

void KardsGTInterface::turnOnWar()
{
    selectGameAction("war");
    setupGame();
}

void KardsGTInterface::turnOnHearts()
{
    selectGameAction("hearts");
    setupGame();
}

void KardsGTInterface::turnOnSpades()
{
    selectGameAction("spades");
    setupGame();
}

void KardsGTInterface::turnOnOldMaid()
{
    selectGameAction("old maid");
    setupGame();
}

void KardsGTInterface::turnOnEuchre()
{
    selectGameAction("euchre");
    setupGame();
}

void KardsGTInterface::turnOnCrazyEights()
{
    selectGameAction("crazy eights");
    setupGame();
}
