/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "sallycribbageai.h"
#include "cardproperties.h"
#include "kardsgterror.h"
#include "../../games/cribbage/cribbagerules.h"

#include <vector>
using std::vector;

SallyCribbageAI::SallyCribbageAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand): basicStrategies(rules)
{
    m_playSequence = playSequence;
    m_hand = hand;
    m_pRules = &rules;
}

SallyCribbageAI::~SallyCribbageAI()
{}

Card SallyCribbageAI::selectCard() const
{
    int score = 0;

    // If there is nothing on the board, no need to check special points
    if (! m_playSequence.isEmpty())
    {
        // Calculate the current score
        for (int index = 0, size = m_playSequence.size(); index < size; ++index)
            score += m_pRules->cardValue(m_playSequence[index]);

        // See if we can make 15
        for (int index = 0, size = m_hand.size(); index < size; ++index)
            if ((score + m_pRules->cardValue(m_hand[index])) == 15)
                if (m_pRules->isLegalPlay(m_playSequence, m_hand[index]))
                    return m_hand[index];
    }

    // Player our highest legal card
    int highestCardIndex = 0;
    for (int index = 1, size = m_hand.size(); index < size; ++index)
        if (! m_pRules->isLegalPlay(m_playSequence, m_hand[highestCardIndex]))
            highestCardIndex = index;
        else if (m_pRules->isLegalPlay(m_playSequence, m_hand[index]) && (m_hand[index] > m_hand[highestCardIndex]))
            highestCardIndex = index;
    return m_hand[highestCardIndex];
}

CardSequence SallyCribbageAI::myCrib() const
{
    vector<int> scores;
    vector<CardSequence> scoreCardsRemoved;
    CardSequence testHand;
    int lowestScoreIndex = 0;

    // Calcualte the different scores when we remove each two-card combinations.
    for (int index = 0, endCheck = m_hand.size() - 1; index < endCheck; ++index)
    {
        testHand = m_hand;
        scoreCardsRemoved.push_back(CardSequence());
        // Remove the cards we're testing
        testHand.removeCard(m_hand[index]);
        testHand.removeCard(m_hand[index + 1]);
        // Keep track of the cards we're testing
        scoreCardsRemoved[index].addCard(m_hand[index]);
        scoreCardsRemoved[index].addCard(m_hand[index + 1]);
        // Calculate and store the score for that combination
        scores.push_back(calculateScore(testHand));
    }
    // Remove the last combination, first and last card
    testHand = m_hand;
    scoreCardsRemoved.push_back(CardSequence());
    // Remove the cards we're testing
    testHand.removeCard(m_hand.front());
    testHand.removeCard(m_hand.back());
    // Keep track of the cards we're testing
    scoreCardsRemoved[scoreCardsRemoved.size() - 1].addCard(m_hand.front());
    scoreCardsRemoved[scoreCardsRemoved.size() - 1].addCard(m_hand.back());
    scores.push_back(calculateScore(testHand));

    // Find the lowest score
    for (int index = 1, size = scores.size(); index < size; ++ index)
        if (scores[index] < scores[lowestScoreIndex])
            lowestScoreIndex = index;

    return scoreCardsRemoved[lowestScoreIndex];
}

CardSequence SallyCribbageAI::opponentsCrib() const
{
    vector<int> scores;
    vector<CardSequence> scoreCardsRemoved;
    vector<CardSequence> testHands;
    CardProperties testProp(m_hand);
    int highScoreIndex = 0;

    // Calcualte the different scores when we remove each two-card combinations.
    testHands = testProp.combinations();
    for (int index = 0, endCheck = testHands.size(); index < endCheck; ++index)
    {
        // See if the combinations is four cards
        if (testHands[index].size() != 4)
            continue;

        // Keep track of the cards we're testing
        scoreCardsRemoved.push_back(CardSequence());
        for (int handIndex = 0, size = m_hand.size(); handIndex < size; ++handIndex)
            if (! testHands[index].hasCard(m_hand[handIndex]))
                scoreCardsRemoved.back().addCard(m_hand[handIndex]);

        // Calculate and store the score for that combination
        scores.push_back(calculateScore(testHands[index]));
    }

    // Find the lowest score
    for (int index = 1, size = scores.size(); index < size; ++ index)
        if (scores[index] > scores[highScoreIndex])
            highScoreIndex = index;

    return scoreCardsRemoved[highScoreIndex];
}

int SallyCribbageAI::calculateScore(const CardSequence &hand) const
{
    CardProperties handProp(hand);
    const CribbageRules *rules = static_cast<const CribbageRules *>(m_pRules);
    vector<CardSequence> pairs, fifteens;
    int finalScore = 0;

    // Calculate 15's
    fifteens = handProp.summations(15, *m_pRules);
    for (int i=0; i < static_cast<int>(fifteens.size()); ++i)
        finalScore += 2;

    // Score our pairs
    pairs = rules->createPairTypes(handProp.pairs());
    for (int i=0; i < static_cast<int>(pairs.size()); ++i)
        if (rules->pairType(pairs[i]) != CribbageRules::PAIR_ERR)
            finalScore += rules->pairType(pairs[i]);

    return finalScore;
}
