/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "wallycribbageai.h"
#include "cardproperties.h"
#include "basicgamestrategies.h"

#include <cstdlib>
#include <vector>
using std::vector;

WallyCribbageAI::WallyCribbageAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand)
{
    time_t seconds;

    m_playSequence = playSequence;
    m_pRules = &rules;
    m_hand = hand;
    time(&seconds);
    srand(seconds);
}

WallyCribbageAI::~WallyCribbageAI()
{}

CardSequence WallyCribbageAI::selectCards() const
{
    CardProperties properties(m_hand);
    int playSequenceTotal=0;

    //See what our round score is
    for (int i=0; i < m_playSequence.size(); ++i)
        playSequenceTotal+=m_pRules->cardValue(m_playSequence[i]);

    // Try to get 2 for Fifteen
    if ((playSequenceTotal > 0) && (playSequenceTotal < 15))
        for (int i=0; i < m_hand.size(); ++i)
            if ((playSequenceTotal + m_pRules->cardValue(m_hand[i])) == 15)
                if (m_pRules->isLegalPlay(m_playSequence, m_hand[i]))
                    return m_hand[i];

    // Try to get 2 for Thirty-One
    if (playSequenceTotal > 20)
        for (int i=0; i < m_hand.size(); ++i)
            if ((playSequenceTotal + m_pRules->cardValue(m_hand[i])) == 31)
                if (m_pRules->isLegalPlay(m_playSequence, m_hand[i]))
                    return m_hand[i];

    // Try to get 2 for a Pair
    if (m_playSequence.size() > 0)
        for (int i=0; i < m_hand.size(); ++i)
            if (m_playSequence.back().rank() == m_hand[i].rank())
                if (m_pRules->isLegalPlay(m_playSequence, m_hand[i]))
                    return m_hand[i];

    // Try to get a run of 3
    if (m_playSequence.size() > 1)
    {
        CardSequence test, runs;

        for (int i=0; i < m_hand.size(); ++i)
        {
            test.clear();
            test.addCard(m_playSequence.back());
            test.addCard(m_playSequence[m_playSequence.size() - 2]);
            test.addCard(m_hand[i]);
            CardProperties testRun(test);
            if (testRun.runs().size() > 0)
            {
                runs=testRun.runs().back();
                if (runs.size() > 0)
                    if (m_pRules->isLegalPlay(m_playSequence, m_hand[i]))
                        return m_hand[i];
            }
        }

    }

    // Play a card from a pair to try and lure for a proil
    if (properties.pairs().size() > 0)
        if (m_pRules->isLegalPlay(m_playSequence, properties.pairs().front().front()))
            return properties.pairs().front().front();

    // Try not to play a card that has a value of 5, 6, 7, 8, 9 or 10 while it's possible to make 15
    if (playSequenceTotal < 11)
    {
        int cardValue, possibleValue;

        for (int i=0; i < m_hand.size(); ++i)
        {
            cardValue=m_pRules->cardValue(m_hand[i]);
            possibleValue=playSequenceTotal+cardValue;
            if ((possibleValue == 5) ||
                    (possibleValue == 6) ||
                    (possibleValue == 7) ||
                    (possibleValue == 8) ||
                    (possibleValue == 9) ||
                    (possibleValue == 10))
                continue;
            if (m_pRules->isLegalPlay(m_playSequence, m_hand[i]))
                return m_hand[i];
        }
    }

    BasicGameStrategies basicStrategies(*m_pRules);
    // Player our highest legal card
    int highestCardIndex = 0;
    for (int index = 1, size = m_hand.size(); index < size; ++index)
        if (! m_pRules->isLegalPlay(m_playSequence, m_hand[highestCardIndex]))
            highestCardIndex = index;
        else if (m_pRules->isLegalPlay(m_playSequence, m_hand[index]) && (m_hand[index] > m_hand[highestCardIndex]))
            highestCardIndex = index;
    return m_hand[highestCardIndex];
}

CardSequence WallyCribbageAI::selectDealersCribCards() const
{
    CardProperties properties(m_hand);

    // Do we have a run to protect
    if (properties.runs().size() > 0)
    {
        CardSequence run=properties.runs().back();
        // We can discard the extra card or the last card in our run
        if (run.size() <= 4)
        {
            CardSequence crib;

            // Find cards not in the run
            for (int i=0; i < m_hand.size(); ++i)
            {
                if (! run.hasCard(m_hand[i]))
                    crib.addCard(m_hand[i]);
                if (crib.size() == 2)
                    break;
            }
            return crib;
        }
        else if (run.size() == 5)
        {
            CardSequence crib;

            //Find cards not in the run
            for (int i=0; i < m_hand.size(); ++i)
                if (! run.hasCard(m_hand[i]))
                    crib.addCard(m_hand[i]);

            //Find which card to pull from our run
            if (run[0].rank() != run[1].rank())
            {
                // See if we have a triple run
                if (m_hand[2].rank() == m_hand[3].rank())
                    crib.addCard(run[2]);
                else
                    crib.addCard(run[0]);
            }
            else
            {
                // Test to see if pulling the end card kills our run
                int secondToLastCard=run.size() - 2;
                if (m_hand[secondToLastCard].rank() == m_hand[secondToLastCard - 1].rank())
                    crib.addCard(m_hand[secondToLastCard]);
                else
                    crib.addCard(run.back());
            }
            return crib;
        }
        else
        {
            CardSequence crib;

            // We're just going to ether take the end two or the front two.
            if (run[0].rank() != run[1].rank())
            {
                // See if we have a triple run
                if (m_hand[2].rank() == m_hand[3].rank())
                    crib.addCard(run[2]);
                else
                    crib.addCard(run[0]);
                crib.addCard(run[1]);
            }
            else
            {
                crib.addCard(run.back());
                // Test to see if pulling the end card kills our run
                int secondToLastCard=run.size() - 2;
                if (m_hand[secondToLastCard].rank() == m_hand[secondToLastCard - 1].rank())
                    crib.addCard(m_hand[secondToLastCard]);
                else
                {
                    if (m_hand[secondToLastCard - 1].rank() == m_hand[secondToLastCard - 2].rank())
                        crib.addCard(m_hand[secondToLastCard - 1]);
                    else
                        crib.addCard(run[secondToLastCard]);
                }
            }
            return crib;
        }
    }

    // Try to give our crib a pair
    if (properties.pairs().size() == 2)
        return properties.pairs().front();

    // Put in something that makes 15
    vector<CardSequence> possibleSums;
    possibleSums = properties.summations(15, *m_pRules);
    int size=possibleSums.size();
    for (int i=0; i < size; ++i)
        if (possibleSums[i].size() == 2)
            return possibleSums[i];

    // If nothing else just randomly take cards
    BasicGameStrategies gameai(*m_pRules);
    return gameai.randomCards(2, m_hand, m_playSequence);
}

CardSequence WallyCribbageAI::selectNonDealersCribCards() const
{
    CardSequence hand=m_hand, testHand; // hand are the cards we'll send to the crib
    vector<CardSequence> test;
    CardProperties properties(hand);
    Card testCard;

    // Remove our runs
    test=properties.runs();
    if (testHand.size() > 0)
    {
        testHand=testHand.back();
        for (int i=0; i < testHand.size(); ++i)
            hand.removeCard(testHand[i]);
    }

    // Remove our pairs
    test=properties.pairs();
    for (int j=0; j < static_cast<int>(test.size()); ++j)
    {
        testHand=test[j];
        for (int i=0; i < testHand.size(); ++i)
            hand.removeCard(testHand[i]);
    }

    // Remove our 15s
    test=properties.summations(15, *m_pRules);
    for (int j=0; j < static_cast<int>(test.size()); ++j)
    {
        testHand=test[j];
        for (int i=0; i < testHand.size(); ++i)
            hand.removeCard(testHand[i]);
    }

    BasicGameStrategies valuableCardsAI(*m_pRules); // Used to select from the valuable hands.
    BasicGameStrategies nonValuableCardsAI(*m_pRules); // Used to select from the non-valuable hands.
    switch (hand.size())
    {
    case 0: // We have to break up two of our valuable cards.
        hand=valuableCardsAI.randomCards(2, m_hand, m_playSequence);
        break;
    case 1: // We have to break up one of our valuable cards.
        testCard=valuableCardsAI.randomCards(1, m_hand, m_playSequence).front();
        while (hand.hasCard(testCard))
            testCard=valuableCardsAI.randomCards(1, m_hand, m_playSequence).front();
        hand.addCard(testCard);
        break;
    case 2: // We have two non-valuable cards.
        break;
    default: // All of the remaining cards are non-valuable
        hand=nonValuableCardsAI.randomCards(2, hand, m_playSequence);
        break;
    }
    return hand;
}
