/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2008       John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "playerselection.h"
#include "playerfactory.h"

#include <QDialog>
#include <QWidget>
#include <QMessageBox>

PlayerSelection::PlayerSelection(QWidget *parent): QDialog(parent)
{
    Ui::PlayerSelection::setupUi(this);
    // signals and slots connections
    connect(startGame, SIGNAL(clicked()), this, SLOT(accept()));
    connect(cancel, SIGNAL(clicked()), this, SLOT(reject()));
    connect(m_pPlayersAvailable, SIGNAL(activated(const QString&)), this, SLOT(displayPlayerInfo()));
    connect(m_pButtonHelp, SIGNAL(clicked()), this, SLOT(help()));
    connect(m_pButtonRemovePlayer, SIGNAL(clicked()), this, SLOT(removePlayer()));
    connect(m_pButtonRemovePlayer, SIGNAL(clicked()), this, SLOT(validateNumberOfPlayers()));
    connect(m_pButtonAddPlayer, SIGNAL(clicked()), this, SLOT(addPlayer()));
    connect(m_pButtonAddPlayer, SIGNAL(clicked()), this, SLOT(validateNumberOfPlayers()));
    connect(m_pButtonAddPlayer, SIGNAL(clicked()), this, SLOT(displayPlayerInfo()));
    connect(m_pPlayersSelected, SIGNAL(currentItemChanged(QListWidgetItem*, QListWidgetItem *)), this, SLOT(addRemoveStatus()));
    connect(m_pPlayersSelected, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(addRemoveStatus()));

    vector<QString> players;
    PlayerFactory factory;

    players=factory.playerNames();
    for (int i=0; i < static_cast<int>(players.size()); ++i)
        m_pPlayersAvailable->addItem(players[i]);
    m_pPlayersAvailable->setCurrentIndex(0);
    m_currentNumberOfPlayers = 1;
    m_minimumPlayers = m_maximumPlayers = 0;
    m_gameName = "the unknown game";
    displayPlayerInfo();
}

vector<QString> PlayerSelection::selectedPlayers()
{
    vector<QString> players;

    for (int i=0, size=m_pPlayersSelected->count(); i < size; ++i)
        players.push_back(m_pPlayersSelected->item(i)->text().toLower());
    return players;
}

void PlayerSelection::setPlayersNeeded(int minimum, int maximum)
{
    m_minimumPlayers=minimum;
    m_maximumPlayers=maximum;
    if ((m_minimumPlayers == m_maximumPlayers) && (m_minimumPlayers == 2))
        instructions->setText(QString("Please select %1 player for %2.").arg(m_minimumPlayers - 1).arg(m_gameName));
    else if (m_minimumPlayers == m_maximumPlayers)
        instructions->setText(QString("Please select %1 players for %2.").arg(m_minimumPlayers - 1).arg(m_gameName));
    else
        instructions->setText(QString("Please select %1 to %2 players for %3.").arg(m_minimumPlayers - 1).arg(m_maximumPlayers - 1).arg(m_gameName));
}

void PlayerSelection::setGameName(const QString &name)
{
    m_gameName=name;
}

void PlayerSelection::addPlayer()
{
    if (m_pPlayersAvailable->currentText().isEmpty())
        return;

    m_pPlayersSelected->addItem(m_pPlayersAvailable->currentText());
    m_pPlayersAvailable->removeItem(m_pPlayersAvailable->currentIndex());
    if (m_pPlayersAvailable->count() > 0)
        m_pPlayersAvailable->setCurrentIndex(0);
    m_currentNumberOfPlayers++;
    addRemoveStatus();
}

void PlayerSelection::removePlayer()
{
    if (m_pPlayersSelected->count() == 0)
        return;

    if (m_currentNumberOfPlayers != 1)
    {
        m_pPlayersAvailable->addItem(m_pPlayersSelected->currentItem()->text());
        m_pPlayersSelected->takeItem(m_pPlayersSelected->currentRow());
        m_currentNumberOfPlayers--;
        if (m_pPlayersAvailable->count() > 0)
        {
            m_pPlayersAvailable->setCurrentIndex(0);
            m_pPlayersAvailable->showPopup();
            displayPlayerInfo();
        }
        addRemoveStatus();
    }
}

void PlayerSelection::validateNumberOfPlayers()
{
    if (m_currentNumberOfPlayers < m_minimumPlayers)
        startGame->setEnabled(false);
    else if (m_currentNumberOfPlayers > m_maximumPlayers)
        startGame->setEnabled(false);
    else
        startGame->setEnabled(true);
}

void PlayerSelection::displayPlayerInfo()
{
    if (! m_pPlayersAvailable->currentText().isEmpty())
        m_pPlayerImage->setPixmap(QPixmap(QString(":/players/" + m_pPlayersAvailable->currentText().toLower() + ".png")));
    else
        m_pPlayerImage->setPixmap(QPixmap(":/cardBacks/back00.png"));
}

void PlayerSelection::help()
{
    QMessageBox::information(this, tr("Player Selection Help"), tr("Select the name of the player you want to play against by clicking on the combobox.\n You can add the player by clicking on the add button. \n You can remove players by selecting the name in the selection box and clicking the remove button. \n Once you have selected enough players, the start game button will activate."));
}

void PlayerSelection::addRemoveStatus()
{
    if (m_pPlayersSelected->count() != 0)
        m_pButtonRemovePlayer->setEnabled(true);
    else
        m_pButtonRemovePlayer->setEnabled(false);
    if (m_pPlayersAvailable->count() != 0)
        m_pButtonAddPlayer->setEnabled(true);
    else
        m_pButtonAddPlayer->setEnabled(false);
}
