// Action_recorder.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Action_recorder.hpp"
#include "common.hpp"
#include "Segment.hpp"
#include "error/Record_error.hpp"
#include <cmath>

using Roan_trail::Long_int;
using namespace Roan_trail::Recorder;

//
// Action structure
//

Action& Action::operator=(const Action& a)
{
  if (this == &a)
  {
    goto exit_point;
  }

  // const cast here is OK because
  // because we are in operator=
  *const_cast<int*>(&action_type) = a.action_type;
  *const_cast<Long_int*>(&audio_frame) = a.audio_frame;
  *const_cast<Long_int*>(&index) = a.index;

 exit_point:
  return *this;
}

const int Action::start;
const int Action::new_index;
const int Action::retake;
const int Action::stop;
const int Action::pause;

//
// Constructor/destructor

Action_recorder::Action_recorder()
  : m_current_audio_frame_rate(0), m_index(0), m_actions()
{
  postcondition(mf_invariant(false));
}

Action_recorder::~Action_recorder()
{
  precondition(mf_invariant(false));
}

//
// Accessors
//

bool Action_recorder::segments(vector<Segment>& return_segments, Error_param& return_error) const
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  vector<Segment> segments;
  Error_param error;
  const bool scanned = mf_scan_segments(segments, error);
  on_error(!scanned, new Record_error(error_location(),
                                      Record_error::general,
                                      error()));

  return_segments = segments;
  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  postcondition((return_value || !return_error.need_error() || return_error()));
  return return_value;
}

//
// Control
//

//
//   ...basic Recorder functions
//

void Action_recorder::record(Long_int index, Long_int audio_frame)
{
  precondition((state() != Recorder::state_recording)
               && (audio_frame >= 0)
               && mf_invariant());

  m_index = index;

  if (state() == Recorder::state_stopped)
  {
    // start recording from stop
    m_actions.clear();
    const Action a(Action::start, index, 0);
    m_actions.push_back(a);
  }
  else if (state() == Recorder::state_paused)
  {
    // resume from pause
    const Action a(Action::new_index, index, audio_frame);
    m_actions.push_back(a);
  }
  else
  {
    assert(false && "invalid state when recording in action recorder");
  }

  Recorder::record();

  postcondition((m_index == index)
                && (m_actions.size() > 0)
                && mf_invariant());
}

void Action_recorder::pause()
{
  precondition(mf_invariant());

  Recorder::pause();

  postcondition(mf_invariant());
}

void Action_recorder::stop(Long_int audio_frame)
{
  precondition((audio_frame >= 0)
               && mf_invariant());

  const Action a(Action::stop, m_index, audio_frame);
  m_actions.push_back(a);

  Recorder::stop();

  postcondition((m_actions.size() > 1)
                && mf_invariant());
}

//
//   ...action recorder functions
//

void Action_recorder::new_index(Long_int audio_frame, Long_int index)

{
  precondition((state() == Recorder::state_recording)
               && (audio_frame >= 0)
               && mf_invariant());

  m_index = index;

  const Action a(Action::new_index, index, audio_frame);
  m_actions.push_back(a);

  postcondition((state() == Recorder::state_recording)
                && (index == m_index)
                && (m_actions.size() > 1)
                && mf_invariant());
}

void Action_recorder::retake(Long_int audio_frame)
{
  precondition((state() == Recorder::state_recording)
               && (audio_frame >= 0)
               && mf_invariant());

  const Action a(Action::retake, m_index, audio_frame);
  m_actions.push_back(a);

  postcondition((state() == Recorder::state_recording)
                && (m_actions.size() > 1)
                && mf_invariant());
}

//
// Protected member functions
//

// invariant check
bool Action_recorder::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  const int cur_state = Recorder::state();

  if (Recorder::state_stopped != cur_state)
  {
    if (m_current_audio_frame_rate <= 0)
    {
      // need a positive frame rate when recording
      goto exit_point;
    }
    if (m_actions.size() < 1)
    {
      // we have at least one action when recording
      goto exit_point;
    }
  }

  return_value = (!check_base_class || Recorder::mf_invariant(check_base_class));

 exit_point:
  return return_value;
}

//
// Private member functions
//

bool Action_recorder::mf_scan_segments(vector<Segment>& return_segments, Error_param& return_error) const
{
  precondition((m_current_audio_frame_rate > 0)
               && !return_error());

  bool return_value = false;

  start_error_block();

  vector<Segment> segments;

  Long_int index;
  Long_int save_index = 0;
  Long_int start_frame;
  Long_int end_frame;
  Long_int save_start_frame = 0;

  bool have_start = false;
  bool in_retake = false;
  Long_int segment_number = 0;
  Long_int frame_count;
  for (vector<Action>::const_iterator a = m_actions.begin();
       a != m_actions.end();
       ++a)
  {
    switch (a->action_type)
    {
    case Action::start:
      on_error(have_start, new Record_error(error_location(),
                                            Record_error::parse_segments,
                                            string("duplicate start label found")));
      index = a->index;
      save_index = index;
      save_start_frame = 0;
      have_start = true;
      in_retake = false;
      break;
    case Action::new_index:
      on_error(!have_start, new Record_error(error_location(),
                                             Record_error::parse_segments,
                                             string("start not found for new index action")));
      index = a->index;
      start_frame = a->audio_frame;
      frame_count = (start_frame - 1) - save_start_frame + 1;
      on_error((frame_count < 0), new Record_error(error_location(),
                                                   Record_error::parse_segments,
                                                   string("frame count less than 0 for new index action")));
      if (frame_count > 0)
      {
        const Segment previous_segment(save_index,
                                       save_start_frame,
                                       start_frame - 1,
                                       segment_number,
                                       in_retake);
        ++segment_number;
        segments.push_back(previous_segment);
        save_index = index;
        save_start_frame = start_frame;
      }
      else
      {
        // if no frames, just record the index
        save_index = index;
        // and keep the previous saved start frame
      }
      in_retake = false;
      break;
    case Action::stop:
      on_error(!have_start, new Record_error(error_location(),
                                             Record_error::parse_segments,
                                             string("start not found for stop action")));
      end_frame = a->audio_frame;
      frame_count = end_frame - save_start_frame + 1;
      on_error((frame_count < 0), new Record_error(error_location(),
                                                   Record_error::parse_segments,
                                                   string("frame count less than 0 for stop action")));
      if (frame_count > 0)
      {
        const Segment previous_segment(save_index,
                                       save_start_frame,
                                       end_frame,
                                       segment_number,
                                       in_retake);
        ++segment_number;
        segments.push_back(previous_segment);
      }
      in_retake = false;

      return_value = true;
      return_segments = segments;
      goto exit_point;
      break;
    case Action::retake:
      on_error(!have_start, new Record_error(error_location(),
                                             Record_error::parse_segments,
                                             string("start not found for retake action")));
      index = a->index;
      start_frame = a->audio_frame;
      frame_count = (start_frame - 1) - save_start_frame + 1;
      on_error((frame_count < 0), new Record_error(error_location(),
                                                   Record_error::parse_segments,
                                                   string("frame count less than 0 for retake action")));
      if (frame_count > 0)
      {
        const Segment previous_segment(save_index,
                                       save_start_frame,
                                       start_frame - 1,
                                       segment_number,
                                       in_retake);
        ++segment_number;
        segments.push_back(previous_segment);

        save_index = index;
        save_start_frame = start_frame;
        in_retake = true;
      }
      else
      {
        // if no frames, just record the index
        save_index = index;
        // and keep the previous saved start frame
      }
      break;
    default:
      on_error(true, new Record_error(error_location(),
                                      Record_error::parse_segments,
                                      "invalid action"));
      break;
    }
  }

  // it is an error if we reach this point, because a stop action wasn't found
  on_error(true, new Record_error(error_location(),
                                  Record_error::parse_segments,
                                  "stop action not found"));

  end_error_block();

  default_error_handler_and_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  return return_value;
}
