// Device_enum.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

//
// This code was based on the PortAudio file pa_devs.c:
//
//    This program uses the PortAudio Portable Audio Library.
//    For more information see: http://www.portaudio.com
//    Copyright (c) 1999-2000 Ross Bencina and Phil Burk
//
//    Permission is hereby granted, free of charge, to any person obtaining
//    a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be
//    included in all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
//    IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
//    ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
//    CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//    WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

#include "Device_enum.hpp"
#include <sstream>
#include "error/Portaudio_error.hpp"

using std::endl;
using std::stringstream;
using namespace Roan_trail::Recorder;

//
// Internal helper
//
namespace
{
  void ih_output_sample_rates(stringstream& s,
                              const PaStreamParameters* input_parameters,
                              const PaStreamParameters* output_parameters)
  {
    // negative terminated list
    static double standard_sample_rates[] =
    {
        8000.0,   9600.0,  11025.0,  12000.0,  16000.0,
       22050.0,  24000.0,  32000.0,  44100.0,  48000.0,
       88200.0,  96000.0, 192000.0,     -1
    };

    int count = 0;
    for (int device = 0; standard_sample_rates[device] > 0; device++)
    {
      const PaError err = Pa_IsFormatSupported(input_parameters,
                                               output_parameters,
                                               standard_sample_rates[device]);
      if (paFormatIsSupported == err)
      {
        if (0 == count)
        {
          s << "     " << standard_sample_rates[device];
          count = 1;
        }
        else if (4 == count)
        {
          s << "," << endl << "     " << standard_sample_rates[device];
          count = 1;
        }
        else
        {
          s << ", " << standard_sample_rates[device];
          ++count;
        }
      }
    }
    if (!count)
    {
      s << "     None" << endl;
    }
    else
    {
      s << endl;
    }
  }
}

bool Device_enum::list_devices(PaSampleFormat sample_format,
                               string &return_devices,
                               Error_param& return_error)
{
  bool return_value = false;

  bool PA_initialized = false;

  start_error_block();

  PaError PA_error = Pa_Initialize();
  on_error(PA_error != paNoError, new Portaudio_error(PA_error, "Pa_Initialize"));
  PA_initialized = true;

  stringstream s;
  // s << "PortAudio version number = " << Pa_GetVersion() << endl;
  // s << "PortAudio version text = '" << Pa_GetVersionText() << endl;
  const int num_devices = Pa_GetDeviceCount();
  on_error(num_devices < 0, new Portaudio_error(num_devices, "Pa_GetDeviceCount"));

  s << "Number of devices: " << num_devices << endl;
  for (int device = 0; device < num_devices; device++)
  {
    const PaDeviceInfo* device_info = Pa_GetDeviceInfo(device);
    s << "-----" << endl;
    if (device_info->maxInputChannels == 0)
    {
      s << "[X] "; // mark as not available for input
    }
    s << "Device number: " << device << endl;

    // mark global and API specific default devices
    bool default_displayed = false;
    if(Pa_GetDefaultInputDevice() == device)
    {
      s << "[ Default Input";
      default_displayed = true;
    }
    else if (Pa_GetHostApiInfo(device_info->hostApi )->defaultInputDevice == device)
    {
      const PaHostApiInfo* host_info = Pa_GetHostApiInfo(device_info->hostApi);
      s << "[ Default " << host_info->name << " Input";
      default_displayed = true;
    }
    // no final else needed, we've checked for default devices

    if (Pa_GetDefaultOutputDevice() == device)
    {
      s << (default_displayed ? "," : "[");
      s << " Default Output";
      default_displayed = true;
    }
    else if (Pa_GetHostApiInfo(device_info->hostApi)->defaultOutputDevice == device)
    {
      const PaHostApiInfo* host_info = Pa_GetHostApiInfo(device_info->hostApi);
      s << (default_displayed ? "," : "[");
      s << " Default " << host_info->name << " Output";
      default_displayed = true;
    }
    // no final else needed, we've checked for default devices

    if (default_displayed)
    {
      s << " ]" << endl;
    }

    // print device info fields
    s << "Device name: " << device_info->name << endl;
    s << "Audio subsystem: " <<  Pa_GetHostApiInfo(device_info->hostApi)->name << endl;
    s << "Maximum inputs: " << device_info->maxInputChannels << endl;

    s << "Default sample rate: " << device_info->defaultSampleRate << endl;

    // poll for standard sample rates
    PaStreamParameters input_parameters;
    input_parameters.device = device;
    input_parameters.channelCount = device_info->maxInputChannels;
    input_parameters.sampleFormat = sample_format;
    input_parameters.suggestedLatency = 0; // ignored by Pa_IsFormatSupported()
    input_parameters.hostApiSpecificStreamInfo = 0;

    if (input_parameters.channelCount > 0)
    {
      s << "Supported standard sample rates for " <<  input_parameters.channelCount;
      s << " channels:" << endl;
      ih_output_sample_rates(s, &input_parameters, 0);
    }
  }

  PA_error = Pa_Terminate();
  PA_initialized = false;
  on_error(PA_error != paNoError, new Portaudio_error(PA_error, "Pa_Terminate"));

  s << "-----" << endl << endl;
  s << "Note: [X] by device number indicates it cannot be used as an input." << endl;
  return_devices = s.str();

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (PA_initialized)
  {
    Pa_Terminate(); // ignore error
  }
  return_value = false;
  goto exit_point;

 exit_point:
  return return_value;
}
