// File_manager.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// prevent use of old features
#define BOOST_FILESYSTEM_NO_DEPRECATED

#include "File_manager.hpp"
#include "error/Boost_error.hpp"
#include "error/Posix_error.hpp"
#include <boost/filesystem.hpp>
#include <boost/filesystem/path.hpp>
#include <cstdlib>
#include <cstring>
#include <sys/statfs.h>
#include <sys/stat.h>
#include <unistd.h>

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if ! HAVE_MKSTEMPS
// use our own internal mkstemps
extern "C"
{
  int internal_mkstemps(char *tmpl, int suffixlen);
}
#endif

using namespace boost::filesystem;
using namespace Roan_trail;

//
// Internal constants
//

namespace
{
  const char* ic_temporary_file_prefix= "/temp_";
  const char* ic_temporary_movie_file_prefix= "/movie_";
  const char* ic_temporary_sound_file_prefix= "/recording_";
  const char* ic_temporary_file_template = "XXXXXX";
}

//
// Class member functions
//

//
// --Predicates--
//

bool File_manager::path_exists(const string& path_arg)
{
  bool return_value = false;

  try
  {
    path p(path_arg);

    return_value = exists(p);
  }
  catch (...)
  {
    return_value = false;
  }

  return return_value;
}

bool File_manager::path_is_directory(const string& path_arg)
{
  bool return_value = false;

  try
  {
    path p(path_arg);

    return_value = boost::filesystem::is_directory(p);
  }
  catch (...)
  {
    // intentionally left blank
  }

  return return_value;
}

bool File_manager::path_is_writable(const string& path_arg)
{
  return (access(path_arg.c_str(), W_OK) == 0);
}

//
// -- Accessors --
//

string File_manager::temporary_directory()
{
  const char *temp_dir = getenv("TMPDIR"); // Standards Rule 24 deviation (check_code_ignore)
  try
  {
    if (!temp_dir || (*temp_dir == 0) || !boost::filesystem::is_directory(path(temp_dir)))
    {
      temp_dir = "/tmp";
    }
  }
  catch (...)
  {
    temp_dir = "/tmp";
  }

  return string(temp_dir);;
}

//
// -- File operations --
//

bool File_manager::create_temporary_file(const string& file_extension,
                                         Temporary_file_type file_type,
                                         string& return_file_name,
                                         int& return_file_descriptor,
                                         Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  return_file_name = "";
  string temp_dir = temporary_directory();

  // declarations for error cleanup
  char* temporary_file_buffer = 0;

  start_error_block();

  string temporary_file_prefix;
  if (File_manager::temporary_movie_file == file_type)
  {
    temporary_file_prefix = ic_temporary_movie_file_prefix;
  }
  else if (File_manager::temporary_sound_file == file_type)
  {
    temporary_file_prefix = ic_temporary_sound_file_prefix;
  }
  else
  {
    temporary_file_prefix = ic_temporary_file_prefix;
  }
  // construct the template for mkstemp
  const string file_template = temp_dir + string(temporary_file_prefix)
    + string(ic_temporary_file_template) + file_extension;
  const char* file_system_representation = file_template.c_str();
  const size_t representation_length = strlen(file_system_representation);
  // Standards Rule #163 deviation
  temporary_file_buffer = new char[(representation_length + 1)];
  strncpy(temporary_file_buffer,
          file_system_representation,
          representation_length + 1);

#if HAVE_MKSTEMPS
  int fd = mkstemps(temporary_file_buffer, static_cast<int>(file_extension.length()));
  on_error(Posix_error::error_return == fd, new Posix_error(errno, // (check_code_ignore)
                                                            "mkstemps",
                                                            temporary_file_buffer));
#else
  int fd = internal_mkstemps(temporary_file_buffer, static_cast<int>(file_extension.length()));
  on_error(Posix_error::error_return == fd, new Posix_error(errno, // (check_code_ignore)
                                                            "internal_mkstemps",
                                                            temporary_file_buffer));
#endif

  return_file_name = temporary_file_buffer;
  return_file_descriptor = fd;

  delete [] temporary_file_buffer;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  delete [] temporary_file_buffer;
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(return_value || !return_error.need_error() || return_error());
  return return_value;
}

bool File_manager::move_file(const string& source_file_path,
                             const string& dest_file_path,
                             bool overwrite_dest,
                             Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  try
  {
    path source_file(source_file_path);
    path dest_file(dest_file_path);

    if (overwrite_dest)
    {
      boost::system::error_code remove_error;
      remove(dest_file_path); // ignore error
    }
    boost::filesystem::rename(source_file, dest_file);
  }
  catch (const filesystem_error& e)
  {
    on_error(true, new Boost_error(error_location(), e.code()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

 error_handler:
  if (return_error.need_error())
  {
    handler_error->error_dictionary()[Error::file_path_error_key] = source_file_path;
    handler_error->error_dictionary()[Error::to_path_error_key] = dest_file_path;
    return_error = handler_error;
  }
  goto error_cleanup;

  default_error_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  postcondition(return_value || !return_error.need_error() || return_error());
  return return_value;
}

bool File_manager::move_file_with_backup(const string& source_file_path,
                                         const string& dest_file_path,
                                         bool overwrite_dest,
                                         bool &return_backed_up,
                                         string &return_backup_file_path,
                                         Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  return_backed_up = false;
  return_backup_file_path = "";

  path source_file(source_file_path);
  path dest_file(dest_file_path);

  try
  {
    copy_file(source_file, dest_file); // exception if dest file already exists
  }
  catch (const filesystem_error& e)
  {
    if (overwrite_dest)
    {
      try
      {
        Error_param error;
        const string backup_file_name = backup_path_for_original(dest_file.string());
        const path backup_file(backup_file_name);
        remove(backup_file);
        rename(dest_file, backup_file);
        copy_file(source_file, dest_file);
        // if the backup went well, return that the backup was made and its path
        return_backed_up = true;
        return_backup_file_path = backup_file.string();
      }
      catch (const filesystem_error& f)
      {
        on_error(true, new Boost_error(error_location(),
                                       f.code()));
      }
    }
    else
    {
      on_error(true, new Boost_error(error_location(),
                                     e.code()));
    }
  }

  // if we got to this point, the source file has been copied,
  // now remove it
  try
  {
    remove(source_file);
  }
  catch (const filesystem_error& e)
  {
    on_error(true, new Boost_error(error_location(),
                                     e.code()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

 error_handler:
  if (return_error.need_error())
  {
    handler_error->error_dictionary()[Error::file_path_error_key] = source_file_path;
    handler_error->error_dictionary()[Error::to_path_error_key] = dest_file_path;
    return_error = handler_error;
  }
  goto error_cleanup;

  default_error_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  postcondition(return_value || !return_error.need_error() || return_error());
  return return_value;
}

string File_manager::backup_path_for_original(const string& original_path)
{
  const path original_file_path(original_path);
  const string backup_dir = original_file_path.parent_path().string();
  const string backup_file_stem = original_file_path.stem();
  const string backup_file_extension = original_file_path.extension();
  // create an emacs style backup name based on the original file name
  const string backup_file_name = backup_dir + "/" + backup_file_stem + "~" + backup_file_extension;
  const path backup_file(backup_file_name);

  return backup_file.string();
}

bool File_manager::remove_file(const string& file_path,
                               Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  try
  {
    path file(file_path);

    boost::filesystem::remove(file);
  }
  catch (const filesystem_error& e)
  {
    on_error(true, new Boost_error(error_location(), e.code()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

 error_handler:
  if (return_error.need_error())
  {
    handler_error->error_dictionary()[Error::file_path_error_key] = file_path;
    return_error = handler_error;
  }
  goto error_cleanup;

  default_error_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  postcondition(return_value || !return_error.need_error() || return_error());
  return return_value;
}

//
// -- File system attributes --
//

bool File_manager::file_size_for_path(const string& path_arg,
                                      Long_int& return_file_size,
                                      Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  struct stat stat_buffer;
  int stat_error = stat(path_arg.c_str(), &stat_buffer);
  on_error(Posix_error::error_return == stat_error, new Posix_error(errno, "stat", 0)); // (check_code_ignore)
  return_file_size = static_cast<Long_int>(stat_buffer.st_size);

  return_value = true;
  goto exit_point;

  end_error_block();

 error_handler:
  if (return_error.need_error())
  {
    return_error = handler_error;
  }
  return_file_size = 0.0;
  goto error_cleanup;

  default_error_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  postcondition(return_value || !return_error.need_error() || return_error());
  return return_value;
}

bool File_manager::file_system_available_for_path(const string& path_arg,
                                                  double& return_fraction_available,
                                                  Long_int& return_available_bytes,
                                                  Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  const string use_path = ("" == path_arg) ? "." : path_arg;

  start_error_block();

  struct statfs statfs_buffer;
  int statfs_error = statfs(use_path.c_str(), &statfs_buffer);
  on_error(Posix_error::error_return == statfs_error, new Posix_error(errno, // (check_code_ignore)
                                                                      "statfs",
                                                                      0));

  // fields of struct statfs:
  //
  // f_blocks: total data blocks in file system
  // f_bavail: free blocks avail to non-superuser
  // f_bsize fundamental file system block size

  return_fraction_available = static_cast<double>(statfs_buffer.f_bavail)
    / static_cast<double>(statfs_buffer.f_blocks);
  return_available_bytes = static_cast<Long_int>(statfs_buffer.f_bavail)
    * static_cast<Long_int>(statfs_buffer.f_bsize);

  return_value = true;
  goto exit_point;

  end_error_block();

 error_handler:
  if (return_error.need_error())
  {
    return_error = handler_error;
  }
  return_fraction_available = 0.0;
  return_available_bytes = 0;
  goto error_cleanup;

  default_error_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  postcondition(return_value || !return_error.need_error() || return_error());
  return return_value;
}

bool File_manager::contents_of_directory_at_path(const string& path_arg, vector<string>& return_paths)
{
  bool return_value = false;

  return_paths.clear();

  try
  {
    path p(path_arg);

    if (!boost::filesystem::is_directory(p))
    {
      goto exit_point;
    }

    for (directory_iterator i(p); i != directory_iterator(); ++i)
    {
      if (is_regular_file(i->status()))
      {
        return_paths.push_back(i->path().filename());
      }
    }

    return_value = true;
  }
  catch (...)
  {
    // intentionally left blank
  }

 exit_point:
  return return_value;
}

//
// -- File name helpers --
//

string File_manager::parent_directory_for_path(const string& path_arg)
{
  string return_value("");

  try
  {
    path p(path_arg);
    return_value = p.parent_path().string();
  }
  catch (...)
  {
    // do nothing
  }

  return return_value;
}

string File_manager::file_extension_for_file(const string& file_path)
{
  string return_value("");

  try
  {
    path p(file_path);
    path e = p.extension();
    string extension = e.string();
    if ("" == extension)
    {
      goto exit_point;
    }
    else
    {
      // strip the dot
      return_value = extension.substr(1, extension.length());
    }
  }
  catch (...)
  {
    // do nothing
  }

 exit_point:
  return return_value;
}

string File_manager::file_name_for_file_path(const string& file_path)
{
  string return_value("");

  try
  {
    path p(file_path);
    path n = p.filename();
    return_value = n.string();

    goto exit_point;
  }
  catch (...)
  {
    // do nothing
  }

 exit_point:
  return return_value;
}

string File_manager::replace_file_extension(const string& file_path, const string& extension)
{
  string return_value("");

  try
  {
    path p(file_path);
    path n = p.replace_extension(extension);
    return_value = n.string();

    goto exit_point;
  }
  catch (...)
  {
    // do nothing
  }

 exit_point:
  return return_value;
}

//
// -- Misc --
//

bool File_manager::executable_dir(string& return_dir)
{
  bool return_value = false;

  int buffer_size = 100;
  char* buffer = 0;

  while (true)
  {
    buffer = static_cast<char*>(realloc(buffer, buffer_size));
    // PORTING: "/proc/self/exe" may not be portable to non-Linux systems
    const int char_count = readlink("/proc/self/exe",
                                    buffer,
                                    buffer_size);
    if (char_count < 0)
    {
      free(buffer);
      goto exit_point;
    }
    if (char_count < buffer_size)
    {
      break;
    }
    buffer_size *= 2;
  }

  return_dir = parent_directory_for_path(buffer);
  return_value = true;
  goto exit_point;

exit_point:
  return return_value;
}

//
// Class constants
//

const File_manager::Temporary_file_type File_manager::temporary_file;
const File_manager::Temporary_file_type File_manager::temporary_movie_file;
const File_manager::Temporary_file_type File_manager::temporary_sound_file;
