// Movie_builder_model.cpp
//
// Copyright 2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Movie_builder_model.hpp"
#include "Narration_recorder_importer.hpp"
#include "Sound_file_config.hpp"
#include "Segment.hpp"
#include "error/Build_error.hpp"
#include <libxml++/nodes/node.h>
#include <libxml2/libxml/tree.h>

using xmlpp::Node;
using Roan_trail::Recorder::Narration_recorder_importer;
using Roan_trail::Recorder::Segment;
using Roan_trail::Recorder::Sound_file_config;
using namespace Roan_trail::Builder;

//
// Constructor/destructor
//

Movie_builder_model::Movie_builder_model()
  : m_audio_recording_path(),
    m_audio_recording_frame_rate(Sound_file_config::default_sample_rate),
    m_segments()
{
  postcondition(mf_invariant(false));
}

//
// Load
//

bool Movie_builder_model::load_from_XML(const string& file_path,
                                        const Node* XML_node,
                                        Error_param& return_error)
{
  precondition(XML_node
               && !return_error()
               && mf_invariant());

  vector<Segment> segments;
  string audio_recording_path;
  Long_int audio_recording_frame_rate = 0;
  bool return_value = false;

  start_error_block();

  Error_param error;

  const bool segments_imported = Narration_recorder_importer::import_segments_from_XML(file_path,
                                                                                       XML_node,
                                                                                       segments,
                                                                                       error);
  on_error(!segments_imported, new Build_error(error_location(),
                                               Build_error::general,
                                               error()));

  const bool sound_file_imported =
    Narration_recorder_importer::import_audio_recording_from_XML(file_path,
                                                                 XML_node,
                                                                 audio_recording_path,
                                                                 audio_recording_frame_rate,
                                                                 error);
  on_error(!sound_file_imported, new Build_error(error_location(),
                                                 Build_error::general,
                                                 error()));

  // success, update data members
  m_segments = segments;
  m_audio_recording_path = audio_recording_path;
  m_audio_recording_frame_rate = audio_recording_frame_rate;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  postcondition((return_value || !return_error.need_error() || return_error())
                && mf_invariant());
  return return_value;
}

//
// Protected member functions
//

bool Movie_builder_model::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}
