// Movie_config.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Movie_config.hpp"
#include "Sound_file_config.hpp"
#include <boost/algorithm/string.hpp>

using boost::algorithm::to_lower;
using Roan_trail::Rect_size;
using namespace Roan_trail::Builder;

namespace
{
  bool ih_config_value_for_string(const string& config_string,
                                  int &return_config_value,
                                  const Movie_config::Constant_map& config_map)
  {
    bool return_value = false;

    string to_find = config_string;
    to_lower(to_find);
    Movie_config::Constant_map::const_iterator p = config_map.begin();
    while (p != config_map.end())
    {
      if (p->second == to_find)
      {
        return_config_value = p->first;

        return_value = true;
        goto exit_point;
      }
      p++;
    }

    return_value = false;
    goto exit_point;

  exit_point:
    return return_value;
  }

}

//
// Constructor/destructor

Movie_config::Movie_config()
  // setup some reasonable defaults
  : movie_preset(default_movie_preset),
    frame_rate(default_frame_rate_type),
    frame_format(default_frame_format),
    original_size(),
    keyframe_period(0),
    sound(new Sound_file_config)
{
  if (!m_maps_setup)
  {
    m_maps_setup = true;
    mf_setup_maps();
  }
  output_size_for_movie_config(*this, original_size);
}

Movie_config::~Movie_config()
{
  delete sound;
}

//
// Helpers
//

bool Movie_config::movie_preset_for_string(const string& movie_preset_string, Movie_preset& return_movie_preset)
{
  return ih_config_value_for_string(movie_preset_string,
                                    return_movie_preset,
                                    m_movie_preset_map);
}

void Movie_config::movie_preset_string_for_file_extension(const string& extension,
                                                          string& return_movie_preset_string)
{
  string return_value;
  string extension_LC = extension;
  to_lower(extension_LC);

  if ("mov" == extension_LC)
  {
    return_movie_preset_string = string_for_movie_preset(movie_preset_qtrle);
  }
  else
  {
    // no preset found
    return_movie_preset_string = "";
  }
}

bool Movie_config::frame_rate_type_for_string(const string& frame_rate_type_string,
                                              Frame_rate_type& return_frame_rate)
{
  return ih_config_value_for_string(frame_rate_type_string,
                                    return_frame_rate,
                                    m_frame_rate_type_map);
}

bool Movie_config::audio_sample_rate_compatible_with_frame_rate_type(int audio_sample_rate, Frame_rate_type rate)
{
  precondition((audio_sample_rate >= 0));

  Frame_rate video_frame_rate;
  frame_rate_for_frame_rate_type(rate, video_frame_rate);
  // audio frame rate is compatible if it is a multiple of the frame rate time scale
  return ((audio_sample_rate % video_frame_rate.time_scale()) == 0);
}

void Movie_config::compatible_frame_rates_for_audio_sample_rate(int audio_sample_rate,
                                                                vector<Frame_rate_type>& return_frame_rates)
{
  precondition((audio_sample_rate >= 0)
               && !return_frame_rates.size());

  for (Frame_rate_type rate = frame_rate_5_Hz; rate <= frame_rate_60_Hz; ++rate)
  {
    if (audio_sample_rate_compatible_with_frame_rate_type(audio_sample_rate, rate))
    {
      return_frame_rates.push_back(rate);
    }
  }
}

bool Movie_config::frame_format_for_string(const string& frame_format_string, Frame_format& return_frame_format)
{
  return ih_config_value_for_string(frame_format_string,
                                    return_frame_format,
                                    m_frame_format_map);
}

bool Movie_config::input_size_for_movie_config(const Movie_config& config, Rect_size& return_size)
{
  bool return_value = true;

  if (frame_format_original == config.frame_format)
  {
    return_size = config.original_size;
  }
  else
  {
    map<Frame_format, Rect_size>::const_iterator r = m_frame_format_input_map.find(config.frame_format);
    if (r != m_frame_format_input_map.end())
    {
      return_size = r->second;
    }
    else
    {
      return_value = false;
    }
  }

  return return_value;
}

bool Movie_config::output_size_for_movie_config(const Movie_config& config, Rect_size& return_size)
{
  bool return_value = true;;

  if (frame_format_original == config.frame_format)
  {
    return_size = config.original_size;
    // bound dimensions should be even for best compatibility
    if (return_size.width % 2)
    {
      ++return_size.width;
    }
    if (return_size.height % 2)
    {
      ++return_size.height;
    }
  }
  else
  {
    map<Frame_format, Rect_size>::const_iterator r = m_frame_format_output_map.find(config.frame_format);
    if (r != m_frame_format_output_map.end())
    {
      return_size = r->second;
    }
    else
    {
      return_value = false;
    }
  }

  return return_value;
}

//
// Struct constants
//
const Movie_config::Movie_preset Movie_config::movie_preset_qtrle;
//   frame rate
const Movie_config::Frame_rate_type Movie_config::frame_rate_5_Hz;
const Movie_config::Frame_rate_type Movie_config::frame_rate_10_Hz;
const Movie_config::Frame_rate_type Movie_config::frame_rate_20_Hz;
const Movie_config::Frame_rate_type Movie_config::frame_rate_24_Hz;
const Movie_config::Frame_rate_type Movie_config::frame_rate_25_Hz;
const Movie_config::Frame_rate_type Movie_config::frame_rate_29_97_Hz_exact;
const Movie_config::Frame_rate_type Movie_config::frame_rate_29_97_Hz_approx;
const Movie_config::Frame_rate_type Movie_config::frame_rate_30_Hz;
const Movie_config::Frame_rate_type Movie_config::frame_rate_50_Hz;
const Movie_config::Frame_rate_type Movie_config::frame_rate_60_Hz;
//   frame size
const Movie_config::Frame_format Movie_config::frame_format_601_ntsc_4_to_3;
const Movie_config::Frame_format Movie_config::frame_format_601_ntsc_16_to_9_anamorphic;
const Movie_config::Frame_format Movie_config::frame_format_dv_ntsc_4_to_3;
const Movie_config::Frame_format Movie_config::frame_format_dv_ntsc_16_to_9_anamorphic;
const Movie_config::Frame_format Movie_config::frame_format_601_dv_pal_4_to_3;
const Movie_config::Frame_format Movie_config::frame_format_601_dv_pal_16_to_9_anamorphic;
const Movie_config::Frame_format Movie_config::frame_format_hd_720p;
const Movie_config::Frame_format Movie_config::frame_format_hd_1080i_p;
const Movie_config::Frame_format Movie_config::frame_format_original;
//   default constants
const Movie_config::Movie_preset Movie_config::default_movie_preset = Movie_config::movie_preset_qtrle;
const Movie_config::Frame_rate_type Movie_config::default_frame_rate_type = Movie_config::frame_rate_30_Hz;
const Movie_config::Frame_format Movie_config::default_frame_format = Movie_config::frame_format_601_ntsc_4_to_3;

//
// Private static class members
//

bool Movie_config::m_maps_setup;

Movie_config::Constant_map Movie_config::m_movie_preset_map;
Movie_config::Constant_map Movie_config::m_frame_rate_type_map;
Movie_config::Constant_map Movie_config::m_frame_format_map;
map<Movie_config::Movie_preset, string> Movie_config::m_movie_preset_extension_map;
map<Movie_config::Frame_rate_type, Frame_rate> Movie_config::m_frame_rate_map;
map<Movie_config::Frame_format, Rect_size> Movie_config::m_frame_format_input_map;
map<Movie_config::Frame_format, Rect_size> Movie_config::m_frame_format_output_map;

//
// Private member functions
//

void Movie_config::mf_setup_maps()
{
  // movie preset
  m_movie_preset_map[movie_preset_qtrle] = string("qtrle");
  // frame rate type
  m_frame_rate_type_map[frame_rate_5_Hz]            = string("5_hz");
  m_frame_rate_type_map[frame_rate_10_Hz]           = string("10_hz");
  m_frame_rate_type_map[frame_rate_20_Hz]           = string("20_hz");
  m_frame_rate_type_map[frame_rate_24_Hz]           = string("24_hz");
  m_frame_rate_type_map[frame_rate_25_Hz]           = string("25_hz");
  m_frame_rate_type_map[frame_rate_29_97_Hz_exact]  = string("29_97_hz_exact");
  m_frame_rate_type_map[frame_rate_29_97_Hz_approx] = string("29_97_hz");
  m_frame_rate_type_map[frame_rate_30_Hz]           = string("30_hz");
  m_frame_rate_type_map[frame_rate_50_Hz]           = string("50_hz");
  m_frame_rate_type_map[frame_rate_60_Hz]           = string("60_hz");
  // frame format type
  m_frame_format_map[frame_format_601_ntsc_4_to_3]               = string("601_ntsc_4:3");
  m_frame_format_map[frame_format_601_ntsc_16_to_9_anamorphic]   = string("601_ntsc_16:9");
  m_frame_format_map[frame_format_dv_ntsc_4_to_3]                = string("dv_ntsc_4:3");
  m_frame_format_map[frame_format_dv_ntsc_16_to_9_anamorphic]    = string("dv_ntsc_16:9_anamorphic");
  m_frame_format_map[frame_format_601_dv_pal_4_to_3]             = string("601_dv_pal_4:3");
  m_frame_format_map[frame_format_601_dv_pal_16_to_9_anamorphic] = string("601_dv_pal_16:9");
  m_frame_format_map[frame_format_hd_720p]                       = string("hd_720p");
  m_frame_format_map[frame_format_hd_1080i_p]                    = string("hd_1080");
  m_frame_format_map[frame_format_original]                      = string("original");
  // movie preset extension
  m_movie_preset_extension_map[movie_preset_qtrle] = string("mov");
  // frame rate
  m_frame_rate_map[frame_rate_5_Hz]            = Frame_rate(  500,  100); // compatible with 48 KHz audio
  m_frame_rate_map[frame_rate_10_Hz]           = Frame_rate( 1000,  100); // compatible with 48 KHz audio
  m_frame_rate_map[frame_rate_20_Hz]           = Frame_rate( 2000,  100); // compatible with 48 KHz audio
  m_frame_rate_map[frame_rate_24_Hz]           = Frame_rate( 2400,  100); // compatible with 48 KHz audio
  m_frame_rate_map[frame_rate_25_Hz]           = Frame_rate(  600,   24); // compatible with 48 KHz audio
  m_frame_rate_map[frame_rate_29_97_Hz_exact]  = Frame_rate(30000, 1001);
  m_frame_rate_map[frame_rate_29_97_Hz_approx] = Frame_rate( 2997,  100); // compatible with 47952Hz audio
  m_frame_rate_map[frame_rate_30_Hz]           = Frame_rate( 3000,  100); // compatible with 48 KHz audio
  m_frame_rate_map[frame_rate_50_Hz]           = Frame_rate(  600,   12); // compatible with 48 KHz audio
  m_frame_rate_map[frame_rate_60_Hz]           = Frame_rate( 6000,  100); // compatible with 48 KHz audio
  // frame format
  //   input size
  m_frame_format_input_map[frame_format_601_ntsc_4_to_3]               = Rect_size( 720,  547);
  m_frame_format_input_map[frame_format_601_ntsc_16_to_9_anamorphic]   = Rect_size( 853,  486);
  m_frame_format_input_map[frame_format_dv_ntsc_4_to_3]                = Rect_size( 720,  540);
  m_frame_format_input_map[frame_format_dv_ntsc_16_to_9_anamorphic]    = Rect_size( 853,  480);
  m_frame_format_input_map[frame_format_601_dv_pal_4_to_3]             = Rect_size( 768,  576);
  m_frame_format_input_map[frame_format_601_dv_pal_16_to_9_anamorphic] = Rect_size(1024,  576);
  m_frame_format_input_map[frame_format_hd_720p]                       = Rect_size(1280,  720);
  m_frame_format_input_map[frame_format_hd_1080i_p]                    = Rect_size(1920, 1080);
  m_frame_format_input_map[frame_format_original]                      = Rect_size(   0,    0);
  //   output size
  m_frame_format_output_map[frame_format_601_ntsc_4_to_3]               = Rect_size( 720,  486);
  m_frame_format_output_map[frame_format_601_ntsc_16_to_9_anamorphic]   = Rect_size( 720,  486);
  m_frame_format_output_map[frame_format_dv_ntsc_4_to_3]                = Rect_size( 720,  480);
  m_frame_format_output_map[frame_format_dv_ntsc_16_to_9_anamorphic]    = Rect_size( 720,  480);
  m_frame_format_output_map[frame_format_601_dv_pal_4_to_3]             = Rect_size( 720,  576);
  m_frame_format_output_map[frame_format_601_dv_pal_16_to_9_anamorphic] = Rect_size( 720,  576);
  m_frame_format_output_map[frame_format_hd_720p]                       = Rect_size(1280,  720);
  m_frame_format_output_map[frame_format_hd_1080i_p]                    = Rect_size(1920, 1080);
  m_frame_format_output_map[frame_format_original]                      = Rect_size(   0,    0);
}
