// Video_track.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef VIDEO_TRACK_HPP_
#define VIDEO_TRACK_HPP_

#include "common.hpp"
#include "Image_types.hpp"
#include "Movie_types.hpp"

struct AVFormatContext;
struct AVStream;
struct AVCodecContext;
struct AVFrame;
struct SwsContext;

namespace Roan_trail
{
  class Error;
  class Error_param;
  struct Rect_size;

  namespace Builder
  {
    class Video_track_config;
    typedef int Movie_preset_type;

    class Video_track
    {
    public:
      // constructor/destructor/initialization
      Video_track(int track_ID,
                  const Frame_rate& frame_rate,
                  const Rect_size& output_frame_size,
                  Movie_preset_type movie_preset,
                  int keyframe_period);
      virtual ~Video_track();
      bool setup(AVFormatContext* format_context, Error_param& return_error);
      // accessors
      const Frame_rate& frame_rate() const { return m_frame_rate; }
      AVStream* stream() { return m_stream; }
      // control
      virtual bool start(Error_param& return_error);
      virtual bool add(const uint8_t* raw_frame,
                       Long_int frame_count,
                       Error_param& return_error);
      virtual bool end(Error_param& return_error);
      // helpers
      Long_int estimate_frame_size(const uint8_t* raw_frame);
      Long_int estimate_segment_size(Long_int frame_size, Long_int frame_count);
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      //
      bool m_setup;
      bool m_started;
      //
      int m_track_ID;
      Frame_rate m_frame_rate;
      Rect_size m_output_frame_size;
      Movie_preset_type m_movie_preset;
      Long_int m_current_track_time;
      // ffmpeg
      //   format reference
      AVFormatContext* m_format_context;
      //   stream
      AVStream* m_stream;
      //   codec
      AVCodecContext* m_codec_context;
      //   frame
      AVFrame* m_output_frame;
      //   output buffer
      uint8_t* m_output_buffer;
      Long_int m_output_buffer_size;
      // keyframe
      int m_keyframe_period;
      //
      bool mf_compressed_size_of_pixels(const uint8_t* raw_frame,
                                        Long_int& return_compressed_size);
      void mf_encode_raw_frame(const uint8_t* raw_frame, int& return_output_size);
      void mf_setup_codec_context_from_preset(Movie_preset_type preset,
                                              AVCodecContext& return_video_codec_context);
      void mf_cleanup();
      void mf_fill_in_output_frame(const uint8_t* raw_frame);
      // prevent compiler from generating
      Video_track(const Video_track& m);
      Video_track& operator=(const Video_track& m);
    };
  }
}

#endif // VIDEO_TRACK_HPP_
