// Kinetophone_builder_app.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// include *mm first to avoid conflicts
#include <gtkmm/main.h>
#include <gdkmm/pixbuf.h>

#include "Kinetophone_builder_app.hpp"
#include "Kinetophone_builder_config.hpp"
#include "Kinetophone_builder_model.hpp"
#include "Kinetophone_build_manager.hpp"
#include "Kinetophone_builder_view.hpp"
#include "../base/Logger.hpp"
#include "../base/Movie_builder_config.hpp"
#include "../base/Movie_config.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/Sound_recorder_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/Segment.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <memory>
#include <iostream>
#include <sstream>
#include <sigc++/sigc++.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef PACKAGE_VERSION
#define PACKAGE_VERSION "(unknown)"
#endif

using std::stringstream;
using std::cerr;
using std::endl;
using sigc::pointer_functor1;
using sigc::ptr_fun;
using Gdk::Pixbuf;
using Gtk::Main;
using Roan_trail::Logger;
using Roan_trail::Builder::Movie_builder_config;
using Roan_trail::Builder::Movie_config;
using Roan_trail::Recorder::Segment;
using Roan_trail::Recorder::Sound_recorder_config;
using Roan_trail::Recorder::Sound_file_config;
using Roan_trail::Source::Slide_collection;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//

namespace
{
  void ih_glib_exception_handler()
  {
    try
    {
      throw;
    }
    catch (const Glib::Exception& e)
    {
      cerr << "Unhandled exception: " << e.what() << endl;
      cerr << "Terminating." << endl;
    }
    catch (const std::exception& e)
    {
      cerr << "Unhandled exception: " << e.what() << endl;
      cerr << "Terminating." << endl;
    }
    // anything else goes to the function set in std::set_terminate()
  }
}

//
// Constructor/destructor
//

Kinetophone_builder_app::Kinetophone_builder_app(int argc, const char** argv)
  : Application(argc, argv),
    m_config(new Kinetophone_builder_config),
    m_model(new Kinetophone_builder_model("directory")),
    m_build_manager(new Kinetophone_build_manager(*m_config, *m_model))
{
  precondition(argv);

  postcondition(mf_invariant(false));
}

Kinetophone_builder_app::~Kinetophone_builder_app()
{
  precondition(mf_invariant(false));

  delete m_build_manager;
  delete m_model;
  delete m_config;
}

int Kinetophone_builder_app::run()
{
  precondition(mf_invariant());

  int return_value = 1;
  Error_param error;
  int user_error_code = 0;

  start_error_block();

  // see what the user wants to do
  int builder_command = m_config->parse_program_options(argc(), argv(), error);
  on_error(Kinetophone_builder_config::command_error == builder_command,
           new Kinetophone_error(error_location(),
                                 Kinetophone_error::command_line,
                                 error()));

  // update the installation directory if the program options override the default one
  if ("" != m_config->custom_installation_dir)
  {
    Application::application()->set_installation_dir(m_config->custom_installation_dir);
  }

  switch (builder_command)
  {
  case Kinetophone_builder_config::command_help:
    // HELP
    Kinetophone_builder_view::output_message(m_config->help_message);
    break;
  case Kinetophone_builder_config::command_output_version:
    // VERSION
    {
      stringstream version;
      version << "Kinetophone Builder Version " << string(PACKAGE_VERSION) << endl;
      Kinetophone_builder_view::output_message(version.str());
    }
    break;
  case Kinetophone_builder_config::command_build:
    // BUILD
    {
      Error_param error;
      bool build_success = mf_build(error);
      on_error(!build_success, new Kinetophone_error(error_location(),
                                                     Kinetophone_error::general,
                                                     error()));
    }
    break;
  default:
    assert(0 && "Invalid command");
    break;
  }

  return_value =  0;
  goto exit_point;

  end_error_block();

 error_handler:
  {
    string error_message;
    user_error_code = handler_error->format_message_for_chain(m_config->detailed_error, error_message);
    Kinetophone_builder_view::output_error(error_message);
  }
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_value = user_error_code;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
  return return_value;
}

void Kinetophone_builder_app::terminate(int code, const string& message)
{
  precondition(mf_invariant());

  m_build_manager->cancel();

  Application::terminate(code, message);

  // no postcondition, above terminate() call does not return
}

//
// Protected member functions
//

bool Kinetophone_builder_app::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  // object allocation checking
  if (!m_config
      || !m_model
      || !m_build_manager)
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions
//

bool Kinetophone_builder_app::mf_build(Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  Error_param error;

  start_error_block();

  Logger& logger = Logger::default_logger();
  if (m_config->movie_builder_config->quiet)
  {
    logger.set_output_level(Logger::quiet);
  }
  else if (m_config->movie_builder_config->verbose)
  {
    logger.set_output_level(Logger::verbose);
  }
  else
  {
    logger.set_output_level(Logger::normal);
  }

  logger << Logger::info << "Loading session file: " << m_config->session_file_path << endl;

  const bool imported = m_model->load_from_XML(m_config->session_file_path, error);
  on_error(!imported, new Kinetophone_error(error_location(),
                                            Kinetophone_error::general,
                                            error()));

  // update config sound file (from session file) if not overridden on command line
  if ("" == m_config->movie_builder_config->movie->sound->file_name)
  {
    m_config->movie_builder_config->movie->sound->file_name = m_model->audio_recording_path();
  }
  // update original size
  if (Movie_config::frame_format_original == m_config->movie_builder_config->movie->frame_format)
  {
    logger << Logger::info;
    logger << "Calculating maximum bounds of source images" << endl;
    Rect_size max_bounds = m_model->slides()->max_bounds()
      * m_config->movie_builder_config->movie_attributes.full_resolution_scale;
    m_config->movie_builder_config->movie->original_size = max_bounds;
    logger << "  bounds: " << max_bounds.width << " X " << max_bounds.height << endl;
  }

  Kinetophone_builder_view view;

  const bool setup_success = m_build_manager->setup_for_build(error);
  on_error(!setup_success, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::general,
                                                 error()));

  if (!m_build_manager->prebuild_checks(error))
  {
    logger << Logger::warning;
    logger << "Not performing build after prebuild warning(s)" << endl;
    return_value = true;
    goto exit_point;
  }

  // TODO: check this:
  // Note: minimize heap allocation after this point, should be limited to error handling
  //   and library functions (Ref. Standards Rule #206)

  const bool build_success = m_build_manager->build(error);
  on_error(!build_success, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::general,
                                                 error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error, return_value, false);

 exit_point:
  return return_value;
}

//////////
// main //
//////////

int main (int argc, const char **argv)
{
  int argc_arg = argc;
  char** argv_arg = const_cast<char**>(argv);

  Kinetophone_builder_app application(argc, argv);

  Main gtk_application(argc_arg, argv_arg);
  Glib::add_exception_handler(&ih_glib_exception_handler);

  int return_code = application.run();

  return return_code;
}
