// Kinetophone_narrator_controller.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef KINETOPHONE_NARRATOR_CONTROLLER_HPP_
#define KINETOPHONE_NARRATOR_CONTROLLER_HPP_

#include "../base/common.hpp"
#include <string>
#include <vector>
#include <sigc++/sigc++.h>

using std::string;
using std::vector;
using sigc::connection;

namespace Roan_trail
{
  class Error;
  class Error_param;

  namespace Kinetophone
  {
    class Kinetophone_narrator_model;
    struct Kinetophone_narrator_config;

    class Kinetophone_narrator_controller
    {
    public:
      // constructor/destructor/copy
      Kinetophone_narrator_controller(Kinetophone_narrator_model& model, Kinetophone_narrator_config& view);
      void setup();
      // accessors
      //   signals
      //     manual model update signals
      sigc::signal<void> signal_model_slides_will_update() { return m_signal_model_slides_will_update; }
      sigc::signal<void> signal_model_recorder_will_update() { return m_signal_model_recorder_will_update; }
      sigc::signal<void> signal_model_slides_updated() { return m_signal_model_slides_updated; }
      sigc::signal<void> signal_model_recorder_updated() { return m_signal_model_recorder_updated; }
      //     polled updates
      sigc::signal<void, bool> signal_update_time() { return m_signal_update_time; }
      sigc::signal<void> signal_update_levels() { return m_signal_update_levels; }
      sigc::signal<void> signal_update_available_space() { return m_signal_update_available_space; }
      sigc::signal<void> signal_update_statusbar() { return m_signal_update_statusbar; }
      //     UI
      sigc::signal<void> signal_toggle_metering() { return m_signal_toggle_metering; }
      sigc::signal<bool> signal_overwrite_recording() { return m_signal_overwrite_recording; }
      sigc::signal<void, const string&> signal_session_info() { return m_signal_session_info; }
      sigc::signal<int, const Error&, bool> signal_error() { return m_signal_error; }
      // slots
      void retake();
      void previous();
      void slide(int slide_index);
      void next();
      void thumbnail_choose(int slide_index);
      void mute();
      void pause();
      void record();
      void toggle_metering();
      void notes_changed();
      void notes_editing_done(const string &curent_notes);
      void export_session(const string& session_file);
      void quit();
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      Kinetophone_narrator_model& m_model;
      Kinetophone_narrator_config& m_config;
      int m_update_rate;
      Long_int m_update_tick_count;
      int m_update_period_for_record;
      int m_time_update_period_for_pause;
      int m_flash_period_for_pause;
      int m_space_update_period_for_nonrecord;
      bool m_setup;
      // signals
      //   manual model update signals
      sigc::signal<void> m_signal_model_slides_will_update;
      sigc::signal<void> m_signal_model_recorder_will_update;
      sigc::signal<void> m_signal_model_slides_updated;
      sigc::signal<void> m_signal_model_recorder_updated;
      //   polled updates
      sigc::signal<void, bool> m_signal_update_time;
      sigc::signal<void> m_signal_update_levels;
      sigc::signal<void> m_signal_update_available_space;
      sigc::signal<void> m_signal_update_statusbar;
      //   UI
      sigc::signal<void> m_signal_toggle_metering;
      sigc::signal<bool> m_signal_overwrite_recording;
      sigc::signal<void, const string&> m_signal_session_info;
      sigc::signal<int, const Error&, bool> m_signal_error;
      // connections
      connection m_update_tick_connection;
      //
      static Kinetophone_narrator_controller* m_instance; // singleton by convention
      // member functions
      // update
      void mf_initial_update();
      bool mf_update_tick();
      void mf_update_available_space(Long_int& return_bytes_available);
      // prevent compiler from generating
      Kinetophone_narrator_controller(const Kinetophone_narrator_controller& c);
      Kinetophone_narrator_controller& operator=(const Kinetophone_narrator_controller& c);
    };
  }
}

#endif // KINETOPHONE_NARRATOR_CONTROLLER_HPP_
