// Level_meter_widget.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// include *mm first to avoid conflicts
#include <gdkmm/drawable.h>
#include <gdkmm/general.h>

#include "Level_meter_widget.hpp"
#include "../base/common.hpp"
#include "../base/Level_meter_view.hpp"
#include "../base/error/Error.hpp"
#include <iostream>
#include <cstring>

using Gdk::Window;
using Glib::RefPtr;
using Gtk::Allocation;
using Gtk::Requisition;
using Gtk::Button;
using Roan_trail::Error_param;
using namespace Gdk::Cairo;
using namespace Roan_trail::Kinetophone;

namespace
{
  const double ic_minimum_displayable_peak = 0.001;
  const int ic_default_bar_length = 120;
  const int ic_default_bar_thickness = 10;
  const int ic_default_clip_indicator_size = 6;
  const int ic_default_spacing = 2;
  const int ic_default_border_width = 1;
  const double ic_warn_level = Level_meter_widget::scale_IEC(-9.0); // -9 dBFS is the warning level (yellow)
  const double ic_critical_level = Level_meter_widget::scale_IEC(-3.0); // -3 dBFs is the critical level (red)
  Error_param iv_ignore_error(false);
  bool ih_is_valid_rect(const Rectangle& rect)
  {
    return ((rect.get_x() >= 0)
            && (rect.get_y() >= 0)
            && (rect.get_width() >= 0)
            && (rect.get_height() >= 0));
  }
}

// Note: The GType name will be gtkmm__CustomObject_Level_meter_widget
//
// Note: This shows the GType name, which must be used in the RC file:
//   std::cout << "GType name: " << G_OBJECT_TYPE_NAME(gobj()) << std::endl;
//
// Note: This show that the GType still derives from GtkWidget:
//   std::cout << "Gtype is a GtkWidget?:" << GTK_IS_WIDGET(gobj()) << std::endl;

Level_meter_widget::Level_meter_widget(int channels,
                                       int rate,
                                       bool is_vertical,
                                       bool has_clip_indicator,
                                       bool has_peak,
                                       bool model_ballistics,
                                       int attack_period,
                                       int decay_period,
                                       int peak_hold_period,
                                       int bar_thickness,
                                       int bar_length,
                                       int clip_indicator_size,
                                       int spacing,
                                       int border_width)
  : ObjectBase("Level_meter_widget"),
    Level_meter_view(channels,
                     rate,
                     false,
                     is_vertical,
                     model_ballistics,
                     (attack_period < 1 ? Level_meter_view::default_attack_period : attack_period),
                     (decay_period < 1 ? Level_meter_view::default_decay_period : decay_period),
                     (peak_hold_period < 1 ? Level_meter_view::default_peak_hold_period : peak_hold_period)),
    Button(),
    m_ref_window(0),
    m_enabled(true),
    m_show_warning(),
    m_show_critical(),
    m_show_peak(),
    m_bar_background_rect(),
    m_normal_background_rect(),
    m_warning_background_rect(),
    m_critical_background_rect(),
    m_clip_indicator_background_rect(),
    m_normal_rect(),
    m_warning_rect(),
    m_critical_rect(),
    m_peak_rect(),
    m_peak_color(),
    m_has_clip_indicator(has_clip_indicator),
    m_has_peak(has_peak),
    m_bar_thickness(bar_thickness < 1 ? ic_default_bar_thickness : bar_thickness),
    m_bar_length(bar_length < 1 ? ic_default_bar_length : bar_length),
    m_clip_indicator_size(clip_indicator_size < 1 ? ic_default_clip_indicator_size : clip_indicator_size),
    m_spacing(spacing < 0 ? ic_default_spacing : spacing),
    m_border_width(border_width < 0 ? ic_default_border_width : border_width),
    m_widget_sized(false)
{
  precondition((channels > 0)
               && (rate > 0));

  set_has_window(false);
  set_focus_on_click(false);

  if (!m_have_colors)
  {
    // setup singleton colors
    m_have_colors = true;
    mf_setup_colors();
  }

  mf_resize_vectors();

  postcondition(mf_invariant(false));
}

Level_meter_widget::~Level_meter_widget()
{
  precondition(mf_invariant(false));
}

//
// View update
//

bool Level_meter_widget::update(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  static_cast<void>(return_error); // avoid unused warning

  Level_meter_view::update();

  mf_update_meters();

  queue_draw(); // invalidate entire area of widget, triggering an expose event to redraw

  postcondition(!return_error()
                && mf_invariant());
  return true;
}

//
// Mutators
//

void Level_meter_widget::set_levels(const vector<double>& levels)
{
  precondition(mf_invariant());

  Level_meter_view::set_levels(levels);

  update(iv_ignore_error); // ignore error

  postcondition(mf_invariant());
}

//
// Protected member functions
//

bool Level_meter_widget::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  {
    if (get_realized() && !m_ref_window)
    {
      goto exit_point;
    }

    // check attributes
    if ((m_bar_thickness < 1)
        || (m_bar_length < 1)
        || (m_clip_indicator_size < 0)
        || (m_spacing < 0)
        || (m_border_width < 0))
    {
      goto exit_point;
    }
    if (m_has_clip_indicator && (m_clip_indicator_size < 1))
    {
      goto exit_point;
    }
    const int channels = Level_meter_view::channels();
    // check vector sizes
    if (// update logic
        (m_enabled.size() != static_cast<size_t>(channels))
        || (m_show_warning.size() != static_cast<size_t>(channels))
        || (m_show_critical.size() != static_cast<size_t>(channels))
        || ((m_has_peak && (m_show_peak.size() != static_cast<size_t>(channels))) || !m_has_peak)
        // background rects
        || (m_bar_background_rect.size() != static_cast<size_t>(channels))
        || (m_normal_background_rect.size() != static_cast<size_t>(channels))
        || (m_warning_background_rect.size() != static_cast<size_t>(channels))
        || (m_critical_background_rect.size() != static_cast<size_t>(channels))
        || ((m_has_clip_indicator && (m_clip_indicator_background_rect.size() != static_cast<size_t>(channels)))
            || !m_has_clip_indicator)
        // rects
        || (m_normal_rect.size() != static_cast<size_t>(channels))
        || (m_warning_rect.size() != static_cast<size_t>(channels))
        || (m_critical_rect.size() != static_cast<size_t>(channels))
        || (m_peak_rect.size() != static_cast<size_t>(channels))
        // colors
        || (m_peak_color.size() != static_cast<size_t>(channels)))
    {
      goto exit_point;
    }

    // check rectangle validity
    for (size_t i = 0; i < static_cast<size_t>(channels); ++i)
    {
      if (!ih_is_valid_rect(m_bar_background_rect[i])
          ||!ih_is_valid_rect(m_normal_background_rect[i])
          || !ih_is_valid_rect(m_warning_background_rect[i])
          || !ih_is_valid_rect(m_critical_background_rect[i])
          || !ih_is_valid_rect(m_clip_indicator_background_rect[i])
          || !ih_is_valid_rect(m_normal_rect[i])
          || !ih_is_valid_rect(m_warning_rect[i])
          || !ih_is_valid_rect(m_critical_rect[i])
          || !ih_is_valid_rect(m_peak_rect[i]))
      {
        goto exit_point;
      }
    }
  }
  return_value = !check_base_class || Level_meter_view::mf_invariant(check_base_class);

 exit_point:
  return return_value;
}

//
// Signal handlers
//

void Level_meter_widget::on_size_request(Requisition* return_requisition)
{
  *return_requisition = Requisition();

  // set minimum space requested
  int height;
  int width;
  mf_calc_widget_size(width, height);
  return_requisition->width = width;
  return_requisition->height = height;
}

void Level_meter_widget::on_size_allocate(Allocation& allocation)
{
  // we will get height/width >= what was requested, use what is given
  set_allocation(allocation);

  Rectangle window_rect(allocation.get_x(),
                        allocation.get_y(),
                        allocation.get_width(),
                        allocation.get_height());
  if (m_ref_window)
  {
    m_ref_window->move_resize(window_rect.get_x(),
                              window_rect.get_y(),
                              window_rect.get_width(),
                              window_rect.get_height());
  }

  mf_calc_fixed_rects(window_rect);
  mf_update_meters();

  m_widget_sized = true;
}

void Level_meter_widget::on_map()
{
  Widget::on_map();
}

void Level_meter_widget::on_unmap()
{
  Widget::on_unmap();
}

void Level_meter_widget::on_realize()
{
  Widget::on_realize();

  ensure_style();

  if(!m_ref_window)
  {
    GdkWindowAttr attributes;
    memset(&attributes, 0, sizeof(attributes));

    Allocation allocation = get_allocation();

    // set initial position and size of the Gdk::Window:
    attributes.x = allocation.get_x();
    attributes.y = allocation.get_y();
    attributes.width = allocation.get_width();
    attributes.height = allocation.get_height();

    attributes.event_mask = get_events () | Gdk::EXPOSURE_MASK;
    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.wclass = GDK_INPUT_OUTPUT;

    RefPtr<Window> parent_window = get_window();
    m_ref_window = Window::create(parent_window,
                                  &attributes,
                                  GDK_WA_X | GDK_WA_Y);
    set_has_window();
    set_window(m_ref_window);
    m_ref_window->reference(); // Note: causes a leak, but can be ignored

    // setup the widget to receive expose events
    m_ref_window->set_user_data(gobj());
  }

}

void Level_meter_widget::on_unrealize()
{
  m_ref_window.clear();

  Widget::on_unrealize();
}

bool Level_meter_widget::on_expose_event(GdkEventExpose* event)
{
  {
    if (!m_ref_window)
    {
      goto exit_point;
    }

    Cairo::RefPtr<Cairo::Context> context = m_ref_window->create_cairo_context();

    if (event)
    {
      // clip to the area that needs to be re-exposed so we don't draw any
      // more than we need to.
      context->rectangle(event->area.x,
                         event->area.y,
                         event->area.width,
                         event->area.height);
      context->clip();
    }

    // paint the background
    set_source_color(context, get_parent()->get_style()->get_bg(Gtk::STATE_NORMAL));
    context->paint();

    // draw the meters from what was calculated when the meter was updated
    size_t number_channels = static_cast<size_t>(channels());
    for (size_t channel = 0; channel < number_channels; ++channel)
    {
      // draw the meters
      if (!is_sensitive() || !m_enabled[channel])
      {
        // draw disabled
        set_source_color(context, m_disabled_color);
        add_rectangle_to_path(context, m_normal_background_rect[channel]);
        add_rectangle_to_path(context, m_warning_background_rect[channel]);
        add_rectangle_to_path(context, m_critical_background_rect[channel]);
        if (m_has_clip_indicator)
        {
          add_rectangle_to_path(context, m_clip_indicator_background_rect[channel]);
        }
        context->fill();
      }
      else
      {
        // dimmed background fill
        set_source_color(context, m_background_normal_color);
        add_rectangle_to_path(context, m_normal_background_rect[channel]);
        context->fill();
        set_source_color(context, m_background_warning_color);
        add_rectangle_to_path(context, m_warning_background_rect[channel]);
        context->fill();
        set_source_color(context, m_background_critical_color);
        add_rectangle_to_path(context, m_critical_background_rect[channel]);
        if (m_has_clip_indicator)
        {
          add_rectangle_to_path(context, m_clip_indicator_background_rect[channel]);
        }
        context->fill();

        // draw bar
        // normal part first
        set_source_color(context, m_normal_color);
        add_rectangle_to_path(context, m_normal_rect[channel]);
        context->fill();
        // warning part, if any
        if (m_show_warning[channel])
        {
          add_rectangle_to_path(context, m_warning_rect[channel]);
          set_source_color(context, m_warning_color);
          context->fill();
          // critical part, if any
          if (m_show_critical[channel])
          {
            set_source_color(context, m_critical_color);
            add_rectangle_to_path(context, m_critical_rect[channel]);
            context->fill();
          }
        }
        // clip indicator
        if (m_has_clip_indicator && clipped(channel))
        {
          set_source_color(context, m_critical_color);
          add_rectangle_to_path(context, m_clip_indicator_background_rect[channel]);
          context->fill();
        }
        // peak
        if (m_has_peak && m_show_peak[channel])
        {
          set_source_color(context, m_peak_color[channel]);
          add_rectangle_to_path(context, m_peak_rect[channel]);
          context->fill();
        }
      }
    }
  }

exit_point:
  return true;
}

void Level_meter_widget::on_clicked()
{
  reset_clipped();

  update(iv_ignore_error); // ignore error
}

//
// Private class data members
//

bool Level_meter_widget::m_have_colors = false;
Color Level_meter_widget::m_normal_color;
Color Level_meter_widget::m_warning_color;
Color Level_meter_widget::m_critical_color;
Color Level_meter_widget::m_background_normal_color;
Color Level_meter_widget::m_background_warning_color;
Color Level_meter_widget::m_background_critical_color;
Color Level_meter_widget::m_disabled_color;

//
// Private member functions
//

void Level_meter_widget::mf_resize_vectors()
{
  const int channels = Level_meter_view::channels();
  // logic
  m_enabled.resize(channels);
  m_show_warning.resize(channels);
  m_show_critical.resize(channels);
  if (m_has_peak)
  {
    m_show_peak.resize(channels);
  }
  // rectangles
  m_bar_background_rect.resize(channels);
  m_normal_background_rect.resize(channels);
  m_warning_background_rect.resize(channels);
  m_critical_background_rect.resize(channels);
  if (m_has_clip_indicator)
  {
    m_clip_indicator_background_rect.resize(channels);
  }
  m_normal_rect.resize(channels);
  m_warning_rect.resize(channels);
  m_critical_rect.resize(channels);
  m_peak_rect.resize(channels);
  // colors
  m_peak_color.resize(channels);

  // initialize
  for (size_t channel = 0; channel < static_cast<size_t>(channels); ++channel)
  {
    m_enabled[channel] = true;
  }
}

// calculate the tight bounds that the widget needs
void Level_meter_widget::mf_calc_widget_size(int& return_width, int& return_height)
{
  const int channels = Level_meter_view::channels();
  if (vertical())
  {
    return_height = m_bar_length
      + (m_has_clip_indicator ? (m_clip_indicator_size + m_spacing) : 0)
      + m_border_width * 2;
    return_width = m_bar_thickness * channels
      + m_border_width * 2
      + m_spacing * (channels - 1);
  }
  else
  {
    return_height = m_bar_thickness * channels
      + m_border_width * 2
      + m_spacing * (channels - 1);
    return_width = m_bar_length
      + (m_has_clip_indicator ? (m_clip_indicator_size + m_spacing) : 0)
      + m_border_width * 2;
  }
}

// calculate rectangles that are fixed between widget resize events
void Level_meter_widget::mf_calc_fixed_rects(const Rectangle& rect)
{
  int widget_width;
  int widget_height;
  mf_calc_widget_size(widget_width, widget_height);
  const int extra_x = (rect.get_width() > widget_width) ? (rect.get_width() - widget_width) : 0;
  const int extra_y = (rect.get_height() > widget_height) ? (rect.get_height() - widget_height) : 0;
  const int extra_x_padding = extra_x / 2;
  const int extra_y_padding = extra_y / 2;
  const int bar_length = m_bar_length;
  const int clip_indicator_size = m_has_clip_indicator ? (m_clip_indicator_size + m_spacing) : 0;
  const bool vertical = Level_meter_view::vertical();

  // align the base of the meters
  // and center the group of meters in the direction perpendicular to travel
  int current_bar_origin_x = m_border_width + (vertical ? extra_x_padding : 0);
  int current_bar_origin_y = m_border_width + (vertical ? (extra_y + clip_indicator_size)
                                                 : extra_y_padding);

  const size_t number_channels = static_cast<size_t>(channels());
  for (size_t channel = 0; channel < number_channels; ++channel)
  {
    // the bar rectangle is the bounds for the main bar of the
    // meter (excluding padding, clip indicator, etc.)
    Rectangle& bar_background_rect = m_bar_background_rect[channel];
    bar_background_rect = Rectangle(current_bar_origin_x,
                                    current_bar_origin_y,
                                    (vertical ? m_bar_thickness : bar_length),
                                    (vertical ? bar_length : m_bar_thickness));
    // assign a "working" rectangle to the parts
    Rectangle& normal_background_rect = m_normal_background_rect[channel];
    normal_background_rect = bar_background_rect;
    Rectangle& warning_background_rect = m_warning_background_rect[channel];
    warning_background_rect = bar_background_rect;
    Rectangle& critical_background_rect = m_critical_background_rect[channel];
    critical_background_rect = bar_background_rect;
    // adjust the rectangles of the parts
    if (vertical)
    {
      // normal background
      const int normal_height = round(bar_length * ic_warn_level);
      normal_background_rect.set_height(normal_height);
      normal_background_rect.set_y(normal_background_rect.get_y() + (bar_length - normal_height));
      // warning background
      const int warning_height = round(bar_length * (ic_critical_level - ic_warn_level));
      warning_background_rect.set_height(warning_height);
      warning_background_rect.set_y(normal_background_rect.get_y() - warning_height);
      // critical background
      const int critical_height = round(bar_length * (1.0 - ic_critical_level));
      critical_background_rect.set_height(critical_height);
      critical_background_rect.set_y(warning_background_rect.get_y() - critical_height);
      // clip indicator background
      if (m_has_clip_indicator)
      {
        Rectangle& clip_indicator_background_rect = m_clip_indicator_background_rect[channel];
        clip_indicator_background_rect = bar_background_rect;
        clip_indicator_background_rect.set_y(critical_background_rect.get_y()
                                             - m_spacing - m_clip_indicator_size);
        clip_indicator_background_rect.set_height(m_clip_indicator_size);
      }
      // adjust for the next meter
      current_bar_origin_x += m_bar_thickness + m_spacing;
    }
    else
    {
      // normal background
      normal_background_rect.set_width(round(bar_length * ic_warn_level));
      // warning background
      warning_background_rect.set_x(current_bar_origin_x + normal_background_rect.get_width());
      warning_background_rect.set_width(round(bar_length * (ic_critical_level - ic_warn_level)));
      // critical background
      critical_background_rect.set_x(warning_background_rect.get_x() + warning_background_rect.get_width());
      critical_background_rect.set_width(round(bar_length * (1.0 - ic_critical_level)));
      // clip indicator background
      if (m_has_clip_indicator)
      {
        Rectangle& clip_indicator_background_rect = m_clip_indicator_background_rect[channel];
        clip_indicator_background_rect = bar_background_rect;
        clip_indicator_background_rect.set_x(current_bar_origin_x
                                             + bar_background_rect.get_width() + m_spacing);
        clip_indicator_background_rect.set_width(m_clip_indicator_size);
      }
      // adjust for the next meter
      current_bar_origin_y += m_bar_thickness + m_spacing;
    }
  }
}

void Level_meter_widget::mf_setup_colors()
{
  // green for normal
  m_normal_color.set_rgb(0,
                         65535,
                         0);
  // yellow for warning
  m_warning_color.set_rgb(65535,
                          65535,
                          0);
  // red for critical
  m_critical_color.set_rgb(65535,
                           0,
                           0);
  // dim green for normal background
  m_background_normal_color.set_rgb(0,
                                    65535 / 4,
                                    0);
  // dim yellow for warning background
  m_background_warning_color.set_rgb(65535 / 4,
                                     65535 / 4,
                                     0);
  // dim red for critical background
  m_background_critical_color.set_rgb(65535 / 4,
                                      0,
                                      0);
  // grey for disabled meters
  m_disabled_color.set_rgb(65535 / 3 * 2,
                           65535 / 3 * 2,
                           65535 / 3 * 2);
}

// calculate rectangles and colors needed when meter is updated
void Level_meter_widget::mf_update_meters()
{
  const bool vertical = Level_meter_view::vertical();
  size_t number_channels = static_cast<size_t>(channels());
  for (size_t channel = 0; channel < number_channels; ++channel)
  {
    const double scaled_level = Level_meter_view::scaled_level(channel);

    if (m_enabled[channel])
    {
      const bool show_warning = (scaled_level > ic_warn_level);
      m_show_warning[channel] = show_warning;
      const bool show_critical = (scaled_level > ic_critical_level);
      m_show_critical[channel] = show_critical;

      // calculate rects for bars
      const int bar_length = m_bar_length;
      // normal part first
      Rectangle& normal_rect = m_normal_rect[channel];
      normal_rect = m_normal_background_rect[channel];
      const double normal_bar_scale = fmin(ic_warn_level, scaled_level);
      if (!show_warning)
      {
        // normal part is only partially lit, adjust
        if (vertical)
        {
          const Rectangle normal_reference_rect = normal_rect;
          const int normal_height = round(bar_length * normal_bar_scale);
          normal_rect.set_height(normal_height);
          normal_rect.set_y(normal_reference_rect.get_y()
                              + (normal_reference_rect.get_height() - normal_height));
        }
        else
        {
          normal_rect.set_width(round(bar_length * normal_bar_scale));
        }
      }
      else
      {
        // warning part, if present
        Rectangle& warning_rect = m_warning_rect[channel];
        warning_rect = m_warning_background_rect[channel];
        const double warning_bar_scale = fmin((ic_critical_level - ic_warn_level),
                                              (scaled_level - ic_warn_level));
        if (!show_critical)
        {
          // warning part is only partially lit, adjust
          if (vertical)
          {
            const Rectangle warning_reference_rect = warning_rect;
            const int warning_height = round(bar_length * warning_bar_scale);
            warning_rect.set_height(warning_height);
            if (!show_critical)
            {
              warning_rect.set_y(warning_reference_rect.get_y()
                                 + (warning_reference_rect.get_height() - warning_height));
            }
          }
          else
          {
            warning_rect.set_width(round(bar_length * warning_bar_scale));
          }
        }
        else
        {
          // critical part, if present
          Rectangle& critical_rect =  m_critical_rect[channel];
          critical_rect = m_critical_background_rect[channel];
          double critical_bar_scale = fmin((1.0 - ic_critical_level), (scaled_level - ic_critical_level));
          if (vertical)
          {
            const Rectangle& critical_reference_rect = critical_rect;
            const int critical_height = round(bar_length * critical_bar_scale);
            critical_rect.set_height(critical_height);
            critical_rect.set_y(critical_reference_rect.get_y()
                                + (critical_reference_rect.get_height() - critical_height));
          }
          else
          {
            critical_rect.set_width(round(bar_length * critical_bar_scale));
          }
        }
      }
      // peak indicator
      if (m_has_peak)
      {
        const double current_peak = peak(channel);
        m_show_peak[channel] = (current_peak > ic_minimum_displayable_peak);
        if (m_show_peak[channel])
        {
          m_peak_color[channel] = (current_peak < ic_warn_level) ? m_normal_color
            : ((current_peak < ic_critical_level) ? m_warning_color : m_critical_color);
          Rectangle& peak_rect = m_peak_rect[channel];
          peak_rect = m_bar_background_rect[channel];
          if (vertical)
          {
            double peak_y = peak_rect.get_y() + bar_length - min(1.0, current_peak) * bar_length;
            peak_y = round(peak_y);
            peak_rect.set_y(peak_y);
            peak_rect.set_height(1);
          }
          else
          {
            double peak_x = peak_rect.get_x() + min(1.0, current_peak) * bar_length;
            peak_x = round(peak_x) - 1;
            peak_rect.set_x(peak_x);
            peak_rect.set_width(1);
          }
        }
      }
    }
  }
}
