/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "../../gfx.h"
#include "gl-utils.h"
#include "gl-utils-internal.h"

/*
 * Pushes a key event.
 */
static void
key_down (mod_gl_utils_keyboard_state_t * state, int keysym, int unicode,
	  char *label)
{
  (state->keypress_queue)++;
  if ((state->keypress_queue) < MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE)
    {
      state->keypress_current =
	((state->keypress_current) + 1) % MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE;
      state->keypress_buffer[state->keypress_current].keysym = keysym;
      state->keypress_buffer[state->keypress_current].unicode = unicode;
      state->keypress_buffer[state->keypress_current].label = label;
    }

  if (keysym >= 0 && keysym < MOD_GL_UTILS_KEYPRESS_STATE_SIZE)
    {
      state->keypress_state[keysym] = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("incorrect keysym (%d) for keydown event"), keysym);
    }
}

static void
key_up (mod_gl_utils_keyboard_state_t * state, int sym)
{
  if (sym >= 0 && sym < MOD_GL_UTILS_KEYPRESS_STATE_SIZE)
    {
      state->keypress_state[sym] = 0;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("incorrect keysym (%d) for keydown event"), sym);
    }
}

void
update_mouse_pos (mod_gl_utils_mouse_state_t * state, int x, int y)
{
  state->x = x;
  state->y = y;
}

void
mouse_button_down (mod_gl_utils_mouse_state_t * state, int sdl_button)
{
  switch (sdl_button)
    {
    case SDL_BUTTON_LEFT:
      state->button_left.pressed = 1;
      state->button_left.queue++;
      break;
    case SDL_BUTTON_RIGHT:
      state->button_right.pressed = 1;
      state->button_right.queue++;
      break;
    case SDL_BUTTON_WHEELUP:
      state->wheel_up.queue++;
      break;
    case SDL_BUTTON_WHEELDOWN:
      state->wheel_down.queue++;
      break;
    }
}

void
mouse_button_up (mod_gl_utils_mouse_state_t * state, int sdl_button)
{
  switch (sdl_button)
    {
    case SDL_BUTTON_LEFT:
      state->button_left.pressed = 0;
      break;
    case SDL_BUTTON_RIGHT:
      state->button_right.pressed = 0;
      break;
    }
}

static int
poll_button (mod_gl_utils_button_state_t * state)
{
  int ret = 0;

  if (state->queue > 0)
    {
      ret++;
      state->queue--;
    }

  return ret;
}

static int
check_joystick_index (int i)
{
  int ret = 0;

  if (i >= 0 && i < MOD_GL_NB_JOYSTICKS)
    {
      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("joystick index %d out of range"), i);
    }

  return ret;
}

static int
check_joystick_button_index (int b)
{
  int ret = 0;

  if (b >= 0 && b < MOD_GL_NB_JOYSTICK_BUTTONS)
    {
      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("joystick button index %d out of range"), b);
    }

  return ret;
}

static void
update_joystick_axis (mod_gl_utils_context_t * utils_context, int which,
		      int axis, int value)
{
  mod_gl_utils_joystick_state_t *joystick_state = NULL;
  int ticks;

  if (check_joystick_index (which))
    {
      joystick_state =
	&(utils_context->joysticks_state.joystick_state[which]);
      ticks = mod_gl_utils_get_ticks (utils_context);
      switch (axis)
	{
	case 0:
	  if (value < utils_context->const_data.joystick_limit_left)
	    {
	      if (!joystick_state->left.pressed)
		{
		  joystick_state->left.pressed = 1;
		  (joystick_state->left.queue)++;
		  joystick_state->left.last_press = ticks;
		  joystick_state->left.last_repeat = 0;
		}
	      else
		{
		  joystick_state->left.pressed = 0;
		}
	    }
	  else
	    {
	      joystick_state->left.pressed = 0;
	    }
	  if (value > utils_context->const_data.joystick_limit_right)
	    {
	      if (!joystick_state->right.pressed)
		{
		  joystick_state->right.pressed = 1;
		  (joystick_state->right.queue)++;
		  joystick_state->right.last_press = ticks;
		  joystick_state->right.last_repeat = 0;
		}
	      else
		{
		  joystick_state->right.pressed = 0;
		}
	    }
	  else
	    {
	      joystick_state->right.pressed = 0;
	    }
	  break;
	case 1:
	  if (value < utils_context->const_data.joystick_limit_up)
	    {
	      if (!joystick_state->up.pressed)
		{
		  joystick_state->up.pressed = 1;
		  (joystick_state->up.queue)++;
		  joystick_state->up.last_press = ticks;
		  joystick_state->up.last_repeat = 0;
		}
	      else
		{
		  joystick_state->up.pressed = 0;
		}
	    }
	  else
	    {
	      joystick_state->up.pressed = 0;

	    }
	  if (value > utils_context->const_data.joystick_limit_down)
	    {
	      if (!joystick_state->down.pressed)
		{
		  joystick_state->down.pressed = 1;
		  (joystick_state->down.queue)++;
		  joystick_state->down.last_press = ticks;
		  joystick_state->down.last_repeat = 0;
		}
	      else
		{
		  joystick_state->down.pressed = 0;
		}
	    }
	  else
	    {
	      joystick_state->down.pressed = 0;

	    }
	  break;
	}
    }
}

static void
joystick_button_down (mod_gl_utils_context_t * utils_context, int i, int b)
{
  mod_gl_utils_joystick_state_t *joystick_state;

  if (check_joystick_index (i))
    {
      joystick_state = &(utils_context->joysticks_state.joystick_state[i]);

      if (check_joystick_button_index (b))
	{
	  joystick_state->buttons[b].pressed = 1;
	  joystick_state->buttons[b].queue++;
	  joystick_state->buttons[b].last_press =
	    mod_gl_utils_get_ticks (utils_context);
	  joystick_state->buttons[b].last_repeat = 0;
	}
    }
}

static void
joystick_button_up (mod_gl_utils_context_t * utils_context, int i, int b)
{
  mod_gl_utils_joystick_state_t *joystick_state;

  if (check_joystick_index (i))
    {
      joystick_state = &(utils_context->joysticks_state.joystick_state[i]);

      if (check_joystick_button_index (b))
	{
	  joystick_state->buttons[b].pressed = 0;
	}
    }
}

static void
update_button_repeat (mod_gl_utils_button_state_t * button_state, int ticks,
		      int delay, int interval)
{
  if (button_state->pressed)
    {
      if ((delay > 0 && button_state->last_repeat == 0
	   && button_state->last_press <= ticks - delay) || (interval > 0
							     &&
							     button_state->
							     last_repeat != 0
							     && button_state->
							     last_repeat <=
							     ticks -
							     interval))
	{
	  (button_state->queue)++;
	  button_state->last_repeat = ticks;
	}
    }
}

static void
update_joystick_repeat (mod_gl_utils_context_t * utils_context)
{
  int i, b, ticks, delay, interval;

  ticks = mod_gl_utils_get_ticks (utils_context);
  delay = utils_context->repeat_settings.delay;
  interval = utils_context->repeat_settings.interval;
  for (i = 0; i < MOD_GL_NB_JOYSTICKS; ++i)
    {
      update_button_repeat (&
			    (utils_context->joysticks_state.joystick_state[i].
			     left), ticks, delay, interval);
      update_button_repeat (&
			    (utils_context->joysticks_state.joystick_state[i].
			     right), ticks, delay, interval);
      update_button_repeat (&
			    (utils_context->joysticks_state.joystick_state[i].
			     up), ticks, delay, interval);
      update_button_repeat (&
			    (utils_context->joysticks_state.joystick_state[i].
			     down), ticks, delay, interval);
      for (b = 0; b < MOD_GL_NB_JOYSTICK_BUTTONS; ++b)
	{
	  update_button_repeat (&
				(utils_context->joysticks_state.
				 joystick_state[i].buttons[b]), ticks, delay,
				interval);
	}
    }
}

static void
log_event (SDL_Event * event)
{
  switch (event->type)
    {
    case SDL_ACTIVEEVENT:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _("SDL event type=SDL_ACTIVEEVENT gain=%d state=%d"),
		  (int) event->active.gain, (int) event->active.state);
      break;
    case SDL_KEYDOWN:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_KEYDOWN state=%d scancode=%d sym=%d unicode=%d"),
		  (int) event->key.state,
		  (int) event->key.keysym.scancode,
		  (int) event->key.keysym.sym,
		  (int) event->key.keysym.mod,
		  (int) event->key.keysym.unicode);
      break;
    case SDL_KEYUP:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_KEYUP state=%d scancode=%d sym=%d unicode=%d"),
		  (int) event->key.state,
		  (int) event->key.keysym.scancode,
		  (int) event->key.keysym.sym,
		  (int) event->key.keysym.mod,
		  (int) event->key.keysym.unicode);
      break;
    case SDL_MOUSEMOTION:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_MOUSEMOTION state=%d x=%d y=%d xrel=%d yrel=%d"),
		  (int) event->motion.state, (int) event->motion.x,
		  (int) event->motion.y, (int) event->motion.xrel,
		  (int) event->motion.yrel);
      break;
    case SDL_MOUSEBUTTONDOWN:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_MOUSEBUTTONDOWN button=%d state=%d x=%d y=%d"),
		  (int) event->button.button, (int) event->button.state,
		  (int) event->motion.x, (int) event->motion.y);
      break;
    case SDL_MOUSEBUTTONUP:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_MOUSEBUTTONUP button=%d state=%d x=%d y=%d"),
		  (int) event->button.button, (int) event->button.state,
		  (int) event->motion.x, (int) event->motion.y);
      break;
    case SDL_JOYAXISMOTION:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYAXISMOTION which=%d axis=%d value=%d"),
		  (int) event->jaxis.which, event->jaxis.axis,
		  (int) event->jaxis.value);
      break;
    case SDL_JOYBALLMOTION:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYBALLMOTION which=%d ball=%d xrel=%d yrel=%d"),
		  (int) event->jball.which, (int) event->jball.ball,
		  (int) event->jball.xrel, (int) event->jball.yrel);
      break;
    case SDL_JOYHATMOTION:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYHATMOTION which=%d hat=%d value=%d"),
		  (int) event->jhat.which, (int) event->jhat.hat,
		  (int) event->jhat.value);
      break;
    case SDL_JOYBUTTONDOWN:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYBUTTONDOWN which=%d button=%d state=%d"),
		  (int) event->jbutton.which, (int) event->jbutton.button,
		  (int) event->jbutton.state);
      break;
    case SDL_JOYBUTTONUP:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYBUTTONUP which=%d button=%d state=%d"),
		  (int) event->jbutton.which, (int) event->jbutton.button,
		  (int) event->jbutton.state);
      break;
    case SDL_QUIT:
      lw6sys_log (LW6SYS_LOG_DEBUG, _("SDL event type=SDL_QUIT"));
      break;
    case SDL_SYSWMEVENT:
      lw6sys_log (LW6SYS_LOG_DEBUG, _("SDL event type=SDL_SYSWMEVENT"));
      break;
    case SDL_VIDEORESIZE:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _("SDL event type=SDL_VIDEORESIZE w=%d h=%d"),
		  (int) event->resize.w, (int) event->resize.h);
      break;
    case SDL_VIDEOEXPOSE:
      lw6sys_log (LW6SYS_LOG_DEBUG, _("SDL event type=SDL_VIDEOEXPOSE"));
      break;
    case SDL_USEREVENT:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _("SDL event type=SDL_USEREVENT code=%d"),
		  (int) event->user.code);
      break;
    default:
// should never get there, unless API changes...
      lw6sys_log (LW6SYS_LOG_WARNING, _("SDL event with unknown type=%d"),
		  event->type);
    }
}

/*
 * Internal poll function.
 */
static int
poll (mod_gl_utils_context_t * utils_context)
{
  int ret = 0;
  mod_gl_utils_keyboard_state_t *keyboard_state =
    &utils_context->keyboard_state;
  mod_gl_utils_mouse_state_t *mouse_state = &utils_context->mouse_state;

  SDL_Event event;

  while (SDL_PollEvent (&event))
    {
      ret = 1;

      log_event (&event);
      switch (event.type)
	{
	case SDL_KEYDOWN:
	  key_down (keyboard_state,
		    event.key.keysym.sym,
		    event.key.keysym.unicode,
		    SDL_GetKeyName (event.key.keysym.sym));
	  break;
	case SDL_KEYUP:
	  key_up (keyboard_state, event.key.keysym.sym);
	  break;
	case SDL_QUIT:
	  utils_context->quit_state.quit = 1;
	  break;
	case SDL_MOUSEMOTION:
	  utils_context->mouse_state.moved = 1;
	  update_mouse_pos (mouse_state, event.motion.x, event.motion.y);
	  break;
	case SDL_MOUSEBUTTONDOWN:
	  update_mouse_pos (mouse_state, event.button.x, event.button.y);
	  mouse_button_down (mouse_state, event.button.button);
	  break;
	case SDL_MOUSEBUTTONUP:
	  update_mouse_pos (mouse_state, event.button.x, event.button.y);
	  mouse_button_up (mouse_state, event.button.button);
	  break;
	case SDL_JOYAXISMOTION:
	  update_joystick_axis (utils_context, event.jaxis.which,
				event.jaxis.axis, event.jaxis.value);
	  break;
	case SDL_JOYBUTTONDOWN:
	  joystick_button_down (utils_context, event.jbutton.which,
				event.jbutton.button);
	  break;
	case SDL_JOYBUTTONUP:
	  joystick_button_up (utils_context, event.jbutton.which,
			      event.jbutton.button);
	  break;
	case SDL_VIDEORESIZE:
	  mod_gl_utils_resize_video_mode (utils_context, event.resize.w,
					  event.resize.h);
	  break;
	case SDL_VIDEOEXPOSE:
	  mod_gl_utils_sync_mode (utils_context, 0);
	  break;
	}
    }

  return ret;
}

/*
 * Polls a key press.
 */
int
mod_gl_utils_poll_key (mod_gl_utils_context_t * utils_context,
		       lw6gui_keypress_t * keypress)
{
  int ret = 0;

  poll (utils_context);

  if (utils_context->keyboard_state.keypress_queue > 0)
    {
      ret = 1;
      (utils_context->keyboard_state.keypress_queue)--;
      if (keypress)
	{
	  int pos;

	  pos = (MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE
		 + utils_context->keyboard_state.keypress_current
		 - utils_context->keyboard_state.keypress_queue)
	    % MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE;
	  *keypress = (utils_context->keyboard_state.keypress_buffer)[pos];
	}
    }

  return ret;
}

/*
 * Following functions are used to export to Guile the internal
 * SDL esc/up/down/left/right/enter keycodes which are the basics
 * to handle menu stuff. Other keys are basically ascii keys 
 * and/or special control keys but which do not have any special
 * meaning since they're used for controlling cursors, independently
 * of their expected behavior. Point is esc/up/down/left/right/enter
 * are necessary for menu stuff. One option would be to export all
 * SLDK_ symbols to Guile, but it's 1) useless and 2) it would tie
 * scheme code to SDL, which is not needed nor required nor wished.
 */

int
mod_gl_utils_is_key_esc (mod_gl_utils_context_t * utils_context, int keysym)
{
  return ((keysym == SDLK_ESCAPE) || (keysym == SDLK_HOME));
}

int
mod_gl_utils_is_key_enter (mod_gl_utils_context_t * utils_context, int keysym)
{
  return ((keysym == SDLK_RETURN) || (keysym == SDLK_KP_ENTER));
}

int
mod_gl_utils_is_key_up (mod_gl_utils_context_t * utils_context, int keysym)
{
  return ((keysym == SDLK_UP) || (keysym == SDLK_KP8));
}

int
mod_gl_utils_is_key_down (mod_gl_utils_context_t * utils_context, int keysym)
{
  return ((keysym == SDLK_DOWN) || (keysym == SDLK_KP2));
}

int
mod_gl_utils_is_key_left (mod_gl_utils_context_t * utils_context, int keysym)
{
  return ((keysym == SDLK_LEFT) || (keysym == SDLK_KP4));
}

int
mod_gl_utils_is_key_right (mod_gl_utils_context_t * utils_context, int keysym)
{
  return ((keysym == SDLK_RIGHT) || (keysym == SDLK_KP6));
}

/*
 * Polls a quit event
 */
int
mod_gl_utils_poll_quit (mod_gl_utils_context_t * utils_context)
{
  poll (utils_context);

  return utils_context->quit_state.quit;
}

/*
 * Sends a quit event
 */
void
mod_gl_utils_send_quit (mod_gl_utils_context_t * utils_context)
{
  utils_context->quit_state.quit = 1;
}

int
mod_gl_utils_get_key_state (mod_gl_utils_context_t * utils_context,
			    int keysym)
{
  int ret = 0;

  if (keysym >= 0 && keysym < MOD_GL_UTILS_KEYPRESS_STATE_SIZE)
    {
      ret = utils_context->keyboard_state.keypress_state[keysym];
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("can't get key state for incorrect keysym (%d)"), keysym);
    }

  return ret;
}

void
mod_gl_utils_get_default_keys_state (mod_gl_utils_context_t * utils_context,
				     int *up, int *down, int *left,
				     int *right, int *esc, int *enter)
{
  if (up)
    {
      (*up) = mod_gl_utils_get_key_state (utils_context, SDLK_UP) ||
	mod_gl_utils_get_key_state (utils_context, SDLK_KP8);
    }
  if (down)
    {
      (*down) = mod_gl_utils_get_key_state (utils_context, SDLK_DOWN) ||
	mod_gl_utils_get_key_state (utils_context, SDLK_KP2);
    }
  if (left)
    {
      (*left) = mod_gl_utils_get_key_state (utils_context, SDLK_LEFT) ||
	mod_gl_utils_get_key_state (utils_context, SDLK_KP4);
    }
  if (right)
    {
      (*right) = mod_gl_utils_get_key_state (utils_context, SDLK_RIGHT) ||
	mod_gl_utils_get_key_state (utils_context, SDLK_KP6);
    }
  if (esc)
    {
      (*esc) = mod_gl_utils_get_key_state (utils_context, SDLK_ESCAPE) ||
	mod_gl_utils_get_key_state (utils_context, SDLK_HOME);
    }
  if (enter)
    {
      (*enter) = mod_gl_utils_get_key_state (utils_context, SDLK_RETURN) ||
	mod_gl_utils_get_key_state (utils_context, SDLK_KP_ENTER);
    }
}

/*
 * Returns true if mouse is moved. Even if not moved, x & y
 * contains the position of the mouse.
 */
int
mod_gl_utils_poll_mouse_move (mod_gl_utils_context_t * utils_context,
			      int *x, int *y)
{
  int ret = 0;

  poll (utils_context);

  (*x) = utils_context->mouse_state.x;
  (*y) = utils_context->mouse_state.y;

  if (utils_context->mouse_state.moved)
    {
      utils_context->mouse_state.moved = 0;
      ret = 1;
    }

  return ret;
}

int
mod_gl_utils_poll_mouse_button_left (mod_gl_utils_context_t * utils_context)
{
  int ret = 0;

  poll (utils_context);

  ret = poll_button (&(utils_context->mouse_state.button_left));

  return ret;
}

int
mod_gl_utils_poll_mouse_button_right (mod_gl_utils_context_t * utils_context)
{
  int ret = 0;

  poll (utils_context);

  ret = poll_button (&(utils_context->mouse_state.button_right));

  return ret;
}

int
mod_gl_utils_poll_mouse_wheel_up (mod_gl_utils_context_t * utils_context)
{
  int ret = 0;

  poll (utils_context);

  ret = poll_button (&(utils_context->mouse_state.wheel_up));

  return ret;
}

int
mod_gl_utils_poll_mouse_wheel_down (mod_gl_utils_context_t * utils_context)
{
  int ret = 0;

  poll (utils_context);

  ret = poll_button (&(utils_context->mouse_state.wheel_down));

  return ret;
}

void
mod_gl_utils_get_mouse_state (mod_gl_utils_context_t * utils_context,
			      int *x, int *y, int *button_left,
			      int *button_right)
{
  mod_gl_utils_mouse_state_t *mouse_state = &utils_context->mouse_state;

  (*x) = mouse_state->x;
  (*y) = mouse_state->y;
  (*button_left) = mouse_state->button_left.pressed;
  (*button_right) = mouse_state->button_right.pressed;
}

int
mod_gl_utils_warp_mouse (mod_gl_utils_context_t * utils_context, int x, int y)
{
  int ret = 0;

  if (x >= 0 && x < utils_context->screen.width && y >= 0
      && y < utils_context->screen.height)
    {
      ret = 1;
      SDL_WarpMouse (x, y);
    }

  return ret;
}

void
mod_gl_utils_show_mouse (mod_gl_utils_context_t * utils_context, int state)
{
  SDL_ShowCursor (state ? SDL_ENABLE : SDL_DISABLE);
}

static SDL_Joystick *
get_joystick (mod_gl_utils_context_t * utils_context, int i)
{
  SDL_Joystick *joystick = NULL;

  if (check_joystick_index (i))
    {
      joystick =
	utils_context->joysticks_state.joystick_state[i].sdl_joystick;
    }

  return joystick;
}

int
mod_gl_utils_poll_joystick_up (mod_gl_utils_context_t * utils_context, int i)
{
  int ret = 0;

  if (utils_context->joysticks_state.supported)
    {
      if (check_joystick_index (i))
	{
	  poll (utils_context);
	  update_joystick_repeat (utils_context);

	  ret =
	    poll_button (&
			 (utils_context->joysticks_state.
			  joystick_state[i].up));
	}
    }

  return ret;
}

int
mod_gl_utils_poll_joystick_down (mod_gl_utils_context_t * utils_context,
				 int i)
{
  int ret = 0;

  if (utils_context->joysticks_state.supported)
    {
      if (check_joystick_index (i))
	{
	  poll (utils_context);
	  update_joystick_repeat (utils_context);

	  ret =
	    poll_button (&
			 (utils_context->joysticks_state.
			  joystick_state[i].down));
	}
    }

  return ret;
}

int
mod_gl_utils_poll_joystick_left (mod_gl_utils_context_t * utils_context,
				 int i)
{
  int ret = 0;

  if (utils_context->joysticks_state.supported)
    {
      if (check_joystick_index (i))
	{
	  poll (utils_context);
	  update_joystick_repeat (utils_context);

	  ret =
	    poll_button (&
			 (utils_context->joysticks_state.
			  joystick_state[i].left));
	}
    }

  return ret;
}

int
mod_gl_utils_poll_joystick_right (mod_gl_utils_context_t * utils_context,
				  int i)
{
  int ret = 0;

  if (utils_context->joysticks_state.supported)
    {
      if (check_joystick_index (i))
	{
	  poll (utils_context);
	  update_joystick_repeat (utils_context);

	  ret =
	    poll_button (&
			 (utils_context->joysticks_state.
			  joystick_state[i].right));
	}
    }

  return ret;
}

int
mod_gl_utils_poll_joystick_button (mod_gl_utils_context_t *
				   utils_context, int i, int b)
{
  int ret = 0;

  if (utils_context->joysticks_state.supported)
    {
      if (check_joystick_index (i) && check_joystick_button_index (b))
	{
	  poll (utils_context);
	  update_joystick_repeat (utils_context);

	  ret =
	    poll_button (&
			 (utils_context->joysticks_state.
			  joystick_state[i].buttons[b]));
	}
    }

  return ret;
}

void
mod_gl_utils_get_joystick_state (mod_gl_utils_context_t *
				 utils_context, int *x, int *y,
				 int *button_a, int *button_b, int i, int b1,
				 int b2)
{
  SDL_Joystick *joystick = NULL;
  int num_axes;
  int axe_value;
  int num_buttons;

  if (x)
    {
      (*x) = 0;
    }
  if (y)
    {
      (*y) = 0;
    }
  if (button_a)
    {
      (*button_a) = 0;
    }
  if (button_b)
    {
      (*button_b) = 0;
    }

  if (utils_context->joysticks_state.supported)
    {
      joystick = get_joystick (utils_context, i);
      if (joystick)
	{
	  num_axes = SDL_JoystickNumAxes (joystick);

	  if (num_axes >= 1)
	    {
	      if (x)
		{
		  axe_value = SDL_JoystickGetAxis (joystick, 0);
		  if (axe_value >
		      utils_context->const_data.joystick_limit_right)
		    {
		      (*x) = 1;
		    }
		  else if (axe_value <
			   utils_context->const_data.joystick_limit_left)
		    {
		      (*x) = -1;
		    }
		}
	    }
	  else
	    {
	      lw6sys_log (LW6SYS_LOG_INFO,
			  _
			  ("only %d axes for joystick %d, x axis can't be fetched"),
			  num_axes, i);
	    }

	  if (num_axes >= 2)
	    {
	      if (y)
		{
		  axe_value = SDL_JoystickGetAxis (joystick, 1);
		  if (axe_value >
		      utils_context->const_data.joystick_limit_down)
		    {
		      (*y) = 1;
		    }
		  else if (axe_value <
			   utils_context->const_data.joystick_limit_up)
		    {
		      (*y) = -1;
		    }
		}
	    }
	  else
	    {
	      lw6sys_log (LW6SYS_LOG_INFO,
			  _
			  ("only %d axes for joystick %d, y axis can't be fetched"),
			  num_axes, i);
	    }

	  if (check_joystick_button_index (b1))
	    {
	      num_buttons = SDL_JoystickNumButtons (joystick);

	      if (num_buttons >= b1)
		{
		  if (button_a)
		    {
		      (*button_a) = SDL_JoystickGetButton (joystick, b1);
		    }
		}
	      else
		{
		  lw6sys_log (LW6SYS_LOG_INFO,
			      _
			      ("only %d buttons for joystick %d, button %d can't be fetched"),
			      num_buttons, i, b1);
		}
	    }

	  if (check_joystick_button_index (b2))
	    {
	      num_buttons = SDL_JoystickNumButtons (joystick);

	      if (num_buttons >= b2)
		{
		  if (button_b)
		    {
		      (*button_b) = SDL_JoystickGetButton (joystick, b2);
		    }
		}
	      else
		{
		  lw6sys_log (LW6SYS_LOG_INFO,
			      _
			      ("only %d buttons for joystick %d, button %d can't be fetched"),
			      num_buttons, i, b2);
		}
	    }
	}
    }
}

int
mod_gl_utils_set_key_repeat (mod_gl_utils_context_t * utils_context,
			     int key_repeat_delay, int key_repeat_interval)
{
  int ret = 0;

  if (key_repeat_delay >= 0 && key_repeat_interval >= 0)
    {
      utils_context->repeat_settings.delay = key_repeat_delay;
      utils_context->repeat_settings.interval = key_repeat_interval;
      SDL_EnableKeyRepeat (key_repeat_delay, key_repeat_interval);
    }

  return ret;
}
